<?php

namespace App\Traits;

use App\Enums\Locale;
use App\Models\City;
use App\Models\CityI18n;
use App\Models\HotelAddress;
use App\Models\HotelRecommendation;
use App\Models\MostBookedHotel;

trait HotelRecommendationFilter
{
    /**
     * Filter hotel recommendations based on user preferences.
     *
     * @param array $recommendations
     * @param array $preferences
     * @return array
     */
    public function hotelRecommendationFilter($result, $requestData)
    {
        $hotelIds = [];
        $priorityRanking = [
            'override' => 4,
            'high' => 3,
            'medium' => 2,
            'low' => 1
        ];

        $recommendedFilter = [
            'recommendationType' => '',
            'hotelId' => [],
            'destination' => '',
            'logicRule' => '',
            'priority' => '',
        ];

        $cities = City::select('id', 'iso_code')
            ->where('latitude', $requestData['geolocation']['latitude'])
            ->where('longitude', $requestData['geolocation']['longitude'])
            ->first();

        if (!$cities) {
            $result['data']['recommendedFilter'] = $recommendedFilter;
            return $result;
        }

        $citiesDetails = CityI18n::where('city_id', $cities->id)
            ->select('city_id', 'city_name')
            ->where('language_code', Locale::English->value)
            ->first();

        if (!$citiesDetails) {
            $result['data']['recommendedFilter'] = $recommendedFilter;
            return $result;
        }


        $recommendationsList = HotelRecommendation::whereRaw('? LIKE CONCAT("%", destination, "%")', [$citiesDetails->city_name])
            ->where('status', 'active')
            ->whereDate('effective_date_from', '<=', now())
            ->whereDate('effective_date_to', '>=', now())
            ->orderByDesc('created_at')
            ->get();

        if ($recommendationsList->isEmpty()) {
            $result['data']['recommendedFilter'] = $recommendedFilter;
            return $result;
        }

     
        $bestRecommendation = $recommendationsList->sortByDesc(function ($item) use ($priorityRanking) {
            return $priorityRanking[$item->priority_level] ?? 0;
        })->first();

        if ($bestRecommendation->recommendation_type === 'general_logic' && $bestRecommendation->logic_rule === 'most_booked_hotels') {
            $mostBookedHotels = MostBookedHotel::select('hotel_id')
                ->whereRaw('? LIKE CONCAT("%", city_name, "%")', [$citiesDetails->city_name])
                ->where('booked_count', '>', 0)
                ->orderByDesc('booked_count')
                ->limit(10)
                ->get();

            $hotelIds = $mostBookedHotels->pluck('hotel_id')->toArray();
        }

        $recommendedFilter = [
            'recommendationType' => $bestRecommendation->recommendation_type,
            'hotelId' => $bestRecommendation->recommendation_type === 'manual_selection'
                ? explode(',', $bestRecommendation->hotels)
                : $hotelIds,
            'destination' => $bestRecommendation->destination,
            'logicRule' => $bestRecommendation->recommendation_type === 'manual_selection' ? '' : $bestRecommendation->logic_rule,
            'priority' => $bestRecommendation->priority_level,
        ];

        $result['data']['recommendedFilter'] = $recommendedFilter;

        return $result;
    }




    protected function sortByManualSelection($recommendations, $requestData)
    {
        $hotelDetails = $requestData['data']['Hotels'];
        $hotelIdList = explode(',', $recommendations->hotels);

        $sorted = [];
        $remaining = [];

        if ($recommendations->status == 'active' && $recommendations->effective_date_from <= now() && $recommendations->effective_date_to >= now()) {
            foreach ($hotelDetails as $hotel) {
                if (in_array($hotel['ID'], $hotelIdList)) {
                    $hotel['is_recommended'] = true;
                }
            }
        }


        foreach ($hotelDetails as $hotel) {
            if (in_array($hotel['ID'], $hotelIdList)) {
                $sorted[$hotel['ID']] = $hotel;
            } else {
                $remaining[] = $hotel;
            }
        }

        $orderedSorted = [];
        foreach ($hotelIdList as $id) {
            if (isset($sorted[$id])) {
                $orderedSorted[] = $sorted[$id];
            }
        }

        return array_merge($orderedSorted, $remaining);
    }
}
