<?php

namespace App\Traits;

use Exception;
use Carbon\Carbon;
use App\Models\City;

use App\Enums\Locale;
use App\Traits\Uuids;
use App\Enums\Currency;
use App\Models\Setting;
use App\Models\CityI18n;
use App\Models\HotelName;
use App\Models\Suppliers;
use App\Enums\ServiceType;
use App\Traits\UrlFetcher;
use App\Traits\HotelHelper;
use App\Models\GoogleMapLog;
use App\Models\HotelAddress;
use App\Traits\GooglePlaces;
use App\Models\DefaultMarkup;
use App\Models\HotelFacility;
use App\Traits\CommonService;
use App\Models\BookingCheckout;
use App\Models\HotelFacilityAr;
use App\Traits\DateLocalization;
use App\Models\HotelDescriptionAr;
use App\Models\HotelDescriptionEn;
use App\Models\SalesServiceMarkup;
use App\Services\TranslateService;
use App\Traits\LoyaltyPointsService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use App\Models\GimmonixSearchSession;
use Illuminate\Support\Facades\Cache;
use App\Services\HotelFacilityService;
use App\Models\GimmonixHotelDescription;
use App\Services\HotelDescriptionService;
use Google\Service\Dfareporting\Resource\Cities;

trait Gimmonix
{
    use CommonService, DateLocalization, LoyaltyPointsService, GooglePlaces, HotelHelper, UrlFetcher, Uuids, HotelRecommendationFilter;

    private $gimmonixBaseUrl;
    private $gimmonixDynamicUrl;
    private $gimmonixUsername;
    private $gimmonixPassword;
    private $gimmonixCredential;

    // Error code constants
    const ERROR_GENERAL_SYSTEM = 'E10000';
    const ERROR_SESSION_INVALID = 'E20001';
    const ERROR_BUSINESS_LOGIC = 'E20213';
    const ERROR_BOOKING_FAILED = 'E30001';
    const ERROR_PAYMENT_FAILED = 'E30002';
    const ERROR_INVENTORY_UNAVAILABLE = 'E30003';
    const ERROR_CONTENT_NOT_AVAILABLE = 'E40001';
    const ERROR_PACKAGE_NOT_FOUND = 'E40002';

    // Retry configuration
    const MAX_RETRIES = 3;
    const RETRY_DELAY_SECONDS = 1;
    const SESSION_CACHE_MINUTES = 25;
    const SESSION_EXPIRY_MINUTES = 30;


    const ALLOWED_FACILITIES = ['Restaurant(s)', 'Laundry Service', 'Internet access', 'Free Wifi'];

    /**
     * Initialize credentials from database
     */
    private function initializeGimmonixCredentials()
    {
        try {
            $this->gimmonixCredential = env('GIMMONIX_API_CREDENTIALS');

            if ($this->gimmonixCredential === 'live') {
                $this->gimmonixBaseUrl = env('GIMMONIX_API_LIVE_API_ENDPOINT');
                $this->gimmonixDynamicUrl = env('GIMMONIX_API_LIVE_API_DYNAMIC_ENDPOINT');
                $this->gimmonixUsername = env('GIMMONIX_API_LIVE_USER_NAME');
                $this->gimmonixPassword = env('GIMMONIX_API_LIVE_PASSWORD');
            } else {
                $this->gimmonixBaseUrl = env('GIMMONIX_API_TEST_API_ENDPOINT');
                $this->gimmonixDynamicUrl = env('GIMMONIX_API_TEST_API_DYNAMIC_ENDPOINT');
                $this->gimmonixUsername = env('GIMMONIX_API_TEST_USER_NAME');
                $this->gimmonixPassword = Setting::where('config_key', 'gimmonix|api|test|password')->value('value');
            }

            // Validate required credentials
            if (empty($this->gimmonixBaseUrl) || empty($this->gimmonixUsername) || empty($this->gimmonixPassword)) {
                throw new Exception('Missing required Gimmonix credentials');
            }
        } catch (Exception $e) {
            \Log::error('Failed to initialize Gimmonix credentials: ' . $e->getMessage());
            throw new Exception('Configuration error: Unable to load Gimmonix settings');
        }
    }

    /**
     * Handle specific Gimmonix error codes
     */
    private function handleGimmonixError($error, $context = [])
    {
        $errorCode = $error['ErrorCode'] ?? '';
        $errorMessage = $error['Message'] ?? $error['ErrorText'] ?? 'Unknown error';

        switch ($errorCode) {
            case self::ERROR_GENERAL_SYSTEM:
                \Log::error("Gimmonix General System Error (E10000): {$errorMessage}", $context);
                return [
                    'recoverable' => true,
                    'user_message' => 'The booking system is temporarily unavailable. Please try again in a few moments.',
                    'retry_after' => 30,
                    'requires_new_session' => false
                ];

            case self::ERROR_SESSION_INVALID:
                \Log::warning("Gimmonix Session Invalid (E20001): {$errorMessage}", $context);
                return [
                    'recoverable' => true,
                    'user_message' => 'Session expired. Retrying with new session...',
                    'requires_new_session' => true,
                    'retry_after' => 5
                ];

            case self::ERROR_BUSINESS_LOGIC:
                \Log::info("Gimmonix Business Logic Error (E20213): {$errorMessage}", $context);
                return [
                    'recoverable' => false,
                    'user_message' => 'Invalid request parameters. Please check your search criteria and try again.',
                    'detailed_message' => $errorMessage,
                    'requires_new_session' => false
                ];

            case self::ERROR_BOOKING_FAILED:
                \Log::error("Gimmonix Booking Failed (E30001): {$errorMessage}", $context);
                return [
                    'recoverable' => false,
                    'user_message' => 'Booking could not be completed. Please try with different dates or contact support.',
                    'detailed_message' => $errorMessage,
                    'requires_new_session' => false
                ];

            case self::ERROR_PAYMENT_FAILED:
                \Log::error("Gimmonix Payment Failed (E30002): {$errorMessage}", $context);
                return [
                    'recoverable' => false,
                    'user_message' => 'Payment processing failed. Please check your payment details and try again.',
                    'detailed_message' => $errorMessage,
                    'requires_new_session' => false
                ];

            case self::ERROR_INVENTORY_UNAVAILABLE:
                \Log::warning("Gimmonix Inventory Unavailable (E30003): {$errorMessage}", $context);
                return [
                    'recoverable' => false,
                    'user_message' => 'The selected room is no longer available. Please choose a different option.',
                    'detailed_message' => $errorMessage,
                    'requires_new_session' => false
                ];

            case self::ERROR_CONTENT_NOT_AVAILABLE:
                \Log::info("Gimmonix Content Not Available (E40001): {$errorMessage}", $context);
                return [
                    'recoverable' => false,
                    'user_message' => 'Hotel content is not available at the moment.',
                    'detailed_message' => $errorMessage,
                    'requires_new_session' => false
                ];

            case self::ERROR_PACKAGE_NOT_FOUND:
                \Log::info("Gimmonix Package Not Found (E40002): {$errorMessage}", $context);
                return [
                    'recoverable' => false,
                    'user_message' => 'The requested package is not available.',
                    'detailed_message' => $errorMessage,
                    'requires_new_session' => false
                ];

            default:
                \Log::error("Gimmonix Unknown Error ({$errorCode}): {$errorMessage}", $context);
                return [
                    'recoverable' => false,
                    'user_message' => 'An unexpected error occurred. Please try again later.',
                    'detailed_message' => $errorMessage,
                    'requires_new_session' => false
                ];
        }
    }

    /**
     * Parse and handle errors from Gimmonix response
     */
    private function parseGimmonixErrors($response, $context = [])
    {
        $errors = [];
        $recoverableErrors = [];
        $nonRecoverableErrors = [];

        if (isset($response['Errors']) && is_array($response['Errors'])) {
            foreach ($response['Errors'] as $error) {
                $errorInfo = $this->handleGimmonixError($error, $context);
                $errors[] = $errorInfo;

                if ($errorInfo['recoverable']) {
                    $recoverableErrors[] = $errorInfo;
                } else {
                    $nonRecoverableErrors[] = $errorInfo;
                }
            }
        }

        return [
            'has_errors' => !empty($errors),
            'all_errors' => $errors,
            'recoverable_errors' => $recoverableErrors,
            'non_recoverable_errors' => $nonRecoverableErrors,
            'requires_new_session' => collect($errors)->contains('requires_new_session', true)
        ];
    }

    /**
     * Create a new session with Gimmonix with retry logic
     */
    private function createGimmonixSession(): array
    {
        $maxRetries = self::MAX_RETRIES;
        $retryDelay = self::RETRY_DELAY_SECONDS;

        for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
            try {
                $request = [
                    'credentials' => [
                        'UserName' => $this->gimmonixUsername,
                        'Password' => $this->gimmonixPassword
                    ]
                ];

                $response = Http::timeout(30)->post($this->gimmonixBaseUrl . 'login', $request);

                if (!$response->successful()) {
                    throw new Exception("HTTP Error: " . $response->status() . " - " . $response->body());
                }

                $sessionResponse = $response->json();

                if (isset($sessionResponse['LoginResult']['SessionId']) && $sessionResponse['LoginResult']['SessionId'] != '') {
                    \Log::info("Gimmonix session created successfully", [
                        'attempt' => $attempt,
                        'session_id' => substr($sessionResponse['LoginResult']['SessionId'], 0, 8) . '...'
                    ]);

                    return [
                        'status' => true,
                        'data' => $sessionResponse['LoginResult']['SessionId'],
                        'message' => ''
                    ];
                }

                // Handle login errors
                $errors = $sessionResponse['LoginResult']['Errors'] ?? [];
                if (!empty($errors)) {
                    $errorAnalysis = $this->parseGimmonixErrors(['Errors' => $errors], [
                        'operation' => 'login',
                        'attempt' => $attempt
                    ]);

                    // If all errors are recoverable and we have retries left, continue
                    if (empty($errorAnalysis['non_recoverable_errors']) && $attempt < $maxRetries) {
                        \Log::warning("Gimmonix login failed (attempt {$attempt}), retrying...", $errors);
                        sleep($retryDelay * $attempt); // Exponential backoff
                        continue;
                    }

                    return [
                        'status' => false,
                        'data' => '',
                        'message' => $errorAnalysis['non_recoverable_errors'][0]['user_message'] ?? 'Login failed'
                    ];
                }
            } catch (Exception $e) {
                \Log::error("Gimmonix session creation error (attempt {$attempt}): " . $e->getMessage());

                if ($attempt < $maxRetries) {
                    sleep($retryDelay * $attempt);
                    continue;
                }

                return [
                    'status' => false,
                    'data' => '',
                    'message' => 'Unable to establish connection with booking system'
                ];
            }
        }

        return [
            'status' => false,
            'data' => '',
            'message' => 'Failed to create session after multiple attempts'
        ];
    }

    /**
     * Get or create a session ID with caching
     */
    private function getGimmonixSessionId(): string
    {
        // Try to get cached session first
        //$cacheKey = 'gimmonix_session_' . $this->gimmonixCredential;
        //$cachedSession = Cache::get($cacheKey);

        //if ($cachedSession) {
        //    return $cachedSession;
        //}

        $sessionResponse = $this->createGimmonixSession();
        if ($sessionResponse['status']) {
            $sessionId = $sessionResponse['data'];
            // Cache session for specified minutes (sessions typically expire in 30 minutes)
            //Cache::put($cacheKey, $sessionId, now()->addMinutes(self::SESSION_CACHE_MINUTES));
            return $sessionId;
        }

        return '';
    }

    /**
     * Clear cached session
     */
    private function clearCachedSession()
    {
        $cacheKey = 'gimmonix_session_' . $this->gimmonixCredential;
        Cache::forget($cacheKey);
    }

    /**
     * Validate session from database and check expiry
     */
    private function validateDatabaseSession($logId)
    {
        if (empty($logId)) {
            return null;
        }

        $sessionData = GimmonixSearchSession::where('log_id', $logId)->first();

        if (!$sessionData) {
            \Log::warning("Session not found in database for log_id: {$logId}");
            return null;
        }

        $sessionCreated = Carbon::parse($sessionData->created_at)->timezone('UTC');
        $sessionExpiry = $sessionCreated->addMinutes(self::SESSION_EXPIRY_MINUTES);

        if (now('UTC')->greaterThan($sessionExpiry)) {
            \Log::info("Database session expired for log_id: {$logId}");
            return null;
        }

        return $sessionData->session_id;
    }

    /**
     * Make an authenticated API call to Gimmonix with enhanced error handling
     */
    private function makeGimmonixRequest(string $type = 'static', string $endpoint, array $data = [], string $method = 'POST')
    {
        $maxRetries = self::MAX_RETRIES;

        for ($attempt = 1; $attempt <= $maxRetries; $attempt++) {
            $sessionId = $this->getGimmonixSessionId();

            if (empty($sessionId)) {
                \Log::error('Failed to get Gimmonix session ID');
                return [
                    'error' => true,
                    'message' => 'Unable to establish session with booking system'
                ];
            }

            $httpUrl = $type == 'dynamic' ? $this->gimmonixDynamicUrl : ($this->gimmonixBaseUrl . $endpoint);

            try {
                \Log::debug("Making Gimmonix request", [
                    'type' => $type,
                    'endpoint' => $endpoint,
                    'attempt' => $attempt,
                    'url' => $httpUrl
                ]);

                $response = Http::timeout(60)->withHeaders([
                    'Session-ID' => $sessionId
                ])->$method($httpUrl, $data);

                if ($response->successful()) {
                    $responseData = $response->json();

                    // Check for application-level errors in the response
                    $errorPaths = [
                        'SearchResult.Errors',
                        'GetPageResult.Errors',
                        'GetHotelContentResult.Errors',
                        'ServiceRequestResult.Errors',
                        'LoginResult.Errors'
                    ];

                    foreach ($errorPaths as $path) {
                        $errors = data_get($responseData, $path);
                        if (!empty($errors)) {
                            $errorAnalysis = $this->parseGimmonixErrors(['Errors' => $errors], [
                                'operation' => $endpoint,
                                'attempt' => $attempt,
                                'session_id' => substr($sessionId, 0, 8) . '...',
                                'type' => $type
                            ]);

                            if ($errorAnalysis['requires_new_session'] && $attempt < $maxRetries) {
                                \Log::info("Session invalid, clearing cache and retrying...");
                                $this->clearCachedSession();
                                continue 2; // Continue outer loop to retry with new session
                            }

                            if (!empty($errorAnalysis['recoverable_errors']) && $attempt < $maxRetries) {
                                \Log::warning("Recoverable errors detected, retrying...", $errors);
                                sleep(self::RETRY_DELAY_SECONDS * $attempt);
                                continue 2;
                            }

                            // Add error information to response
                            $responseData['error_analysis'] = $errorAnalysis;
                        }
                    }

                    return $responseData;
                }

                // Handle HTTP errors (400, 401, etc.)
                if (in_array($response->status(), [400, 401]) && $attempt < $maxRetries) {
                    \Log::warning("HTTP {$response->status()} error, clearing session and retrying...");
                    $this->clearCachedSession();
                    continue;
                }

                throw new Exception("HTTP Error: " . $response->status() . " - " . $response->body());
            } catch (Exception $e) {
                \Log::error("Gimmonix request error (attempt {$attempt}): " . $e->getMessage(), [
                    'type' => $type,
                    'endpoint' => $endpoint,
                    'attempt' => $attempt
                ]);

                if ($attempt < $maxRetries) {
                    sleep(self::RETRY_DELAY_SECONDS * $attempt);
                    continue;
                }

                return [
                    'error' => true,
                    'message' => 'Request failed after multiple attempts: ' . $e->getMessage()
                ];
            }
        }

        return [
            'error' => true,
            'message' => 'Maximum retry attempts exceeded'
        ];
    }

    /**
     * Enhanced Google Places integration with error handling
     */
    public function getGooglePlaces($hotelDetails)
    {
        $placeTypes = ['airport', 'amusement_park', 'aquarium', 'attractions', 'bus_station', 'landmark', 'museum', 'restaurant', 'subway_station', 'tourist_attraction', 'train_station', 'shopping_mall', 'point_of_interest', 'establishment'];

        $results = [];
        foreach ($placeTypes as $placeType) {
            try {
                $placeUrl = 'https://maps.googleapis.com/maps/api/place/nearbysearch/json?type=' . $placeType . '&location=' . $hotelDetails['GeoLocation']['Latitude'] . urlencode(',') . $hotelDetails['GeoLocation']['Longitude'] . '&radius=500&key=' . env('GOOGLE_PLACES_MAP_KEY');
                $response = $this->fetchUrlData($placeUrl);

                if ($response['status'] == 1) {
                    if ($response['data']['status'] == 'OK') {
                        $userId = Auth::id();
                        GoogleMapLog::createGoogleMapLog($userId);
                        $results[$placeType] = $response['data']['results'];
                    } else {
                        $results[$placeType] = [];
                    }
                } else {
                    $results[$placeType] = [];
                }
            } catch (Exception $e) {
                \Log::warning("Google Places API error for type {$placeType}: " . $e->getMessage());
                $results[$placeType] = [];
            }
        }
        return $results;
    }

    /**
     * Enhanced hotel facilities retrieval
     */
    public function getHotelFacilities(array $hotelIds)
    {
        try {
            return HotelFacility::getByHotelIds($hotelIds);
        } catch (Exception $e) {
            \Log::error('Error getting hotel facilities: ' . $e->getMessage(), ['hotel_ids' => $hotelIds]);
            return [];
        }
    }

    /**
     * Get hotel facilities - With service (includes caching)
     */
    public function getHotelFacilitiesWithCache(array $hotelIds)
    {
        try {
            return app(HotelFacilityService::class)->getFacilities($hotelIds);
        } catch (Exception $e) {
            \Log::error('Error getting cached hotel facilities: ' . $e->getMessage(), ['hotel_ids' => $hotelIds]);
            return [];
        }
    }

    /**
     * Get hotel descriptions in Arabic with cache
     */
    public function getHotelDescriptionsArWithCache(array $hotelIds): array
    {
        try {
            return app(HotelDescriptionService::class)->getDescriptionsAr($hotelIds);
        } catch (Exception $e) {
            \Log::error('Error getting cached Arabic hotel descriptions: ' . $e->getMessage(), ['hotel_ids' => $hotelIds]);
            return [];
        }
    }

    /**
     * Get hotel descriptions in English with cache
     */
    public function getHotelDescriptionsEnWithCache(array $hotelIds): array
    {
        try {
            return app(HotelDescriptionService::class)->getDescriptionsEn($hotelIds);
        } catch (Exception $e) {
            \Log::error('Error getting cached English hotel descriptions: ' . $e->getMessage(), ['hotel_ids' => $hotelIds]);
            return [];
        }
    }

    /**
     * Enhanced hotel short details with better error handling
     */
    public function getGimmonixHotelShortDetails($sessionId, $hotelId, $locale)
    {

        $detailsPayload = $this->transformGimmonixPayload('details', [
            'sessionId' => $sessionId,
            'hotelId' => $hotelId,
            'locale' => $locale
        ]);

        $result = [
            'status' => false,
            'data' => [
                'description' => '',
                'facilities' => []
            ],
            'message' => ''
        ];

        try {
            $response = $this->makeGimmonixRequest('static', 'GetHotelContent', $detailsPayload);

            // Check for request-level errors
            if (isset($response['error']) && $response['error']) {
                $result['message'] = $response['message'];
                return $result;
            }

            if ($response && isset($response['GetHotelContentResult']) && $response['GetHotelContentResult']) {
                // Check for application-level errors
                if (isset($response['error_analysis'])) {
                    $errorAnalysis = $response['error_analysis'];
                    if (!empty($errorAnalysis['non_recoverable_errors'])) {
                        $result['message'] = $errorAnalysis['non_recoverable_errors'][0]['user_message'];
                        return $result;
                    }
                }

                $hotelContent = $response['GetHotelContentResult'];

                if ($locale == Locale::English->value) {
                    if (isset($hotelContent['Descriptions']) && is_array($hotelContent['Descriptions']) && count($hotelContent['Descriptions']) > 0) {
                        foreach ($hotelContent['Descriptions'] as $description) {
                            $result['data']['description'] = $description['Descrption'];
                            break; // Take first description
                        }
                    }
                }

                if ($locale === Locale::Arabic->value) {
                    $cacheKey = "hotel_description_ar_{$hotelId}";

                    try {
                        // $paragraph = Cache::rememberForever($cacheKey, function () use ($hotelId) {
                        //     return HotelDescriptionAr::where('hotelid', $hotelId)
                        //         ->select('paragraph')
                        //         ->value('paragraph');
                        // });
                        $paragraph = HotelDescriptionAr::where('hotel_id', $hotelId)
                            ->select('paragraph')
                            ->value('paragraph');

                        if ($paragraph) {
                            $result['data']['description'] = $paragraph;
                        }
                    } catch (Exception $e) {
                        \Log::warning("Error getting Arabic description for hotel {$hotelId}: " . $e->getMessage());
                    }
                }

                if (isset($hotelContent['Facilities']) && is_array($hotelContent['Facilities']) && count($hotelContent['Facilities']) > 0) {
                    $result['data']['facilities'] = $hotelContent['Facilities'];
                }

                // $allowedFacilities = ['Restaurant(s)', 'Laundry Service', 'Internet access', 'Free Wifi'];

                // if (isset($hotelContent['Facilities']) && is_array($hotelContent['Facilities']) && count($hotelContent['Facilities']) > 0) {
                //     $hotelContent['Facilities'] = array_values(array_filter($hotelContent['Facilities'], function ($facilityItem) use ($allowedFacilities) {
                //         return isset($facilityItem['Facility']) && in_array($facilityItem['Facility'], $allowedFacilities, true);
                //     }));
                //     $result['data']['facilities'] = $hotelContent['Facilities'];
                // } else {
                //     $hotelContent['Facilities'] = [];
                //     $result['data']['facilities'] = [];
                // }


                if (isset($result['data']['description']) || isset($result['data']['facilities'])) {
                    $result['status'] = true;
                }
            }
        } catch (Exception $e) {
            \Log::error("Error getting hotel short details: " . $e->getMessage(), [
                'hotel_id' => $hotelId,
                'locale' => $locale
            ]);
            $result['message'] = 'Unable to fetch hotel details';
        }

        return $result;
    }

    /**
     * Enhanced Arabic hotel names retrieval
     */
    public function getArabicHotelNames($hotelIds, $hotels)
    {
        try {
            $hotelsName = HotelName::whereIn('hotel_id', $hotelIds)
                ->where('language', Locale::Arabic->value)
                ->select('hotel_id', 'display_name')
                ->get()
                ->mapToGroups(function ($item) {
                    return [$item->hotel_id => $item->display_name];
                })
                ->toArray();

            $hotelsNameIds = array_keys($hotelsName);

            foreach ($hotels as &$hotel) {
                $hotelId = $hotel['ID'];
                if (in_array($hotelId, $hotelsNameIds)) {
                    $hotel['DisplayName'] = $hotelsName[$hotelId][0];
                }
            }
        } catch (Exception $e) {
            \Log::error('Error getting Arabic hotel names: ' . $e->getMessage(), ['hotel_ids' => $hotelIds]);
        }

        return $hotels;
    }

    /**
     * Enhanced description and facilities retrieval
     */
    public function getDescriptionFacilities($locale, $hotelIds, $hotels, $hotelIdKey = 'ID', $requestData = [])
    {
        try {
            $facilities = $this->getHotelFacilitiesWithCache($hotelIds);

            $descriptions = ($locale == Locale::Arabic->value)
                ? $this->getHotelDescriptionsArWithCache($hotelIds)
                : $this->getHotelDescriptionsEnWithCache($hotelIds);

            $unMatchedHotelIds = $hotelIds;

            $recommendationFilter = $this->hotelRecommendationFilter([], $requestData);
            $recommendedHotelIds = !empty($recommendationFilter['data']['recommendedFilter']['hotelId'])
                ? $recommendationFilter['data']['recommendedFilter']['hotelId']
                : [];

            // Identify hotels missing facilities in DB
            $hotelsMissingFacilities = array_filter($hotelIds, function ($id) use ($facilities) {
                return empty($facilities[$id]);
            });

            // Determine which hotel IDs we will fetch from Gimmonix
            if (!empty($recommendedHotelIds)) {
                // Only recommended hotels missing facilities get fetched
                $hotelIdsToFetch = array_values(array_intersect($recommendedHotelIds, $hotelsMissingFacilities));
            } else {
                // No recommendation filter: first 10 hotels missing facilities
                $hotelIdsToFetch = array_slice($hotelsMissingFacilities, 0, 10);
            }

            $sessionId = $this->getGimmonixSessionId();
            if (!$sessionId) {
                return ['message' => 'Session ID could not be found or created.'];
            }

            // Fetch facilities from Gimmonix only for selected hotel IDs
            foreach ($hotelIdsToFetch as $hotelId) {
                $detailsResponse = $this->getGimmonixHotelShortDetails($sessionId, $hotelId, $locale);
                if (!empty($detailsResponse['status']) && $detailsResponse['status'] === true) {
                    $fetchedFacilities = $detailsResponse['data']['facilities'] ?? [];
                    if (!empty($fetchedFacilities)) {
                        $facilities[$hotelId] = $fetchedFacilities;
                    }
                }
            }

            if (count($descriptions)) {
                foreach ($hotels as &$hotel) {
                    $hotelId = $hotel[$hotelIdKey];

                    // Set description
                    if (isset($descriptions[$hotelId])) {
                        $descriptionTxt = '';
                        foreach ($descriptions[$hotelId] as $description) {
                            $headline = $locale == Locale::Arabic->value ? 'عنوان رئيسي' : 'headline';
                            if ($description['title'] == $headline) {
                                $descriptionTxt = $description['paragraph'];
                                break;
                            }
                        }
                        $hotel['Description'] = $descriptionTxt;
                        $hotel['DescriptionExists'] = $descriptionTxt !== '';
                    } else {
                        $hotel['Description'] = '';
                        $hotel['DescriptionExists'] = false;
                    }

                    $allowedFacilities = self::ALLOWED_FACILITIES;

                    if (!empty($facilities[$hotelId])) {
                        // Filter only allowed facilities
                        $filteredFacilities = array_filter($facilities[$hotelId], function ($item) use ($allowedFacilities) {
                            return isset($item['facility_name']) && in_array($item['facility_name'], $allowedFacilities, true);
                        });

                        // Reindex array to remove preserved keys
                        $filteredFacilities = array_values($filteredFacilities);

                        if (!empty($filteredFacilities)) {
                            // Map the filtered facilities
                            $facilityData = self::translateHotelListFacilities($filteredFacilities, $hotelId, $locale);
                            $hotel['Facilities'] = $facilityData['Facilities'];
                            $hotel['FacilitiesExists'] = $facilityData['FacilitiesExists'];
                        } else {
                            $hotel['Facilities'] = [];
                            $hotel['FacilitiesExists'] = false;
                        }
                    } else {
                        $hotel['Facilities'] = [];
                        $hotel['FacilitiesExists'] = false;
                    }



                    $hotel['DataSource'] = 'DB';

                    if ($hotel['DescriptionExists'] || $hotel['FacilitiesExists']) {
                        $keyToRemove = array_search($hotelId, $unMatchedHotelIds);
                        if ($keyToRemove !== false) {
                            array_splice($unMatchedHotelIds, $keyToRemove, 1);
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            \Log::error('Error getting description and facilities: ' . $e->getMessage(), [
                'locale' => $locale,
                'hotel_ids' => $hotelIds
            ]);
        }

        return [
            'hotels' => $hotels,
            'unMatchedHotelIds' => $unMatchedHotelIds
        ];
    }


    private static function translateHotelListFacilities($facilities, int $hotelId, string $locale): array
    {
        $structureType = 'unknown';

        if (empty($facilities)) {
            return [
                'Facilities' => [],
                'FacilitiesExists' => false,
                'StructureType' => $structureType
            ];
        }

        // 🔍 Detect structure type
        if (is_object($facilities)) {
            $facilities = (array) $facilities;
            $structureType = 'object_of_objects';
        } elseif (is_array($facilities)) {
            $keys = array_keys($facilities);
            $isSequential = ($keys === range(0, count($facilities) - 1));
            $structureType = $isSequential ? 'array_of_objects' : 'object_of_objects';
        }


        $originalKeys = array_keys($facilities);


        $normalized = array_values($facilities);


        if ($locale === Locale::English->value) {
            $translated = array_map(fn($item) => [
                'Facility'     => $item['facility_name'] ?? ($item['Facility'] ?? ''),
                'FacilityType' => $item['category'] ?? ($item['FacilityType'] ?? ''),
                'HotelID'      => $hotelId,
                'SupplierID'   => $item['supplier_id'] ?? ($item['SupplierID'] ?? 0),
            ], $normalized);
        } else {
            $facilityNames = array_filter(array_map(function ($item) {
                return trim($item['facility_name'] ?? ($item['Facility'] ?? ''));
            }, $normalized));

            if (empty($facilityNames)) {
                return [
                    'Facilities' => [],
                    'FacilitiesExists' => false,
                    'StructureType' => $structureType
                ];
            }

            $existingTranslations = HotelFacilityAr::whereIn('facility_name_en', $facilityNames)
                ->pluck('facility_name_ar', 'facility_name_en')
                ->mapWithKeys(fn($v, $k) => [strtolower($k) => $v])
                ->toArray();

            $newTranslations = [];
            $translated = [];

            foreach ($normalized as $item) {
                $enName = trim($item['facility_name'] ?? ($item['Facility'] ?? ''));
                if (!$enName) continue;

                $lookupKey = strtolower($enName);
                $arName = $existingTranslations[$lookupKey] ?? null;

                if (!$arName) {
                    try {
                        $arName = TranslateService::translateText($enName, 'en', 'ar') ?: $enName;
                    } catch (\Throwable $e) {
                        $arName = $enName;
                    }

                    $newTranslations[] = [
                        'facility_name_en' => $enName,
                        'facility_name_ar' => $arName,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];

                    $existingTranslations[$lookupKey] = $arName;
                }

                $translated[] = [
                    'Facility'     => $enName,
                    'FacilityAr'     => $arName,
                    'FacilityType' => $item['category'] ?? ($item['FacilityType'] ?? ''),
                    'HotelID'      => $hotelId,
                    'SupplierID'   => $item['supplier_id'] ?? ($item['SupplierID'] ?? 0),
                ];
            }


            if (!empty($newTranslations)) {
                HotelFacilityAr::insertOrIgnore($newTranslations);
            }
        }

        $finalFacilities = [];
        foreach (array_values($originalKeys) as $i => $key) {
            $finalFacilities[$key] = $translated[$i] ?? [
                'Facility' => '',
                'FacilityType' => '',
                'HotelID' => $hotelId,
                'SupplierID' => 0,
            ];
        }

        if ($structureType === 'array_of_objects') {
            $finalFacilities = array_values($finalFacilities);
        }

        return [
            'Facilities' => $finalFacilities,
            'FacilitiesExists' => !empty($finalFacilities),
            'StructureType' => $structureType
        ];
    }



    public function getIndividualDescriptionFacilities($locale, $hotelIds, $hotels, $hotelIdKey = 'ID')
    {
        try {
            $facilities = $this->getHotelFacilitiesWithCache($hotelIds);

            if ($locale == Locale::Arabic->value) {
                $descriptions = $this->getHotelDescriptionsArWithCache($hotelIds);
            } else {
                $descriptions = $this->getHotelDescriptionsEnWithCache($hotelIds);
            }

            $unMatchedHotelIds = $hotelIds;

            if (count($facilities) > 0 && count($descriptions)) {
                foreach ($hotels as &$hotel) {
                    $hotelId = $hotel[$hotelIdKey];

                    if (array_key_exists($hotelId, $facilities) && array_key_exists($hotelId, $descriptions)) {
                        $descriptionTxt = '';

                        foreach ($descriptions[$hotelId] as $description) {
                            $headline = $locale == Locale::Arabic->value ? 'عنوان رئيسي' : 'headline';
                            if ($description['title'] == $headline) {
                                $descriptionTxt = $description['paragraph'];
                                break;
                            }
                        }

                        $hotel['DataSource'] = 'DB';
                        $hotel['Description'] = $descriptionTxt;
                        $hotel['Facilities'] = array_map(function ($item) use ($hotelId) {
                            return [
                                'Facility' => $item['facility_name'] ?? '',
                                'FacilityType' => $item['category'] ?? '',
                                'HotelID' => $hotelId,
                                'SupplierID' => $item['supplier_id'] ?? 0
                            ];
                        }, $facilities[$hotelId]);

                        $hotel['DescriptionExists'] = $descriptionTxt != '' ? true : false;
                        $hotel['FacilitiesExists'] = (count($hotel['Facilities']) > 0) ? true : false;

                        // Map facilities if available (either from DB cache or fetched Gimmonix data)
                        $allowedFacilities = self::ALLOWED_FACILITIES;;

                        if (!empty($facilities[$hotelId])) {
                            // Filter only allowed facilities
                            $filteredFacilities = array_filter($facilities[$hotelId], function ($item) use ($allowedFacilities) {
                                return isset($item['facility_name']) && in_array($item['facility_name'], $allowedFacilities, true);
                            });

                            if (!empty($filteredFacilities)) {
                                $facilityData = self::translateHotelListFacilities($filteredFacilities, $hotelId, $locale);
                                $hotel['Facilities'] = $facilityData['Facilities'];
                                $hotel['FacilitiesExists'] = $facilityData['FacilitiesExists'];
                                // Map the filtered facilities
                                // $hotel['Facilities'] = array_map(function ($item) use ($hotelId) {
                                //     return [
                                //         'Facility'     => $item['facility_name'] ?? '',
                                //         'FacilityType' => $item['category'] ?? '',
                                //         'HotelID'      => $hotelId,
                                //         'SupplierID'   => $item['supplier_id'] ?? 0
                                //     ];
                                // }, $filteredFacilities);
                                // $hotel['FacilitiesExists'] = true;
                            } else {
                                $hotel['Facilities'] = [];
                                $hotel['FacilitiesExists'] = false;
                            }
                        } else {
                            $hotel['Facilities'] = [];
                            $hotel['FacilitiesExists'] = false;
                        }


                        if ($hotel['DescriptionExists'] || $hotel['FacilitiesExists']) {
                            $keyToRemove = array_search($hotelId, $unMatchedHotelIds);
                            if ($keyToRemove !== false) {
                                array_splice($unMatchedHotelIds, $keyToRemove, 1);
                            }
                        }
                    }
                }
            }
        } catch (Exception $e) {
            \Log::error('Error getting description and facilities: ' . $e->getMessage(), [
                'locale' => $locale,
                'hotel_ids' => $hotelIds
            ]);
        }

        return [
            'hotels' => $hotels,
            'unMatchedHotelIds' => $unMatchedHotelIds
        ];
    }

    /**
     * Fetch City Latitude and Longitude using  Hotel Lat and log
     */
    private static function getCityLatitude($lat, $lon)
    {
        $hotelAddress = HotelAddress::whereLat($lat)->whereLng($lon)->first();
        if (!$hotelAddress) {
            return null;
        }
        $cityName = $hotelAddress->cityname ?? null;
        if (!$cityName) {
            return null;
        }
        $cityI18n = CityI18n::where('city_name', 'LIKE', '%' . $cityName . '%')->first();
        if (!$cityI18n) {
            return null;
        }
        $city = City::whereId($cityI18n->city_id)->first();
        return $city ?? null;
    }


    /**
     * Enhanced hotel search with comprehensive error handling
     */
    public function getGimmonixHotels($type, $requestData, $customerId, $logId = '')
    {
        $result = [
            'status' => false,
            'data' => [],
            'message' => ''
        ];

        try {
            if (!$this->gimmonixBaseUrl) {
                $this->initializeGimmonixCredentials();
            }

            $currency = $requestData['currency'];
            $locale = $requestData['locale'];

            // Handle session ID from log or create new one
            if ($logId != '') {
                $sessionId = $this->validateDatabaseSession($logId);
                if (!$sessionId) {
                    $sessionId = $this->getGimmonixSessionId();
                }
            } else {
                $sessionId = $this->getGimmonixSessionId();
            }

            if (empty($sessionId)) {
                $result['message'] = 'Unable to establish session with booking system';
                return $result;
            }

            $payload = $this->transformGimmonixPayload($type, $requestData, $sessionId);
            $endPointPrefix = 'search';

            if ($type == 'filter-list') {
                $endPointPrefix = 'GetPage';
            } else {
                $payload['searchService']['SessionID'] = $sessionId;
            }

            $response = $this->makeGimmonixRequest('static', $endPointPrefix, $payload);

            // Check for request-level errors
            if (isset($response['error']) && $response['error']) {
                $result['message'] = $response['message'];
                return $result;
            }

            $this->createSupplierLog('Hotel', 'Gimmonix', 'hotel-list-by-geo-location-' . $type, json_encode($payload), json_encode($response), $customerId, '', '', $logId);

            // Handle search results
            if ($response && $type != 'filter-list' && isset($response['SearchResult']) && $response['SearchResult']) {
                // Check for application-level errors
                if (isset($response['error_analysis'])) {
                    $errorAnalysis = $response['error_analysis'];
                    if (!empty($errorAnalysis['non_recoverable_errors'])) {
                        $result['message'] = $errorAnalysis['non_recoverable_errors'][0]['user_message'];
                        return $result;
                    }
                }

                $result['status'] = true;
                $hotels = $response['SearchResult']['Hotels'] ?? [];

                if (count($response['SearchResult']['Hotels']) > 0) {
                    $getCity = self::getCityLatitude(
                        $requestData['geolocation']['latitude'] ?? null,
                        $requestData['geolocation']['longitude'] ?? null
                    );

                    $response['SearchResult']['Hotels'] = $this->getHotelsPriceWithMarkup($response['SearchResult']['Hotels'], 'LowestPackagePrice');
                    $hotelIds = [];

                    foreach ($response['SearchResult']['Hotels'] as &$hotel) {
                        $loyaltyPointsPayload = [
                            'service' => 'Hotel Booking',
                            'amount' => $hotel['Price'],
                            'fromCurrency' => Currency::SAR->value,
                            'toCurrency' => $currency
                        ];
                        $hotel['Points'] = $this->getLoyaltyPoints($loyaltyPointsPayload);

                        $image = $hotel['DefaultImage']['FullSize'] ?? '';
                        $image = str_replace(env('HOTEL_SUPPLIER_LIST_IMAGE_DOMAIN'), env('REHLTE_IMAGE_DOMAIN'), $image);
                        $hotel['DefaultImage']['FullSize'] = $image;
                        $hotel['city_latitude'] = $getCity->latitude ?? '';
                        $hotel['city_longitude'] = $getCity->longitude ?? '';
                        array_push($hotelIds, $hotel['ID']);
                        $hotel['Description'] = '';
                        $hotel['Facilities'] = [];
                        $hotel['DescriptionExists'] = false;
                        $hotel['FacilitiesExists'] = false;
                    }

                    if ($locale == Locale::Arabic->value && count($hotelIds) > 0) {
                        $response['SearchResult']['Hotels'] = $this->getArabicHotelNames($hotelIds, $response['SearchResult']['Hotels']);
                    }

                    $descriptionFacilitiesResponse = $this->getDescriptionFacilities($locale, $hotelIds, $response['SearchResult']['Hotels'], 'ID', $requestData);
                    $response['SearchResult']['Hotels'] = $descriptionFacilitiesResponse['hotels'];
                    $response['SearchResult']['UnMatchedHotelIds'] = $descriptionFacilitiesResponse['unMatchedHotelIds'];
                }

                $result['data'] = $response['SearchResult'];

                $logId = getRandomString();
                GimmonixSearchSession::createGimmonixSearchSession([
                    'log_id' => $logId,
                    'session_id' => $sessionId,
                    'search_id' => $response['SearchResult']['SearchId'],
                ]);
                $result['data']['log_id'] = $logId;
            } elseif ($response && $type == 'filter-list' && isset($response['GetPageResult']) && $response['GetPageResult']) {
                // Check for application-level errors
                if (isset($response['error_analysis'])) {
                    $errorAnalysis = $response['error_analysis'];
                    if (!empty($errorAnalysis['non_recoverable_errors'])) {
                        $result['message'] = $errorAnalysis['non_recoverable_errors'][0]['user_message'];
                        return $result;
                    }
                }

                $result['status'] = true;
                $hotels = $response['GetPageResult']['Hotels'] ?? [];

                // Filter hotels with StarRating > 2
                $filteredHotels = array_filter($hotels, function ($hotel) {
                    return isset($hotel['StarRating']) && (float)$hotel['StarRating'] > 2;
                });

                $filteredHotels = array_values($filteredHotels);
                $response['GetPageResult']['Hotels'] = $filteredHotels;

                if (count($response['GetPageResult']['Hotels']) > 0) {
                    $response['GetPageResult']['Hotels'] = $this->getHotelsPriceWithMarkup($response['GetPageResult']['Hotels'], 'LowestPackagePrice');
                    $hotelIds = [];

                    foreach ($response['GetPageResult']['Hotels'] as &$hotel) {
                        $loyaltyPointsPayload = [
                            'service' => 'Hotel Booking',
                            'amount' => $hotel['Price'],
                            'fromCurrency' => Currency::SAR->value,
                            'toCurrency' => $currency
                        ];
                        $hotel['Points'] = $this->getLoyaltyPoints($loyaltyPointsPayload);

                        array_push($hotelIds, $hotel['ID']);
                        $hotel['DescFacilitiesExists'] = false;
                        $hotel['Description'] = '';
                        $hotel['Facilities'] = [];
                    }

                    if ($locale == Locale::Arabic->value && count($hotelIds) > 0) {
                        $response['GetPageResult']['Hotels'] = $this->getArabicHotelNames($hotelIds, $response['GetPageResult']['Hotels']);
                    }

                    $descriptionFacilitiesResponse = $this->getDescriptionFacilities($locale, $hotelIds, $response['GetPageResult']['Hotels'], 'ID', $requestData);
                    $response['GetPageResult']['Hotels'] = $descriptionFacilitiesResponse['hotels'];
                    $response['GetPageResult']['UnMatchedHotelIds'] = $descriptionFacilitiesResponse['unMatchedHotelIds'];
                }

                $result['data'] = $response['GetPageResult'];
                $result['data']['log_id'] = $logId;
            } else {
                $result['message'] = 'No search results returned from booking system';
            }
        } catch (Exception $e) {
            $result['status'] = false;
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
            \Log::error('Gimmonix API Error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'customer_id' => $customerId,
                'type' => $type
            ]);
        }

        return $result;
    }

    /**
     * Extract search criteria from a complete search object
     * @param array $searchObject - The complete search object containing occupancies
     * @return array - Search criteria in the format expected by groupHotelRooms
     */
    function extractSearchCriteria($searchObject)
    {
        $searchCriteria = [];

        // Check if occupancies exist in the search object
        if (isset($searchObject['occupancies']) && is_array($searchObject['occupancies'])) {
            foreach ($searchObject['occupancies'] as $occupancy) {
                $adultsCount = (int)$occupancy['adults'];
                $kidsAges = [];

                // Extract children ages from paxes array if it exists
                if (isset($occupancy['paxes']) && is_array($occupancy['paxes'])) {
                    foreach ($occupancy['paxes'] as $pax) {
                        if (isset($pax['type']) && $pax['type'] === 'CH' && isset($pax['age'])) {
                            $kidsAges[] = (int)$pax['age'];
                        }
                    }
                }

                $searchCriteria[] = [
                    'AdultsCount' => $adultsCount,
                    'KidsAges' => $kidsAges
                ];
            }
        }

        return $searchCriteria;
    }

    /**
     * Enhanced hotel details with comprehensive error handling
     */
    public function getGimmonixHotelDetails($requestData, $customerId, $logId): array
    {
        $result = [
            'status' => false,
            'data' => [],
            'message' => ''
        ];

        try {
            if (!$this->gimmonixBaseUrl) {
                $this->initializeGimmonixCredentials();
            }

            $currency = $requestData['currency'];
            $locale = $requestData['locale'];
            $hotelId = $requestData['hotelId'];
            $hasLogIdExists = false;

            // Enhanced session validation
            if ($logId) {
                $hasLogIdExists = true;
                $sessionData = GimmonixSearchSession::where('log_id', $logId)->first();

                if ($sessionData) {
                    $sessionCreated = Carbon::parse($sessionData->created_at)->timezone('UTC');
                    $sessionExpiry = $sessionCreated->addMinutes(self::SESSION_EXPIRY_MINUTES);

                    if (now('UTC')->greaterThan($sessionExpiry)) {
                        // Session expired — create a new one
                        $sessionId = $this->getGimmonixSessionId();
                    } else {
                        // Session still valid
                        $sessionId = $sessionData->session_id;
                    }
                } else {
                    // log_id not found — create new session
                    $sessionId = $this->getGimmonixSessionId();
                }
            } else {
                // No logId — new session
                $sessionId = $this->getGimmonixSessionId();
                $logId = getRandomString();
            }

            if (empty($sessionId)) {
                $result['message'] = 'Unable to establish session with booking system';
                return $result;
            }
            $hasErrors = true;
            if ($requestData['isFromListPage'] == 'true') {
                $hotelDetails = ["ID" => $hotelId];
                $hasErrors = false;
            } else {
                $payload = $this->transformGimmonixPayload('list-by-hotel-id', $requestData);
                $payload['searchService']['SessionID'] = $sessionId;
                $response = $this->makeGimmonixRequest('static', 'search', $payload);

                // Check for request-level errors
                if (isset($response['error']) && $response['error']) {
                    $result['message'] = $response['message'];
                    return $result;
                }

                if ($response && isset($response['SearchResult']) && $response['SearchResult']) {
                    // Check for application-level errors
                    if (isset($response['error_analysis'])) {
                        $errorAnalysis = $response['error_analysis'];
                        if (!empty($errorAnalysis['non_recoverable_errors'])) {
                            $result['message'] = $errorAnalysis['non_recoverable_errors'][0]['user_message'];
                            return $result;
                        }
                    }

                    if (count($response['SearchResult']['Hotels']) > 0) {
                        $searchId = $response['SearchResult']['SearchId'];
                        $hotelDetails = $response['SearchResult']['Hotels'][0];

                        $this->createSupplierLog('Hotel', 'Gimmonix', 'hotel-list-by-hotel-id', json_encode($payload), json_encode($response['SearchResult']), $customerId, '', '', $logId);

                        $payload = [
                            'service' => 'Hotel Booking',
                            'amount' => $hotelDetails['LowestPackagePrice'],
                            'fromCurrency' => Currency::SAR->value,
                            'toCurrency' => $currency
                        ];
                        $hotelDetails['Points'] = $this->getLoyaltyPoints($payload);

                        $hasErrors = false;
                    }
                }
            }
            if ($hasErrors == false) {
                // Get Hotel Details - Static content
                $detailsPayload = $this->transformGimmonixPayload('details', [
                    'sessionId' => $sessionId,
                    'hotelId' => $hotelId,
                    'locale' => $locale
                ]);
                $response = $this->makeGimmonixRequest('static', 'GetHotelContent', $detailsPayload);
                if ($response && isset($response['GetHotelContentResult']) && $response['GetHotelContentResult']) {
                    if ($response['GetHotelContentResult']) {
                        $this->createSupplierLog('Hotel', 'Gimmonix', 'hotel-content-by-hotel-id', json_encode($detailsPayload), json_encode($response['GetHotelContentResult']), $customerId, '', '', $logId);

                        $hotelDetails['Descriptions'] = $response['GetHotelContentResult']['Descriptions'];
                        $hotelDetails['Facilities'] = $response['GetHotelContentResult']['Facilities'];
                        $hotelDetails['Photos'] = $response['GetHotelContentResult']['Photos'];
                        $hotelDetails['CheckInFrom'] = $response['GetHotelContentResult']['CheckInFrom'] ?? '';
                        $hotelDetails['CheckOutTo'] = $response['GetHotelContentResult']['CheckOutTo'] ?? '';

                        if (is_array($hotelDetails['Photos']) && count($hotelDetails['Photos']) > 0) {
                            foreach ($hotelDetails['Photos'] as &$photo) {
                                $photo = str_replace(env('HOTEL_SUPPLIER_DETAILS_IMAGE_DOMAIN'), env('REHLTE_IMAGE_DOMAIN'), $photo);
                            }
                        }
                    }
                }

                // Get Room Package Details
                $packagesPayload = $this->transformGimmonixPayload('packages', [
                    'sessionId' => $sessionId,
                    'hotelId' => $hotelId,
                    'locale' => $locale
                ]);
                $response = $this->makeGimmonixRequest('dynamic', '', $packagesPayload);
                if ($response && isset($response['ServiceRequestResult']) && $response['ServiceRequestResult']) {
                    if (
                        $response['ServiceRequestResult'] &&
                        isset($response['ServiceRequestResult']['HotelsGetPackagesResponse']) &&
                        count($response['ServiceRequestResult']['HotelsGetPackagesResponse']) > 0
                    ) {
                        $this->createSupplierLog('Hotel', 'Gimmonix', 'hotel-room-get-packages', json_encode($packagesPayload), json_encode($response['ServiceRequestResult']), $customerId, '', '', $logId);
                        $rooms = $this->getHotelRoomsPriceWithMarkup($response['ServiceRequestResult']['HotelsGetPackagesResponse']['Result'], 'SimplePrice', $currency);
                        $hotelDetails['Rooms'] = $rooms;
                        $hotelDetails['RoomsContent'] = $response['ServiceRequestResult']['HotelsGetPackagesResponse']['RoomsContent'];
                    }
                }
                $hotelDescriptions = [];
                if ($locale == Locale::Arabic->value) {
                    try {
                        $hotelName = HotelName::where('hotel_id', $hotelDetails['ID'])->where('language', Locale::Arabic->value)->select('display_name')->get();
                        if ($hotelName && $hotelName->count() > 0) {
                            $hotelDetails['DisplayName'] = $hotelName[0]->display_name;
                        }
                        $hotelDescriptions = $this->getHotelDescriptions($locale, $hotelDetails['ID']);
                    } catch (Exception $e) {
                        \Log::warning("Error getting Arabic hotel data: " . $e->getMessage());
                    }
                }

                $searchCriteria = $this->extractSearchCriteria($requestData);

                $hotelDetails = $this->transformHotelDetails($locale, $hotelDetails, $hotelDescriptions, $searchCriteria);

                // Set default values for Google services (currently disabled)
                $hotelDetails['ReviewUrl'] = '';
                $hotelDetails['ReviewResponse'] = [];
                $hotelDetails['Reviews'] = [];
                $hotelDetails['AroundThisHotel'] = [];
                $hotelDetails['service_provider'] = 'GIMMONIX';
                $hotelDetails['gimmonix_respose'] = [];

                $result['status'] = true;
                $result['data'] = $hotelDetails;

                if ($hasLogIdExists == false) {
                    GimmonixSearchSession::createGimmonixSearchSession([
                        'log_id' => $logId,
                        'session_id' => $sessionId,
                        'search_id' => $searchId,
                    ]);
                }
                $result['data']['log_id'] = $logId;
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
            \Log::error('Gimmonix Hotel Details API Error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'customer_id' => $customerId,
                'log_id' => $logId
            ]);
        }

        return $result;
    }

    /**
     * Enhanced cancellation policy with better error handling
     */
    public function getGimmonixRoomCancellationPolicy($requestData, $customerId, $logId): array
    {
        $result = [
            'status' => false,
            'data' => [],
            'message' => ''
        ];

        try {
            if (!$this->gimmonixBaseUrl) {
                $this->initializeGimmonixCredentials();
            }

            $currency = $requestData['currency'];
            $locale = $requestData['locale'];

            $sessionId = $this->validateDatabaseSession($logId);
            if (!$sessionId) {
                $result['message'] = 'Invalid or expired session. Please start a new search.';
                return $result;
            }

            $payload = $this->transformGimmonixPayload('cancellation-policy', [
                'hotelId' => $requestData['hotelId'],
                'packageId' => $requestData['packageId'],
                'sessionId' => $sessionId,
            ]);

            $response = $this->makeGimmonixRequest('dynamic', '', $payload);

            // Check for request-level errors
            if (isset($response['error']) && $response['error']) {
                $result['message'] = $response['message'];
                return $result;
            }

            if ($response && isset($response['ServiceRequestResult']) && $response['ServiceRequestResult']) {
                // Check for application-level errors
                if (isset($response['error_analysis'])) {
                    $errorAnalysis = $response['error_analysis'];
                    if (!empty($errorAnalysis['non_recoverable_errors'])) {
                        $result['message'] = $errorAnalysis['non_recoverable_errors'][0]['user_message'];
                        return $result;
                    }
                }

                if (
                    $response['ServiceRequestResult'] &&
                    isset($response['ServiceRequestResult']['HotelCancellationPolicyResponse']) &&
                    count($response['ServiceRequestResult']['Errors']) == 0
                ) {
                    $this->createSupplierLog('Hotel', 'Gimmonix', 'hotel-room-cancellation-policy', json_encode($payload), json_encode($response['ServiceRequestResult']), $customerId, '', '', $logId);
                    $result['status'] = true;
                    $tempCancellationData = $response['ServiceRequestResult']['HotelCancellationPolicyResponse'];
                    $cancellationData = [];

                    if ($tempCancellationData) {
                        $cancellationData['BookingRemarks'] = $tempCancellationData['BookingRemarks'];
                        $cancellationData['CancellationPolicies'] = [];

                        if (array_key_exists('CancellationPolicies', $tempCancellationData)) {
                            $cancellationData['CancellationPoliciesOriginal'] = $tempCancellationData['CancellationPolicies'];
                            $lastIndex = count($tempCancellationData['CancellationPolicies']) - 1;
                            $hasRefundable = false;
                            if (($requestData['refund'] ?? '') == 'Refundable') {
                                $hasRefundable = true;
                                foreach ($tempCancellationData['CancellationPolicies'] as $cancellationPolicy) {
                                    $dateString = ($locale == Locale::Arabic->value) ? 'حتى' : 'Until';
                                    $refundableDate = $this->applyCancellationBufferTime($cancellationPolicy['DateFrom'],true);
                                    $dateString .= ' ' . $refundableDate; 

                                    array_push($cancellationData['CancellationPolicies'], [
                                        "Refundable" => true,
                                        "Currency" => $cancellationPolicy['CancellationFee']['Currency'],
                                        "Amount" => $cancellationPolicy['CancellationFee']['FinalPrice'],
                                        "Title" => "Refundable",
                                        "Subtitle" => $dateString,
                                        "Comments" => "Additional agency and handling fees may apply",
                                    ]);
                                }
                            }

                            $dateString = ($locale == Locale::Arabic->value) ? 'من' : 'From';
                            if(!$hasRefundable){
                                $nonRefundableDate =  $this->applyCancellationBufferTime($tempCancellationData['CancellationPolicies'][$lastIndex]['DateFrom'],true);
                                $dateString .= ' ' . $nonRefundableDate;
                            }else{
                                $nonRefundableDate = $this->applyCancellationBufferTime($refundableDate);
                                $dateString .= ' ' . $nonRefundableDate;
                            }

                            $nonRefundAmount = $tempCancellationData['CancellationPolicies'][$lastIndex]['CancellationFee']['FinalPrice'];

                            // if (isset($requestData['price']) && $requestData['price'] != '' && $requestData['price'] != $nonRefundAmount) {
                            //     $markupAmount = abs($requestData['price'] - $nonRefundAmount);
                            //     $nonRefundAmount += $markupAmount;
                            // }
                            $nonRefundAmount = self::applySalesServiceFee($nonRefundAmount, $requestData['currency']);
                            array_push($cancellationData['CancellationPolicies'], [
                                "Refundable" => false,
                                "Currency" => $tempCancellationData['CancellationPolicies'][$lastIndex]['CancellationFee']['Currency'],
                                "Amount" => $nonRefundAmount,
                                "Title" => "Non-Refundable",
                                "Subtitle" => $dateString,
                                "Comments" => "",
                            ]);
                        }
                    }
                    $result['data'] = $cancellationData;
                } else {
                    $result['message'] = 'Unable to retrieve cancellation policy';
                }
            } else {
                $result['message'] = 'No cancellation policy data returned from booking system';
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
            \Log::error('Gimmonix Cancellation Policy API Error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'customer_id' => $customerId,
                'log_id' => $logId
            ]);
        }

        return $result;
    }

    private static function applySalesServiceFee($originalAmount, $currency)
    {
        if ($originalAmount) {
            $salesMarkup = SalesServiceMarkup::whereServiceName(ServiceType::Hotel->value)
                ->where('status', 'active')
                ->latest()
                ->first();
        } else {
            return $originalAmount;
        }
        if (!$salesMarkup) {
            // $serviceFeeDetails = getServiceFee(ServiceType::Hotel->value);
            // if(!empty($serviceFeeDetails) && isset($serviceFeeDetails['serviceFee'])){
            //     return round($originalAmount + ($serviceFeeDetails['serviceFee'] ?? 0));
            // }
            return $originalAmount;
        }
        $fixedServiceFee = (float)$salesMarkup->fixed_service_fee;
        if ($currency != Currency::SAR->value) {
            $convertedRate = convertCurrencyExchangeRate((float)$salesMarkup->fixed_service_fee, Currency::SAR->value, $currency);
            if ($convertedRate['status'] && isset($convertedRate['data']['convertedRate'])) {
                $fixedServiceFee = $convertedRate['data']['convertedRate'];
            }
        }
        $amount = (float)$originalAmount + $fixedServiceFee;
        $includingServiceFee = $amount + ($amount * (float)$salesMarkup->markup_percentage / 100);
        return round($includingServiceFee, 2);
    }

    /**
     * Enhanced price calculation with error handling and fallbacks
     */
    public function getHotelsPriceWithMarkup($hotelsData, $priceKey)
    {
        try {
            $supplier = 'GIMMONIX';
            // Note: Markup calculation is currently disabled (commented out in original)
            // Keeping the structure for future enabling

            foreach ($hotelsData as $key => $hotelData) {
                $price = $hotelData[$priceKey] ?? 0;
                $hotelsData[$key]['Price'] = number_format($price, '2', '.', '');
            }
        } catch (Exception $e) {
            \Log::error('Error calculating hotel prices with markup: ' . $e->getMessage());
            // Return original prices if calculation fails
            foreach ($hotelsData as $key => $hotelData) {
                $hotelsData[$key]['Price'] = number_format($hotelData[$priceKey] ?? 0, '2', '.', '');
            }
        }

        return $hotelsData;
    }

    /**
     * Enhanced room price calculation with error handling
     */
    public function getHotelRoomsPriceWithMarkup($rooms, $priceKey, $currency)
    {
        try {
            $supplier = 'GIMMONIX';
            // Note: Markup calculation is currently disabled (commented out in original)
            // Keeping the structure for future enabling

            foreach ($rooms as $key => $room) {
                $price = $room[$priceKey] ?? 0;
                $rooms[$key][$priceKey] = number_format($price, '2', '.', '');

                $payload = [
                    'service' => 'Hotel Booking',
                    'amount' => $rooms[$key][$priceKey],
                    'fromCurrency' => Currency::SAR->value,
                    'toCurrency' => $currency
                ];
                $rooms[$key]['LoyaltyPoints'] = $this->getLoyaltyPoints($payload);
            }
        } catch (Exception $e) {
            \Log::error('Error calculating room prices with markup: ' . $e->getMessage());
            // Return original prices if calculation fails
            foreach ($rooms as $key => $room) {
                $rooms[$key][$priceKey] = number_format($room[$priceKey] ?? 0, '2', '.', '');
                $rooms[$key]['LoyaltyPoints'] = 0;
            }
        }

        return $rooms;
    }

    /**
     * Enhanced booking payload generation with validation
     */
    public function generateGimmonixBookingPayload($referenceDetails)
    {
        try {
            $sessionId = $this->validateDatabaseSession($referenceDetails['log_id']);

            if (!$sessionId) {
                throw new Exception('Invalid or expired session for booking');
            }

            $hotelId = $referenceDetails['booking_details']['hotelDetails']['selectedRooms']['HotelId'];
            $roomId = $referenceDetails['booking_details']['hotelDetails']['selectedRooms']['Id'];
            $packageId = $referenceDetails['booking_details']['hotelDetails']['selectedRooms']['PackageId'];
            $leadPaxId = $this->getUuid();
            $passengers = $this->getPassengersFromReference($referenceDetails['booking_details'], $leadPaxId);

            $requestData = [
                'amount' => $referenceDetails['amount'],
                'refId' => $referenceDetails['ref_id'],
                'hotelId' => $hotelId,
                'leadPaxId' => $leadPaxId,
                'roomId' => $roomId,
                'packageId' => $packageId,
                'passengers' => $passengers,
                'sessionId' => $sessionId,
            ];

            $payload = $this->transformGimmonixPayload('book-cash-order', $requestData);
            return $payload;
        } catch (Exception $e) {
            \Log::error('Error generating booking payload: ' . $e->getMessage(), $referenceDetails);
            throw $e;
        }
    }

    /**
     * Enhanced order creation with comprehensive error handling
     */
    public function createGimmonixOrder($payload)
    {
        $result = [
            'status' => false,
            'data' => [],
            'message' => ''
        ];

        try {
            if (!$this->gimmonixBaseUrl) {
                $this->initializeGimmonixCredentials();
            }

            $response = $this->makeGimmonixRequest('dynamic', '', $payload);

            // Check for request-level errors
            if (isset($response['error']) && $response['error']) {
                $result['message'] = $response['message'];
                return $result;
            }

            if ($response && isset($response['ServiceRequestResult']) && $response['ServiceRequestResult']) {
                // Check for application-level errors
                if (isset($response['error_analysis'])) {
                    $errorAnalysis = $response['error_analysis'];
                    if (!empty($errorAnalysis['non_recoverable_errors'])) {
                        $result['status'] = false;
                        $result['message'] = $errorAnalysis['non_recoverable_errors'][0]['user_message'];
                        $result['detailed_message'] = $errorAnalysis['non_recoverable_errors'][0]['detailed_message'] ?? '';
                        return $result;
                    }
                }

                if (
                    $response['ServiceRequestResult'] &&
                    count($response['ServiceRequestResult']['Errors']) > 0
                ) {
                    $result['status'] = false;
                    $result['message'] = $response['ServiceRequestResult']['Errors'];
                } elseif (
                    $response['ServiceRequestResult'] &&
                    array_key_exists('HotelOrderBookResponse', $response['ServiceRequestResult']) == false &&
                    count($response['ServiceRequestResult']['Errors']) == 0
                ) {
                    $result['status'] = false;
                    $result['message'] = [[
                        'ErrorCode' => '',
                        'ErrorText' => '',
                        'Message' => 'Empty response from hotel booking service'
                    ]];
                } else {
                    $result['status'] = true;
                    $result['data'] = $response['ServiceRequestResult']['HotelOrderBookResponse'];
                }
            } else {
                $result['message'] = 'No response received from booking system';
            }
        } catch (Exception $e) {
            $result['message'] = 'Booking failed: ' . $e->getMessage();
            \Log::error('Gimmonix Order Creation Error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'payload' => $payload
            ]);
        }

        return $result;
    }

    /**
     * Enhanced hotel descriptions with error handling
     */
    public function getHotelDescriptions($locale, $hotelId)
    {
        try {
            $descriptionList = HotelDescriptionAr::where('language', $locale)
                ->where('hotel_id', $hotelId)
                ->get();

            if ($descriptionList && $descriptionList->count() > 0) {
                $descriptionList = $descriptionList->toArray();
                $tempDescriptions = [];

                foreach ($descriptionList as $description) {
                    array_push($tempDescriptions, [
                        'Title' => $description['title'],
                        'Line' => $description['title'],
                        'Descrption' => $description['paragraph']
                    ]);
                }

                $descriptions = $this->groupDescriptionForArabic($tempDescriptions);
            } else {
                $descriptions = [
                    'Highlighted_Description_Names' => [],
                    'Description' => '',
                    'Descriptions' => []
                ];
            }
        } catch (Exception $e) {
            \Log::error('Error getting hotel descriptions: ' . $e->getMessage(), [
                'hotel_id' => $hotelId,
                'locale' => $locale
            ]);

            $descriptions = [
                'Highlighted_Description_Names' => [],
                'Description' => '',
                'Descriptions' => []
            ];
        }

        return $descriptions;
    }

    /**
     * Enhanced booking cancellation with better error handling
     */
    public function gimmonixCancelBooking($segmentId)
    {
        $result = [
            'status' => false,
            'data' => [],
            'message' => ''
        ];

        try {
            if (!$this->gimmonixBaseUrl) {
                $this->initializeGimmonixCredentials();
            }

            $payload = $this->transformGimmonixPayload('cancel-booking', [
                'username' => $this->gimmonixUsername,
                'password' => $this->gimmonixPassword,
                'segmentId' => $segmentId,
            ]);
            $result['payload'] = $payload;

            return $response = $this->makeGimmonixRequest('dynamic', '', $payload);

            // Check for request-level errors
            if (isset($response['error']) && $response['error']) {
                $result['message'] = $response['message'];
                return $result;
            }

            if ($response && isset($response['ServiceRequestResult']) && $response['ServiceRequestResult']) {
                // Check for application-level errors
                if (isset($response['error_analysis'])) {
                    $errorAnalysis = $response['error_analysis'];
                    if (!empty($errorAnalysis['non_recoverable_errors'])) {
                        $result['status'] = false;
                        $result['message'] = $errorAnalysis['non_recoverable_errors'][0]['user_message'];
                        return $result;
                    }
                }

                if (
                    $response['ServiceRequestResult'] &&
                    count($response['ServiceRequestResult']['Errors']) > 0
                ) {
                    $errors = $response['ServiceRequestResult']['Errors'];
                    $errorMessages = [];

                    if (is_array($errors)) {
                        foreach ($errors as $error) {
                            if (($error['Sevirity'] ?? $error['Severity'] ?? '') == 'Error') {
                                array_push($errorMessages, ($error['ErrorCode'] . ' ' . $error['ErrorText'] . ' ' . $error['Message']));
                            }
                        }
                        $errorMessage = implode(', ', $errorMessages);
                    } else {
                        $errorMessage = $errors;
                    }

                    $result['status'] = false;
                    $result['message'] = $errorMessage;
                } elseif (
                    $response['ServiceRequestResult'] &&
                    array_key_exists('HotelOrderBookResponse', $response['ServiceRequestResult']) == false &&
                    count($response['ServiceRequestResult']['Errors']) == 0
                ) {
                    $result['status'] = false;
                    $result['message'] = 'Empty response from hotel cancellation service';
                } else {
                    $result['status'] = true;
                    $result['data'] = $response['ServiceRequestResult']['HotelBookCancelResponse'];
                }
            } else {
                $result['message'] = 'No response received from cancellation system';
            }
        } catch (Exception $e) {
            $result['message'] = 'Cancellation failed: ' . $e->getMessage();
            \Log::error('Gimmonix Cancellation Error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'segment_id' => $segmentId
            ]);
        }

        return $result;
    }

    /**
     * Enhanced hotels description retrieval with comprehensive error handling
     */
    public function getGimmonixHotelsDescription($requestData)
    {
        $result = [
            'status' => false,
            'data' => [],
            'message' => '',
            'total' => 0
        ];

        try {
            $locale = $requestData['locale'] ?? Locale::English->value;
            $logId = $requestData['log_id'] ?? '';
            $hotelIDs = $requestData['hotelIDs'] ?? [];
            $type = $requestData['type'] ?? '';

            if (is_array($hotelIDs)) {
                $hotelIDsArr = $hotelIDs;
            } else {
                $hotelIDsArr = explode(",", $hotelIDs);
            }

            $hotels = [];
            foreach ($hotelIDsArr as $hotelID) {
                array_push($hotels, [
                    'HotelID' => $hotelID,
                    'Description' => '',
                    'Facilities' => [],
                    'DescriptionExists' => false,
                    'DataSource' => 'DB'
                ]);
            }

            if ($type != 'description') {
                $descriptionFacilitiesResponse = $this->getIndividualDescriptionFacilities($locale, $hotelIDsArr, $hotels, 'HotelID');
                $hotels = $descriptionFacilitiesResponse['hotels'];
                $unMatchedHotelIds = $descriptionFacilitiesResponse['unMatchedHotelIds'];

                if (count($unMatchedHotelIds) > 0) {
                    if (!$this->gimmonixBaseUrl) {
                        $this->initializeGimmonixCredentials();
                    }

                    if (empty($hotelIDs) || !is_array($hotelIDs)) {
                        $result['message'] = 'Invalid or missing hotel IDs.';
                        return $result;
                    }

                    // Fetch session ID
                    $sessionId = null;
                    if ($logId !== '') {
                        $sessionId = GimmonixSearchSession::where('log_id', $logId)->value('session_id');
                    }
                    if (!$sessionId) {
                        $sessionId = $this->getGimmonixSessionId();
                    }

                    if (!$sessionId) {
                        $result['message'] = 'Session ID could not be found or created.';
                        return $result;
                    }

                    $gimmonixHotels = [];
                    $gimmonixHotels = [];
                    foreach ($hotelIDs as $hotelId) {
                        try {
                            $detailsResponse = $this->getGimmonixHotelShortDetails($sessionId, $hotelId, $locale);
                            if ($detailsResponse['status'] === true && !empty($detailsResponse['data'])) {
                                $facilityData = self::translateHotelListFacilities($detailsResponse['data']['facilities'], $hotelId, $locale);
                                $gimmonixHotels[] = [

                                    'DataSource' => 'Gimmonix',
                                    'HotelID' => $hotelId,
                                    'DescriptionExists' => $detailsResponse['data']['description'] != '' ? true : false,
                                    // 'Facilities' => $detailsResponse['data']['facilities'] ?? [],
                                    'Facilities' => $facilityData['Facilities'] ?? [],
                                ];
                            }
                        } catch (Exception $e) {
                            \Log::warning("Error getting details for hotel {$hotelId}: " . $e->getMessage());
                        }
                    }
                    $hotels = array_merge($hotels, $gimmonixHotels);
                }
            }
            // Get Description From Gimmonix if not available in DB
            else if ($type == 'description') {

                if (!$this->gimmonixBaseUrl) {
                    $this->initializeGimmonixCredentials();
                }

                if (empty($hotelIDs) || !is_array($hotelIDs)) {
                    $result['message'] = 'Invalid or missing hotel IDs.';
                    return $result;
                }

                // Fetch session ID
                $sessionId = null;
                if ($logId !== '') {
                    $sessionId = GimmonixSearchSession::where('log_id', $logId)->value('session_id');
                }
                if (!$sessionId) {
                    $sessionId = $this->getGimmonixSessionId();
                }

                if (!$sessionId) {
                    $result['message'] = 'Session ID could not be found or created.';
                    return $result;
                }

                $gimmonixHotels = [];
                foreach ($hotelIDs as $hotelId) {
                    try {
                        $detailsResponse = $this->getGimmonixHotelShortDetails($sessionId, $hotelId, $locale);
                        if ($detailsResponse['status'] === true && !empty($detailsResponse['data'])) {
                            $gimmonixHotels[] = [
                                'DataSource' => 'Gimmonix',
                                'HotelID' => $hotelId,
                                'Description' => $detailsResponse['data']['description'] ?? '',
                                'DescriptionExists' => $detailsResponse['data']['description'] != '' ? true : false,
                            ];
                        } else {
                            $gimmonixHotels[] = [
                                'HotelID' => $hotelId,
                                'Description' => '',
                                'DescriptionExists' => false,
                                'DataSource' => 'Gimmonix'
                            ];
                        }
                    } catch (Exception $e) {
                        \Log::warning("Error getting description for hotel {$hotelId}: " . $e->getMessage());
                        $gimmonixHotels[] = [
                            'HotelID' => $hotelId,
                            'Description' => '',
                            'DescriptionExists' => false,
                            'DataSource' => 'Gimmonix'
                        ];
                    }
                }
                $hotels = $gimmonixHotels;
            }

            if (!empty($hotels)) {
                $result['status'] = true;
                $result['data'] = $hotels;
                $result['message'] = count($hotels) == 0 ? 'No Hotels Descriptions and Facilities Found' : count($hotels) . ' Hotel Descriptions and Facilities fetched successfully.';
            } else {
                $result['message'] = 'No hotel data was retrieved.';
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
            \Log::error('Gimmonix Hotels Description Error: ' . $e->getMessage(), [
                'request_data' => $requestData
            ]);
        }

        return $result;
    }

    /**
     * Enhanced around hotel details with comprehensive error handling
     */
    public function getAroundthisHotelDetails($requestData, $customerId, $logId): array
    {
        $result = [
            'status' => false,
            'data' => [],
            'message' => ''
        ];

        try {
            if (!$this->gimmonixBaseUrl) {
                $this->initializeGimmonixCredentials();
            }

            $currency = $requestData['currency'];
            $locale = $requestData['locale'];

            if ($logId) {
                $sessionId = $this->validateDatabaseSession($logId);
                if (!$sessionId) {
                    $sessionId = $this->getGimmonixSessionId();
                }
            } else {
                $sessionId = $this->getGimmonixSessionId();
            }

            if (empty($sessionId)) {
                $result['message'] = 'Unable to establish session with booking system';
                return $result;
            }

            $payload = $this->transformGimmonixPayload('list-by-hotel-id', $requestData);
            $payload['searchService']['SessionID'] = $sessionId;
            $response = $this->makeGimmonixRequest('static', 'search', $payload);

            // Check for request-level errors
            if (isset($response['error']) && $response['error']) {
                $result['message'] = $response['message'];
                return $result;
            }

            if ($response && isset($response['SearchResult']) && $response['SearchResult']) {
                // Check for application-level errors
                if (isset($response['error_analysis'])) {
                    $errorAnalysis = $response['error_analysis'];
                    if (!empty($errorAnalysis['non_recoverable_errors'])) {
                        $result['message'] = $errorAnalysis['non_recoverable_errors'][0]['user_message'];
                        return $result;
                    }
                }

                if (count($response['SearchResult']['Hotels']) > 0) {
                    $searchId = $response['SearchResult']['SearchId'];
                    $hotelDetails = $response['SearchResult']['Hotels'][0];

                    $this->createSupplierLog('Hotel', 'Gimmonix', 'hotel-list-by-hotel-id', json_encode($payload), json_encode($response['SearchResult']), $customerId, '', '', $logId);

                    $payload = [
                        'service' => 'Hotel Booking',
                        'amount' => $hotelDetails['LowestPackagePrice'],
                        'fromCurrency' => Currency::SAR->value,
                        'toCurrency' => $currency
                    ];
                    $hotelDetails['Points'] = $this->getLoyaltyPoints($payload);

                    // Get Hotel Details
                    $detailsPayload = $this->transformGimmonixPayload('details', [
                        'sessionId' => $sessionId,
                        'hotelId' => $hotelDetails['ID'],
                        'locale' => $locale
                    ]);

                    $response = $this->makeGimmonixRequest('static', 'GetHotelContent', $detailsPayload);
                    if ($response && isset($response['GetHotelContentResult']) && $response['GetHotelContentResult']) {
                        if ($response['GetHotelContentResult']) {
                            $this->createSupplierLog('Hotel', 'Gimmonix', 'hotel-content-by-hotel-id', json_encode($detailsPayload), json_encode($response['GetHotelContentResult']), $customerId, '', '', $logId);

                            $hotelDetails['Descriptions'] = $response['GetHotelContentResult']['Descriptions'];
                            $hotelDetails['Facilities'] = $response['GetHotelContentResult']['Facilities'];
                            $hotelDetails['Photos'] = $response['GetHotelContentResult']['Photos'];
                            $hotelDetails['CheckInFrom'] = $response['GetHotelContentResult']['CheckInFrom'] ?? '';
                            $hotelDetails['CheckOutTo'] = $response['GetHotelContentResult']['CheckOutTo'] ?? '';
                        }
                    }

                    // Get Package Details
                    $packagesPayload = $this->transformGimmonixPayload('packages', [
                        'sessionId' => $sessionId,
                        'hotelId' => $hotelDetails['ID'],
                        'locale' => $locale
                    ]);

                    $response = $this->makeGimmonixRequest('dynamic', '', $packagesPayload);
                    if ($response && isset($response['ServiceRequestResult']) && $response['ServiceRequestResult']) {
                        if (
                            $response['ServiceRequestResult'] &&
                            isset($response['ServiceRequestResult']['HotelsGetPackagesResponse']) &&
                            count($response['ServiceRequestResult']['HotelsGetPackagesResponse']) > 0
                        ) {
                            $this->createSupplierLog('Hotel', 'Gimmonix', 'hotel-room-get-packages', json_encode($packagesPayload), json_encode($response['ServiceRequestResult']), $customerId, '', '', $logId);
                            $rooms = $this->getHotelRoomsPriceWithMarkup($response['ServiceRequestResult']['HotelsGetPackagesResponse']['Result'], 'SimplePrice', $currency);
                            $hotelDetails['rooms'] = $rooms;
                            $hotelDetails['roomsContent'] = $response['ServiceRequestResult']['HotelsGetPackagesResponse']['RoomsContent'];
                        }
                    }

                    // Enhanced Google Reviews and Places integration
                    try {
                        $googleReviewGetPlaceIdUrl = 'https://maps.googleapis.com/maps/api/place/nearbysearch/json?location=' . $hotelDetails['GeoLocation']['Latitude'] . ',' . $hotelDetails['GeoLocation']['Longitude'] . '&rankby=distance&name=' . urlencode($hotelDetails['DisplayName']) . '&key=' . env('GOOGLE_PLACES_MAP_KEY');
                        $response = $this->fetchUrlData($googleReviewGetPlaceIdUrl);
                        $hotelDetails['reviewUrl'] = $googleReviewGetPlaceIdUrl;
                        $hotelDetails['reviewResponse'] = $response;
                        $hotelDetails['reviews'] = [];

                        if ($response['status'] && count($response['data']['results']) > 0) {
                            $hotelGoogleReviewUrl = 'https://maps.googleapis.com/maps/api/place/details/json?place_id=' . $response['data']['results'][0]['place_id'] . '&https://developers.google.com/maps/documentation/places/web-service/details&language=en&key=' . env('GOOGLE_PLACES_MAP_KEY');
                            $response = $this->fetchUrlData($hotelGoogleReviewUrl);
                            if ($response['status']) {
                                $hotelDetails['reviews'] = $response['data']['result']['reviews'];
                            }
                        }

                        $hotelDetails = $this->getGoogleReviews($hotelDetails);
                        $hotelDetails['aroundThisHotel'] = $this->getGooglePlaces($hotelDetails);
                    } catch (Exception $e) {
                        \Log::warning("Google APIs error: " . $e->getMessage());
                        $hotelDetails['reviewUrl'] = '';
                        $hotelDetails['reviewResponse'] = [];
                        $hotelDetails['reviews'] = [];
                        $hotelDetails['aroundThisHotel'] = [];
                    }

                    $hotelDetails['service_provider'] = 'GIMMONIX';
                    $hotelDetails['gimmonix_respose'] = $response;

                    $result['status'] = true;
                    $result['data'] = $hotelDetails;
                } else {
                    $result['message'] = 'Hotel not found';
                }
            } else {
                $result['message'] = 'No hotel details returned from booking system';
            }
        } catch (Exception $e) {
            $result['message'] = 'Something went wrong: ' . $e->getMessage();
            \Log::error('Gimmonix Around Hotel Details API Error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'customer_id' => $customerId,
                'log_id' => $logId
            ]);
        }

        return $result;
    }
}
