<?php

namespace App\Traits;

use App\Enums\BookingStatus;
use App\Enums\Currency;
use App\Enums\ServiceType as EnumsServiceType;
use App\Models\Bookings;
use App\Models\BookingCheckout;
use App\Models\BookingAdditionalDetail;
use App\Models\BookingHistory;
use App\Models\Cashback;
use App\Models\Country;
use App\Models\CurrencyExchangeRates;
use App\Models\Customer;
use App\Models\CustomerTraveller;
use App\Models\ServiceType;
use App\Models\Setting;
use App\Models\SmsTemplate;
use App\Models\SmsTemplateI18ns;
use App\Models\Suppliers;
use App\Traits\EmailService;
use App\Traits\SmsService;
use App\Traits\WhatsappService;
use App\Traits\LoyaltyPointsService;
use App\Traits\HotelBeds;
use App\Models\DefaultMarkup;
use App\Models\SupplierLog;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Config;
use DateTime;
use Exception;
use App\Models\EditHotelDetail;
use App\Models\EditFlightDetail;

trait BookingHelper
{
    use EmailService, LoyaltyPointsService, SmsService, WhatsappService;

    public function getReferenceDetails($refId)
    {
        $savedCheckoutDetails = BookingCheckout::getBookingCheckout($refId);
        if ($savedCheckoutDetails['status'] == true) {
            $data = $savedCheckoutDetails['data'];
            $data['booking_details'] = json_decode($data['booking_details'], true);
            return [
                'status' => true,
                'data' => $data
            ];
        } else {
            return ['status' => false, 'data' => []];
        }
    }

    public function createBookingHistoryFromBooking($bookingId)
    {
        $bookingDetails = Bookings::whereId($bookingId)->first()->toArray();

        $historyPayload = [
            'booking_id' => $bookingDetails['id'],
            'booking_status' => $bookingDetails['booking_status'],
            'booking_sub_status' => $bookingDetails['booking_sub_status'],
            'booking_status_message' => $bookingDetails['booking_status_message'],
        ];
        BookingHistory::createBookingHistory($historyPayload);
    }

    public function createBookingPayload($referenceDetails)
    {
        $requestPayload = ['booking_status' => BookingStatus::Pending->value];
        $isGuestBooking = ($referenceDetails['booking_details']['userDetails']['userId'] != '') ? false : true;
        $currencyConversionRate = CurrencyExchangeRates::where('from_currency_code', Currency::SAR->value)->where('to_currency_code', $referenceDetails['booking_details']['bookingDetails']['customer_currency'])->get('exchange_rate')[0]['exchange_rate'];
        $processedPrice = $referenceDetails['booking_details']['processedPrice'];

        switch ($referenceDetails['booking_type']) {
            case EnumsServiceType::Flight->value:
                $bookingDescription = '';
                $itineraries = $referenceDetails['booking_details']['flightDetails']['flightInfo'][0]['itineraries'];
                $itineraryCount = count($itineraries);
                foreach ($itineraries as $itinerary) {
                    //count itineraries's count and get origin and destination code to check trip type start
                    if ($itineraryCount == 1) {
                        // Single itinerary
                        $departure = $itineraries[0]['segments'][0]['departure']['iataCode'];
                        $arrival = $itineraries[0]['segments'][count($itineraries[0]['segments']) - 1]['arrival']['iataCode'];
                        $bookingDescription =  "From $departure to $arrival (One-way)";
                    } elseif ($itineraryCount == 2) {
                        // Round-trip or multi-city
                        $firstDeparture = $itineraries[0]['segments'][0]['departure']['iataCode'];
                        $secondArrival = $itineraries[1]['segments'][count($itineraries[1]['segments']) - 1]['arrival']['iataCode'];

                        if ($firstDeparture == $secondArrival) {
                            $bookingDescription =  "From {$itineraries[0]['segments'][0]['departure']['iataCode']} to {$itineraries[1]['segments'][0]['departure']['iataCode']} (Round-trip)";
                        } else {
                            // Multi-city
                            $segments = [];
                            foreach ($itineraries as $itinerary) {
                                foreach ($itinerary['segments'] as $segment) {
                                    $segments[] = $segment['departure']['iataCode'] . '-' . $segment['arrival']['iataCode'];
                                }
                            }
                            $multiCityRoute = implode(' via ', $segments);
                            $bookingDescription =  "From {$itineraries[0]['segments'][0]['departure']['iataCode']} via $multiCityRoute to {$itineraries[$itineraryCount - 1]['segments'][count($itineraries[$itineraryCount - 1]['segments']) - 1]['arrival']['iataCode']} (Multi-city)";
                        }
                    } else {
                        // Multi-city
                        $segments = [];
                        foreach ($itineraries as $itinerary) {
                            foreach ($itinerary['segments'] as $segment) {
                                $segments[] = $segment['departure']['iataCode'] . '-' . $segment['arrival']['iataCode'];
                            }
                        }
                        $multiCityRoute = implode(' via ', $segments);
                        $bookingDescription =  "From {$itineraries[0]['segments'][0]['departure']['iataCode']} via $multiCityRoute to {$itineraries[$itineraryCount - 1]['segments'][count($itineraries[$itineraryCount - 1]['segments']) - 1]['arrival']['iataCode']} (Multi-city)";
                    }
                }
                $processedPrice = $referenceDetails['booking_details']['processedPrice'];
                $bookingDate = '';
                if (array_key_exists('bookingDate', $referenceDetails['booking_details']['bookingDetails'])) {
                    $bookingDate = $referenceDetails['booking_details']['bookingDetails']['bookingDate'];
                } elseif (array_key_exists('lastTicketingDate', $referenceDetails['booking_details']['flightDetails']['flightInfo'][0])) {
                    $bookingDate = $referenceDetails['booking_details']['flightDetails']['flightInfo'][0]['lastTicketingDate'];
                }
                $requestPayload['booking_type'] = EnumsServiceType::Flight->value;
                $requestPayload['booking_start_date'] = $bookingDate;
                $requestPayload['customer_id'] = $referenceDetails['booking_details']['userDetails']['userId'];
                $requestPayload['is_guest'] = $isGuestBooking ? true : false;
                $requestPayload['first_name'] = $referenceDetails['booking_details']['passengerDetails'][0]['first_name'];
                $requestPayload['last_name'] = $referenceDetails['booking_details']['passengerDetails'][0]['last_name'];
                $requestPayload['email'] = $referenceDetails['booking_details']['userDetails']['email'];
                $requestPayload['phone_country_code'] = $referenceDetails['booking_details']['userDetails']['phoneCode'];
                $requestPayload['phone'] = $referenceDetails['booking_details']['userDetails']['phoneNumber'];
                $requestPayload['supplier_currency'] = $referenceDetails['booking_details']['bookingDetails']['customer_currency'];
                $requestPayload['customer_language_code'] = $referenceDetails['booking_details']['bookingDetails']['customer_language_code'];
                $requestPayload['billing_address1'] = (isset($referenceDetails['booking_details']['billingDetails']['address']) ? $referenceDetails['booking_details']['billingDetails']['address'] : '');
                $requestPayload['billing_address2'] = (isset($referenceDetails['booking_details']['billingDetails']['address1']) ? $referenceDetails['booking_details']['billingDetails']['address1'] : '');
                $requestPayload['billing_city'] = (isset($referenceDetails['booking_details']['billingDetails']['city']) ? $referenceDetails['booking_details']['billingDetails']['city'] : '');
                $requestPayload['billing_state'] = (isset($referenceDetails['booking_details']['billingDetails']['state']) ? $referenceDetails['booking_details']['billingDetails']['state'] : '');
                $requestPayload['billing_country'] = (isset($referenceDetails['booking_details']['billingDetails']['country']) ? $referenceDetails['booking_details']['billingDetails']['country'] : '');
                $requestPayload['billing_zip'] = (isset($referenceDetails['booking_details']['billingDetails']['zipcode']) ? $referenceDetails['booking_details']['billingDetails']['zipcode'] : '');
                $requestPayload['supplier_id'] = $referenceDetails['booking_details']['bookingDetails']['supplier_id'];
                $requestPayload['booking_date'] = date('Y-m-d');
                $requestPayload['booking_points'] = $processedPrice['loyaltyPoints'];
                $requestPayload['service_id'] = $referenceDetails['booking_details']['bookingDetails']['service_type_id'];
                $requestPayload['agency_id'] = $referenceDetails['booking_details']['bookingDetails']['agencyId'];
                $requestPayload['description'] = $bookingDescription;

                $requestPayload['customer_currency'] = $processedPrice['currency'];
                $requestPayload['customer_language_code'] = $referenceDetails['booking_details']['bookingDetails']['customer_language_code'];
                $requestPayload['sub_total'] = $processedPrice['totalPrice'];
                $requestPayload['tax'] = $processedPrice['vat'];
                $requestPayload['s_markup_fee'] = $processedPrice['markupFee'];
                $requestPayload['s_service_fee'] = $processedPrice['serviceFee'];
                $requestPayload['s_markup_service_fee'] = $processedPrice['markupServiceFee'];
                $requestPayload['s_tax'] = $processedPrice['markupServiceFee'];
                $requestPayload['s_charge'] = "0";
                $requestPayload['s_discount_type'] = "0";
                $requestPayload['s_discount_value'] = "0";
                $requestPayload['s_discount'] = "0";
                $requestPayload['t_discount_type'] = "0";
                $requestPayload['t_discount_value'] = "0";
                $requestPayload['t_discount'] = "0";
                $requestPayload['t_markup_type'] = $processedPrice['markupType'];
                $requestPayload['t_markup_value'] = $processedPrice['markupValue'];
                $requestPayload['t_markup'] = "0";
                $requestPayload['chargeable_seat_price'] = 0; //$processedPrice['chargeableSeatPrice'];
                $requestPayload['travelers_price'] = 0; //$processedPrice['travelersPrice'];
                $requestPayload['currency_conversion_rate'] = $currencyConversionRate;
                $requestPayload['currency_markup'] = $processedPrice['vat'];
                $requestPayload['booking_from'] = $referenceDetails['booking_details']['searchDetails']['originDevice'];
                $requestPayload['redeem_points'] = 0;
                $requestPayload['redeem_amount'] = 0;
                $requestPayload['sub_total'] = $processedPrice['totalPrice'];
                $requestPayload['total'] = $processedPrice['grandTotal'];
                $requestPayload['additional_details'] = $referenceDetails['booking_details']['searchDetails']['searchType'];
                break;
            case EnumsServiceType::Hotel->value:
                $requestPayload['booking_type'] = EnumsServiceType::Hotel->value;
                $requestPayload['booking_date'] = date('Y-m-d');
                $requestPayload['booking_points'] = $processedPrice['loyaltyPoints'];
                $requestPayload['service_id'] = $referenceDetails['booking_details']['bookingDetails']['service_type_id'];
                $requestPayload['agency_id'] = $referenceDetails['booking_details']['bookingDetails']['agencyId'];
                $requestPayload['customer_currency'] = $processedPrice['currency'];
                $requestPayload['customer_language_code'] = $referenceDetails['booking_details']['bookingDetails']['customer_language_code'];
                $requestPayload['sub_total'] = $processedPrice['totalPrice'];
                $requestPayload['tax'] = $processedPrice['vat'];
                $requestPayload['s_markup_fee'] = $processedPrice['markupFee'];
                $requestPayload['s_service_fee'] = $processedPrice['serviceFee'];
                $requestPayload['s_markup_service_fee'] = $processedPrice['markupServiceFee'];
                $requestPayload['s_tax'] = $processedPrice['markupServiceFee'];
                $requestPayload['s_charge'] = "0";
                $requestPayload['s_discount_type'] = "0";
                $requestPayload['s_discount_value'] = "0";
                $requestPayload['s_discount'] = "0";
                $requestPayload['t_discount_type'] = "0";
                $requestPayload['t_discount_value'] = "0";
                $requestPayload['t_discount'] = "0";
                $requestPayload['t_markup_type'] = $processedPrice['markupType'];
                $requestPayload['t_markup_value'] = $processedPrice['markupValue'];
                $requestPayload['t_markup'] = "0";
                $requestPayload['currency_conversion_rate'] = $currencyConversionRate;
                $requestPayload['currency_markup'] = $processedPrice['vat'];
                $requestPayload['booking_from'] = $referenceDetails['booking_details']['searchDetails']['searchFrom'];
                if ($processedPrice['isRedeemUsed']) {
                    $requestPayload['redeem_points'] = str_replace(',', '', $processedPrice['redeemDetails']['maxRedeemPoints']);
                    $requestPayload['redeem_amount'] = $processedPrice['redeemDetails']['maxRedeemPrice'];
                } else {
                    $requestPayload['redeem_points'] = 0;
                    $requestPayload['redeem_amount'] = 0;
                }
                $requestPayload['sub_total'] = $referenceDetails['booking_details']['processedPrice']['totalPrice'];
                $requestPayload['total'] = $referenceDetails['booking_details']['processedPrice']['grandTotal'];
                $requestPayload['is_guest'] = $isGuestBooking ? true : false;
                $requestPayload['customer_id'] = $referenceDetails['booking_details']['userDetails']['userId'];
                $requestPayload['first_name'] = $referenceDetails['booking_details']['userDetails']['firstName'];
                $requestPayload['last_name'] = $referenceDetails['booking_details']['userDetails']['lastName'];
                $requestPayload['email'] = $referenceDetails['booking_details']['userDetails']['email'];
                $requestPayload['phone_country_code'] = $referenceDetails['booking_details']['userDetails']['phoneCode'];
                $requestPayload['phone'] = $referenceDetails['booking_details']['userDetails']['phoneNumber'];
                $requestPayload['supplier_currency'] = $referenceDetails['booking_details']['bookingDetails']['customer_currency'];
                $requestPayload['customer_language_code'] = $referenceDetails['booking_details']['bookingDetails']['customer_language_code'];
                $requestPayload['billing_address1'] = (isset($referenceDetails['booking_details']['billingDetails']['address']) ? $referenceDetails['booking_details']['billingDetails']['address'] : '');
                $requestPayload['billing_address2'] = (isset($referenceDetails['booking_details']['billingDetails']['address1']) ? $referenceDetails['booking_details']['billingDetails']['address1'] : '');
                $requestPayload['billing_city'] = (isset($referenceDetails['booking_details']['billingDetails']['city']) ? $referenceDetails['booking_details']['billingDetails']['city'] : '');
                $requestPayload['billing_state'] = (isset($referenceDetails['booking_details']['billingDetails']['state']) ? $referenceDetails['booking_details']['billingDetails']['state'] : '');
                $requestPayload['billing_country'] = (isset($referenceDetails['booking_details']['billingDetails']['country']) ? $referenceDetails['booking_details']['billingDetails']['country'] : '');
                $requestPayload['billing_zip'] = (isset($referenceDetails['booking_details']['billingDetails']['zipcode']) ? $referenceDetails['booking_details']['billingDetails']['zipcode'] : '');
                $requestPayload['supplier_id'] = $referenceDetails['booking_details']['bookingDetails']['supplier_id'];
                $requestPayload['supplier_name'] = $referenceDetails['booking_details']['hotelDetails']['selectedRooms']['SupplierName'];
                $requestPayload['booking_start_date'] = $referenceDetails['booking_details']['searchDetails']['stay']['checkIn'];

                $hotelBookingDescription = '';
                if ($referenceDetails['service_provider'] == 'HOTEL_BEDS') {
                    $hotelBookingDescription = $referenceDetails['booking_details']['hotelDetails']['hotelDetails']['DisplayName'];
                } elseif ($referenceDetails['service_provider'] == 'GIMMONIX') {
                    $hotelBookingDescription = $referenceDetails['booking_details']['hotelDetails']['hotelDetails']['DisplayName'];
                }
                $requestPayload['description'] = $hotelBookingDescription;
                break;
        }
        $requestPayload['ref_id'] = $referenceDetails['ref_id'];
        $requestPayload['addon_added'] = $processedPrice['isAddOnUsed'];
        $requestPayload['addon_amount'] = $processedPrice['addOnAmount'];
        $requestPayload['coupon_used'] = $processedPrice['isCouponUsed'];
        if ($processedPrice['isCouponUsed']) {
            $requestPayload['coupon_code'] = $referenceDetails['booking_details']['paymentDetails']['coupon_details']['code'];
        }
        $requestPayload['coupon_amount'] = $processedPrice['couponAmount'];
        $requestPayload['redeem_used'] = $processedPrice['isRedeemUsed'];
        $requestPayload['redeem_amount'] = $processedPrice['redeemAmount'];
        $requestPayload['wallet_used'] = $processedPrice['isWalletUsed'];
        $requestPayload['wallet_amount'] = $processedPrice['walletAmount'];
        $requestPayload['service_provider'] = $referenceDetails['service_provider'];
        return $requestPayload;
    }

    public function createOrUpdateBooking($referenceDetails)
    {
        $result = [
            'status' => false,
            'data' => [],
            'type' => 'create',
            'message' => 'Unable to create initial order. Please try again.'
        ];
        $createBookingPayload = $this->createBookingPayload($referenceDetails);
        $rehlteRefNoDetails = $this->getRehlteRefNoDetails();
        $createBookingPayload['rehtle_ref'] = $rehlteRefNoDetails['value'];

        $refId = $referenceDetails['ref_id'];
        $bookingDetails = Bookings::where('ref_id', $refId)->first();
        if ($bookingDetails) {
            $bookingDetails = $bookingDetails->toArray();
            $this->createBookingHistoryFromBooking($bookingDetails['id']);

            $updateBookingResponse = Bookings::updateBooking($bookingDetails['id'], $createBookingPayload);


            if($createBookingPayload['booking_type']== EnumsServiceType::Hotel->value)
            {
                EditHotelDetail::updateOrCreate(
                    ['booking_id' => $updateBookingResponse['data']->id ?? ''],
                    [
                        'first_name' => $createBookingPayload['first_name'] ?? '',
                        'last_name' => $createBookingPayload['last_name'] ?? '',
                        'email' => $createBookingPayload['email'] ?? '',
                        'booking_date' => $createBookingPayload['booking_date'] ?? '',
                        'booking_start_date' => $createBookingPayload['booking_start_date'] ?? '',
                        'customer_details' => json_encode($referenceDetails['booking_details']['userDetails']) ?? '',
                        'hotel_details' => json_encode($referenceDetails['booking_details']['hotelDetails']) ?? '',
                        'search_details' => json_encode($referenceDetails['booking_details']['searchDetails']) ?? '',
                        'processed_price' => json_encode($referenceDetails['booking_details']['processedPrice']) ?? '',
                        'billing_details' => json_encode($referenceDetails['booking_details']['billingDetails']) ?? '',
                        'booking_details' => json_encode($referenceDetails['booking_details']) ?? '',
                    ]
                );

            }

                if ($createBookingPayload['booking_type'] == EnumsServiceType::Flight->value) {
                    EditFlightDetail::updateOrCreate(
                        ['booking_id' => $updateBookingResponse['data']->id ?? ''],
                        [
                            'first_name' => $createBookingPayload['first_name'] ?? '',
                            'last_name' => $createBookingPayload['last_name'] ?? '',
                            'email' => $createBookingPayload['email'] ?? '',
                            'booking_date' => $createBookingPayload['booking_date'] ?? '',
                            'booking_start_date' => $createBookingPayload['booking_start_date'] ?? '',
                            'customer_details' => json_encode($referenceDetails['booking_details']['userDetails']) ?? '',
                            'flight_details' => json_encode($referenceDetails['booking_details']['flightDetails']['airlines'][0]['flights']) ?? '',
                            'passenger_details' => json_encode($referenceDetails['booking_details']['passengerDetails']) ?? '',
                            'payment_details' => json_encode($referenceDetails['booking_details']['paymentDetails']) ?? '',
                            'processed_price' => json_encode($referenceDetails['booking_details']['processedPrice']) ?? '',
                            'billing_details' => json_encode($referenceDetails['booking_details']['billingDetails']) ?? '',
                            'booking_details' => json_encode($referenceDetails['booking_details']) ?? '',
                        ]
                    );
                }



            if ($updateBookingResponse['status'] == 'true') {
                $bookingDetails = $updateBookingResponse['data'];
                if (is_array($bookingDetails) == false) {
                    $bookingDetails = json_decode($bookingDetails, true);
                }
            }
            $result['status'] = true;
            $result['type'] = 'update';
            $result['data'] = $bookingDetails;
            $result['message'] = '';
        } else {
            $createBookingPayload['booking_ref'] = $this->getBookingRefId();

            $createBookingResponse = Bookings::createBooking($createBookingPayload);



            if($createBookingPayload['booking_type']== EnumsServiceType::Hotel->value)
            {
                EditHotelDetail::create([
                    'booking_id' => $createBookingResponse['data']->id ?? '',
                    'first_name' => $createBookingPayload['first_name'] ?? '',
                    'last_name' => $createBookingPayload['last_name'] ?? '',
                    'email' => $createBookingPayload['email'] ?? '',
                    'booking_date' => $createBookingPayload['booking_date'] ?? '',
                    'booking_start_date' => $createBookingPayload['booking_start_date'] ?? '',
                    'customer_details' => json_encode($referenceDetails['booking_details']['userDetails']) ?? '',
                    'hotel_details' => json_encode($referenceDetails['booking_details']['hotelDetails']) ?? '',
                    'search_details' => json_encode($referenceDetails['booking_details']['searchDetails']) ?? '',
                    'processed_price' => json_encode($referenceDetails['booking_details']['processedPrice']) ?? '',
                    'billing_details' => json_encode($referenceDetails['booking_details']['billingDetails']) ?? '',
                ]);
            }
            if($createBookingPayload['booking_type']== EnumsServiceType::Flight->value)
            {
                EditFlightDetail::create([
                    'booking_id' => $createBookingResponse['data']->id ?? '',
                    'first_name' => $createBookingPayload['first_name'] ?? '',
                    'last_name' => $createBookingPayload['last_name'] ?? '',
                    'email' => $createBookingPayload['email'] ?? '',
                    'booking_date' => $createBookingPayload['booking_date'] ?? '',
                    'booking_start_date' => $createBookingPayload['booking_start_date'] ?? '',
                    'customer_details' => json_encode($referenceDetails['booking_details']['userDetails']) ?? '',
                    'flight_details' => json_encode($referenceDetails['booking_details']['flightDetails']['airlines'][0]['flights']) ?? '',
                    'passenger_details' => json_encode($referenceDetails['booking_details']['passengerDetails']) ?? '',
                    'payment_details' => json_encode($referenceDetails['booking_details']['paymentDetails']) ?? '',
                    'processed_price' => json_encode($referenceDetails['booking_details']['processedPrice']) ?? '',
                    'billing_details' => json_encode($referenceDetails['booking_details']['billingDetails']) ?? '',
                ]);
            }




            if ($createBookingResponse['status'] == 'true') {
                $bookingDetails = $createBookingResponse['data'];
                if (is_array($bookingDetails) == false) {
                    $bookingDetails = json_decode($bookingDetails, true);
                }

                Setting::updateOrCreate(['id' => $rehlteRefNoDetails['id']], ['value' => $rehlteRefNoDetails['value']]);

                $result['status'] = true;
                $result['type'] = 'create';
                $result['data'] = $createBookingResponse['data'];
                $result['message'] = '';
                BookingCheckout::updateBookingByRefId($refId, ['booking_id' => $bookingDetails['id'], 'booking_ref_id' => $bookingDetails['booking_ref']]);
            }
        }
        $additionalDetails = [
            'booking_id' => $bookingDetails['id'],
            'booking_details' => json_encode($referenceDetails['booking_details'])
        ];


        BookingAdditionalDetail::createBookingAdditionalDetail($additionalDetails);




        $booking_id = $additionalDetails['booking_id'];


        $booking_details = json_decode($additionalDetails['booking_details'], true);



        if (isset($booking_details['userDetails']['userId'], $booking_details['paymentDetails']['coupon_details']['cashbackAmount']) && (isset($booking_details['searchDetails']['stay']['checkOut']) || isset($booking_details['searchDetails']['originDestinations']['0']['departureDate']))) {

            $userId = $booking_details['userDetails']['userId'] ?? null;


            $couponId = $booking_details['paymentDetails']['coupon_details']['id'] ?? null;
            $cashbackAmount = $booking_details['paymentDetails']['coupon_details']['cashbackAmount'] ?? 0;


            if(isset($booking_details['searchDetails']['stay']['checkOut']))
            {
                $checkOutDate = $booking_details['searchDetails']['stay']['checkOut'] ?? null;
            }
            if(isset($booking_details['searchDetails']['originDestinations']['0']['departureDate']))
            {
                $checkOutDate = $booking_details['searchDetails']['originDestinations']['0']['departureDate'] ?? null;
            }



            if ($cashbackAmount !== null) {
                Cashback::updateOrCreate(
                    [
                        'booking_id' => $booking_id,
                        'customer_id' => $userId,

                        'coupon_id' => $couponId,
                        'cashback_amount' => $cashbackAmount,
                        'expiry_date' => $checkOutDate,
                    ]
                );
            }
        } else {
            // Handle missing details, maybe log or throw an error

        }



        return $result;
    }

    public function saveTravelersForAmadeus($referenceDetails, $bookingDetails)
    {
        foreach ($referenceDetails['booking_details']['passengerDetails'] as $passenger) {
            $passenger['customer_id'] = '';
            $passenger['booking_id'] = $bookingDetails['id'];
            CustomerTraveller::create($passenger);
        }
    }

    public function saveTravelersForHotelBeds($referenceDetails, $bookingDetails)
    {
        $roomNameBoardDetails = [];
        foreach ($referenceDetails['booking_details']['hotelDetails']['hotelDetails']['rooms'] as $key => $room) {
            foreach ($room['rates'] as $rate) {
                $roomNameBoardDetails[$rate['rateKey']] = [
                    'roomName' => ucwords($room['name']),
                    'boardCode' => $rate['boardCode'],
                    'boardName' => $rate['boardName'],
                ];
            }
        }
        foreach ($referenceDetails['booking_details']['hotelDetails']['selectedRooms']['rooms'] as $key => $selectedRoom) {
            foreach ($selectedRoom['paxes'] as $pax) {
                $additionalDetails = [
                    'room' => ('Room ' . ($key + 1)),
                    'paxType' => (($pax['type'] == 'CH') ? 'Child' : 'Adult'),
                    'rateKey' => $selectedRoom['rateKey'],
                    'roomName' => '',
                    'boardCode' => '',
                    'boardName' => ''
                ];
                if (isset($roomNameBoardDetails[$selectedRoom['rateKey']])) {
                    $additionalDetails['roomName'] = $roomNameBoardDetails[$selectedRoom['rateKey']]['roomName'];
                    $additionalDetails['boardCode'] = $roomNameBoardDetails[$selectedRoom['rateKey']]['boardCode'];
                    $additionalDetails['boardName'] = $roomNameBoardDetails[$selectedRoom['rateKey']]['boardName'];
                }
                $data = [
                    'booking_id' => $bookingDetails['id'],
                    'first_name' => $pax['name'],
                    'last_name' => $pax['surname'],
                    'additional_details' => json_encode($additionalDetails)
                ];
                CustomerTraveller::create($data);
            }
        }
    }
    public function saveTravelersForGimmonix($referenceDetails, $bookingDetails)
    {
        $travelers = [];
        foreach ($referenceDetails['booking_details']['searchDetails']['occupancies'] as $key => $occupancy) {
            for ($i = 0; $i < $occupancy['adults']; $i++) {
                array_push($travelers, [
                    'room' => ('Room ' . ($key + 1)),
                    'paxType' => 'Adult',
                ]);
            }
            if (array_key_exists('children', $occupancy) && $occupancy['children'] > 0) {
                for ($i = 0; $i < $occupancy['children']; $i++) {
                    array_push($travelers, [
                        'room' => ('Room ' . ($key + 1)),
                        'paxType' => 'Child',
                    ]);
                }
            }
        }
        foreach ($travelers as $traveler) {
            $data = [
                'booking_id' => $bookingDetails['id'],
                'first_name' => $referenceDetails['booking_details']['userDetails']['firstName'],
                'last_name' => $referenceDetails['booking_details']['userDetails']['lastName'],
                'additional_details' => json_encode([
                    'room' => $traveler['room'],
                    'paxType' => $traveler['paxType'],
                    'rateKey' => $referenceDetails['booking_details']['hotelDetails']['selectedRooms']['Id'],
                    'roomName' => $referenceDetails['booking_details']['hotelDetails']['selectedRooms']['RoomName'],
                    'boardCode' => '',
                    'boardName' => $referenceDetails['booking_details']['hotelDetails']['selectedRooms']['RoomBasis']
                ])
            ];
            CustomerTraveller::create($data);
        }
    }

    public function saveBookingTravelers($referenceDetails, $bookingDetails)
    {
        switch ($referenceDetails['booking_type']) {
            case EnumsServiceType::Flight->value:
                $this->saveTravelersForAmadeus($referenceDetails, $bookingDetails);
                break;
            case EnumsServiceType::Hotel->value:
                if ($referenceDetails['service_provider'] == 'HOTEL_BEDS') {
                    $this->saveTravelersForHotelBeds($referenceDetails, $bookingDetails);
                } elseif ($referenceDetails['service_provider'] == 'GIMMONIX') {
                    $this->saveTravelersForGimmonix($referenceDetails, $bookingDetails);
                }
                break;
        }
    }

    public function processReturnResult($result)
    {
        if ($result['booking_via'] == 'mobile') {
            if ($result['status'] == BookingStatus::Confirmed->value) {
                $result = [
                    'success' => 'true',
                    'data' => ['booking_ref' => $result['booking_ref_id'], 'email' => $result['email']],
                    'message' => 'Booking have created successfully'
                ];
            } else {
                $result = [
                    'success' => 'false',
                    'data' => [
                        'status' => BookingStatus::Failed->value,
                        'sub_status' => $result['sub_status'],
                    ],
                    'message' => $result['message']
                ];
            }
            return $result;
        } else {
            $redirectUrl = env('WEBSITE_URL') . $result['language'];
            if ($result['status'] == BookingStatus::Confirmed->value) {
                $bookingPage = 'hotel-booking';
                switch ($result['booking_type']) {
                    case EnumsServiceType::Flight->value:
                        $bookingPage = 'flight-booking';
                        break;
                    case EnumsServiceType::Hotel->value:
                        $bookingPage = 'hotel-booking';
                        break;
                }
                $redirectUrl .= '/booking/' . $bookingPage . '?booking_reference=' . $result['booking_ref_id'] . '&email=' . $result['email'] . '&reference=create';
            } else {
                $redirectUrl .= '/payment?ref_id=' . $result['ref_id'] . '&payment_brand=' . $result['payment_brand'] . '&status=failed&sub_status=' . $result['sub_status'] . '&message=' . $result['message'];
            }
            return $redirectUrl;
        }
    }

    public function getFlightOfferPricingPayload($currency, $locale, $flightInfo, $serviceProvider)
    {
        $payload = [
            "agencyId" => "0",
            "type" => "flight-offers-pricing",
            "currencyCode" => $currency,
            "supplier" => $serviceProvider,
            "languageCode" => $locale,
            "flightOffers" => $flightInfo
        ];
        return $payload;
    }
}
