<?php

namespace App\Traits;

use DateTime;
use App\Enums\Locale;
use App\Traits\Uuids;
use App\Models\Coupon;
use App\Enums\Currency;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Markups;
use App\Models\Setting;
use App\Models\Suppliers;
use App\Traits\WithinEarth;
use App\Models\BlockedFlight;
use App\Models\DefaultMarkup;
use App\Traits\CommonService;
use App\Traits\FlightService;
use App\Models\ServiceProviderLog;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\URL;
use App\Services\BrandedFareService;
use App\Services\CurlRequestService;
use App\Traits\LoyaltyPointsService;
use Illuminate\Support\Facades\Storage;

trait AmadeusService
{
    use CommonService, FlightService, LoyaltyPointsService, WithinEarth, Uuids;
    public function checkTokenHealth()
    {

        if ($this->amadeusAPISecret == '') {
            $this->generateToken();
        } else {
            $this->refreshToken();
        }
    }

    public function refreshToken()
    {
        $body = http_build_query([
            'client_id' => $this->amadeusAPIClientID,
            'client_secret' => $this->amadeusAPIClientSecret,
            'grant_type' => $this->amadeusAPIGrantType,
        ]);
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $this->amadeusAPIEndPoint . "/v1/security/oauth2/token",
            'body' => $body,
            'headers' => [
                "Content-Type: application/x-www-form-urlencoded",
            ],
        ]);

        $tokenInformation = json_decode($response, true);

        if (array_key_exists('access_token', $tokenInformation)) {
            $accessToken = $tokenInformation["access_token"];
            $this->amadeusAPISecret = $accessToken;
        } else {
            $this->generateToken();
        }
    }

    public function generateToken()
    {
        $body = http_build_query([
            'client_id' => $this->amadeusAPIClientID,
            'client_secret' => $this->amadeusAPIClientSecret,
            'grant_type' => $this->amadeusAPIGrantType,
        ]);
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $this->amadeusAPIEndPoint . "/v1/security/oauth2/token",
            'body' => $body,
            'headers' => [
                "Content-Type: application/x-www-form-urlencoded",
            ],
        ]);
        $tokenResponse = json_decode($response, true);
        $accessToken = $tokenResponse["access_token"];
        $this->amadeusAPISecret = $accessToken;
        Setting::updateOrCreate(['config_key' => "amadeus|api|secret"], ["value" => $accessToken]);
    }




    /**
     * create flight offer search api with origin and processed response
     */
    public function flightOfferSearch($requestData, $toCurrencyCode = Currency::USD->value, $customerId = '')
    {
        $arrData = json_decode($requestData);

        $this->checkTokenHealth();


        $appURL = $this->amadeusAPIEndPoint . "/v2/shopping/flight-offers";

        $amaClientRef = $this->getUuid();

        $timeLog = ['before_curl' => date('Y-m-d H:i:s')];

        // $response = CurlRequestService::executeAmadeusCurl($method = 'POST', $appURL, $amaClientRef, $amadeusAPISecret, $requestData);

        $response = CurlRequestService::sendRequest([
            'method'  => 'POST',
            'url'     => $appURL,
            'headers' => [
                'Content-Type: application/json',
                "Authorization: Bearer $this->amadeusAPISecret",
                "ama-client-ref: $amaClientRef",
            ],
            'body'    => $requestData,

        ]);

        $timeLog['after_curl'] = date('Y-m-d H:i:s');

        $flightOffersData = json_decode($response, true);
        //   $flightOffersData = $response['response'];

        $logId = '';
        if (isset($flightOffersData['errors'])) {
            $flightOffersData['status'] = false;
            $success = $flightOffersData['errors'][0]['title'];
            unset($flightOffersData['errors']);
            $flightOffersData['data'] = [];
            $flightOffersData['success'] = $success;
            return $flightOffersData;
        } elseif (empty($flightOffersData['data'])) {
            $flightOffersData['airportList'] = [];
            $uniqueLocationArr = [];

            // Step 1: Collect all unique location codes
            foreach ($arrData->originDestinations as $airportInfo) {
                $uniqueLocationArr[] = $airportInfo->originLocationCode;
                $uniqueLocationArr[] = $airportInfo->destinationLocationCode;
            }
            $uniqueLocationArr = array_unique($uniqueLocationArr);

            // Step 2: Fetch all required data in bulk

            // Fetch airports with translations
            $airports = DB::table('airports')
                ->join('airport_i18ns', 'airport_i18ns.airport_id', 'airports.id')
                ->whereIn('airports.iata_code', $uniqueLocationArr)
                ->where('airport_i18ns.language_code', strtolower($arrData->languageCode))
                ->select('airports.iata_code as code', 'airport_i18ns.airport_name')
                ->get()
                ->keyBy('code');

            // Fetch cities with translations (assuming iso_code matches airport codes)
            $cities = DB::table('cities')
                ->join('city_i18ns', 'city_i18ns.city_id', 'cities.id')
                ->whereIn('cities.iso_code', $uniqueLocationArr)
                ->where('city_i18ns.language_code', strtolower($arrData->languageCode))
                ->select('cities.iso_code as code', 'city_i18ns.city_name')
                ->get()
                ->keyBy('code');

            // Fetch countries with translations
            // First get country codes from airports
            $countryCodes = Airport::whereIn('iata_code', $uniqueLocationArr)->pluck('country_code')->unique();

            $countryNames = DB::table('countries')
                ->join('country_i18ns', 'country_i18ns.country_id', 'countries.id')
                ->whereIn('countries.iso_code', $countryCodes)
                ->where('country_i18ns.language_code', strtolower($arrData->languageCode))
                ->select('countries.iso_code as code', 'country_i18ns.country_name')
                ->get()
                ->keyBy('code');

            // Also map airports to country codes to access them later
            $airportCountryMap = Airport::whereIn('iata_code', $uniqueLocationArr)
                ->pluck('country_code', 'iata_code')
                ->all();

            // Step 3: Build airportList without additional queries
            $addedLocations = [];

            foreach ($arrData->originDestinations as $airportInfo) {
                foreach (['originLocationCode', 'destinationLocationCode'] as $key) {
                    $code = $airportInfo->{$key};

                    if (isset($addedLocations[$code])) {
                        continue;
                    }

                    $airportName = $airports->has($code) ? $airports[$code]->airport_name : "";
                    $cityName = $cities->has($code) ? $cities[$code]->city_name : "";
                    $countryCode = $airportCountryMap[$code] ?? null;
                    $countryName = ($countryCode && $countryNames->has($countryCode)) ? $countryNames[$countryCode]->country_name : "";

                    $flightOffersData['airportList'][] = [
                        'code' => $code,
                        'airport_name' => $airportName,
                        'city_name' => $cityName,
                        'country_name' => $countryName,
                    ];

                    $addedLocations[$code] = true;
                }
            }

            $flightOffersData['status'] = false;
            $flightOffersData['success'] = "No direct flights found";
            return $flightOffersData;
        } else {
            $logId = $this->createSupplierLog('Flight', 'Amadeus', 'flight-search', $requestData, $response, $customerId, '', $amaClientRef);
        }
        $timeLog['after_airport_dbquery'] = date('Y-m-d H:i:s');

        $collection = collect($flightOffersData['data']);

        $newFlightOffersData = [];

        //add new object(origin and processed) for same response
        $newArr = $collection->map(function ($item) use ($arrData, $toCurrencyCode) {
            $processedItem = $item;

            foreach ($processedItem['itineraries'] as $iKey => $itineraries) {
                $processedItem['itineraries'][$iKey]['durationText'] = getHourMinute($itineraries['duration']);
            }

            $processedItem['supplier'] = "AMADEUS";
            //get converted total price using traits
            $convertedTotalPrice = convertCurrencyExchangeRate($processedItem['price']['total'], Currency::SAR->value, $toCurrencyCode, []);

            //set requested currency into currency key
            $processedItem['price']['currency'] = ($arrData->languageCode == strtoupper(Locale::Arabic->value)) ? strval($convertedTotalPrice['data']['symbol']) : strval($convertedTotalPrice['data']['toCurrencyCode']);
            $processedItem['price']['total'] = strval($convertedTotalPrice['data']['convertedRate']);

            //get converted base price using traits
            $convertedBasePrice = convertCurrencyExchangeRate($processedItem['price']['base'], Currency::SAR->value, $toCurrencyCode, []);
            $processedItem['price']['base'] = strval($convertedBasePrice['data']['convertedRate']);
            //get converted grandTotal using traits
            $convertedGrandTotal = convertCurrencyExchangeRate($processedItem['price']['grandTotal'], Currency::SAR->value, $toCurrencyCode, []);
            $processedItem['price']['grandTotal'] = strval($convertedGrandTotal['data']['convertedRate']);

            $payload = [
                'service' => 'Airline Ticket',
                'amount' => $processedItem['price']['grandTotal'],
                'fromCurrency' => Currency::SAR->value,
                'toCurrency' => $toCurrencyCode
            ];
            $loyaltyPoints = $this->getLoyaltyPoints($payload);

            $processedItem['price']['rehlte_points'] = $loyaltyPoints;

            $brandedFareExists = $this->checkBrandedFareExists($processedItem);
            $processedItem['brandedFareExists'] = $brandedFareExists;
            $baggageExists = $this->checkBaggageExists($processedItem);
            $processedItem['baggageExists'] = $baggageExists;
            $processedItem['brandedFare'] = [
                'brandedFares' => [$processedItem['id'] => $this->createBrandedFare($processedItem, true)],
                'data' => [$processedItem]
            ];
            $brandedFareBySegment[] = BrandedFareService::createBrandedFare($processedItem);
            $processedItem['brandedFareBySegment'] =  $brandedFareBySegment;

            return [
                'original' => $item,
                'processed' => $processedItem
            ];
        });

        $timeLog['after_price_conversion'] = date('Y-m-d H:i:s');

        $newFlightOffersData['data']['data'] = $newArr->values()->all();
        $flightOffersData['data'] = $newFlightOffersData['data']['data'];


        //dynamically create array for airportList
        $citiesDetails = [];
        $flightOffersData['airportList'] = [];

        $locationCodes = array_keys($flightOffersData['dictionaries']['locations']);
        $languageCode = strtolower($arrData->languageCode);

        // Fetch all airports at once with translations
        $airports = DB::table('airports')
            ->join('airport_i18ns', 'airport_i18ns.airport_id', 'airports.id')
            ->whereIn('airports.iata_code', $locationCodes)
            ->where('airport_i18ns.language_code', $languageCode)
            ->select('airports.iata_code as code', 'airport_i18ns.airport_name')
            ->get()
            ->keyBy('code');

        // Fetch all cities at once with translations
        $cities = DB::table('cities')
            ->join('city_i18ns', 'city_i18ns.city_id', 'cities.id')
            ->whereIn('cities.iso_code', $locationCodes)
            ->where('city_i18ns.language_code', $languageCode)
            ->select('cities.iso_code as code', 'city_i18ns.city_name')
            ->get()
            ->keyBy('code');

        // Fetch all country codes from airports
        $airportCountryMap = Airport::whereIn('iata_code', $locationCodes)
            ->pluck('country_code', 'iata_code')
            ->all();

        //Fetch all countries at once with translations
        $countryCodes = array_unique(array_values($airportCountryMap));

        $countryNames = DB::table('countries')
            ->join('country_i18ns', 'country_i18ns.country_id', 'countries.id')
            ->whereIn('countries.iso_code', $countryCodes)
            ->where('country_i18ns.language_code', $languageCode)
            ->select('countries.iso_code as code', 'country_i18ns.country_name')
            ->get()
            ->keyBy('code');

        //Loop over locations
        foreach ($flightOffersData['dictionaries']['locations'] as $key => $dictionaries) {
            $airportName = $airports->has($key) ? $airports[$key]->airport_name : "";
            $cityName = $cities->has($key) ? $cities[$key]->city_name : "";
            $countryCode = $airportCountryMap[$key] ?? null;
            $countryName = ($countryCode && $countryNames->has($countryCode)) ? $countryNames[$countryCode]->country_name : "";

            $tempArr = [
                'code' => $key,
                'airport_name' => $airportName,
                'city_name' => $cityName,
                'country_name' => $countryName,
            ];

            $citiesDetails[$key] = [
                'city_name' => $cityName,
                'country_name' => $countryName,
            ];

            $flightOffersData['airportList'][] = $tempArr;
        }


        $timeLog['after_airport_list'] = date('Y-m-d H:i:s');

        //dynamically create array for aircraftList
        $flightOffersData['aircraftList'] = [];
        foreach ($flightOffersData['dictionaries']['aircraft'] as $aircraftKey => $aircraft) {
            $tempAircraftArr = [
                'code' => $aircraftKey,
                'name' => $aircraft,

            ];
            array_push($flightOffersData['aircraftList'], $tempAircraftArr);
        }

        $couponList = Coupon::whereStatus('active')
            ->where('product', 'flight')
            ->get()
            ->map(function ($coupon) {
                if ($coupon->banner) {
                    $coupon->banner = url('storage/' . $coupon->banner);
                }
                if ($coupon->thumbnail) {
                    $coupon->thumbnail = url('storage/' . $coupon->thumbnail);
                }
                return $coupon;
            });
        $coupons = [];
        $couponAirlines = [];
        $couponAirlineDetails = [];
        if ($couponList) {
            $coupons = $couponList->toArray();
            if (count($coupons) > 0) {
                foreach ($coupons as &$coupon) {
                    $coupon['is_new'] = true;
                    $coupon['airline_logo'] = '';
                    if ($coupon['airline_name'] != '') {
                        array_push($couponAirlines, $coupon['airline_name']);
                    }
                }
            }
        }

        //dynamically create array for airlineList
        $flightOffersData['airlineList'] = [];

        $carrierCodes = array_keys($flightOffersData['dictionaries']['carriers']);
        $languageCode = strtolower($arrData->languageCode);

        //  Fetch all blocked flights once
        $blockedFlights = BlockedFlight::whereIn('airline_name', $flightOffersData['dictionaries']['carriers'])->get()->keyBy('airline_name');

        //  Fetch all airlines once
        $airlines = Airline::whereIn('airline_code', $carrierCodes)->get()->keyBy('airline_code');

        //Fetch all airline translations
        $airlineTranslations = DB::table('airlines')
            ->join('airline_i18ns', 'airline_i18ns.airline_id', '=', 'airlines.id')
            ->whereIn('airlines.airline_code', $carrierCodes)
            ->where('airline_i18ns.language_code', $languageCode)
            ->select('airlines.airline_code as code', 'airline_i18ns.airline_name')
            ->get()
            ->groupBy('code');

        // Process each carrier without additional queries
        foreach ($flightOffersData['dictionaries']['carriers'] as $codeKey => $carriers) {
            $blockedFlight = $blockedFlights->has($carriers) ? $blockedFlights[$carriers] : null;

            if ($blockedFlight) {
                $blockedAirlineName = $blockedFlight->airline_name;
                $blockedCode = $blockedFlight->airline_id ? Airline::where('id', $blockedFlight->airline_id)->value('airline_code') : null;

                $name = "";

                if ($blockedCode && $airlineTranslations->has($codeKey)) {
                    foreach ($airlineTranslations[$codeKey] as $translation) {
                        if ($translation->airline_name !== $blockedAirlineName && $codeKey === $blockedCode) {
                            $name = $translation->airline_name;
                            break;
                        }
                    }
                }
            } else {
                $name = "";

                if ($airlineTranslations->has($codeKey)) {
                    $name = $airlineTranslations[$codeKey]->first()->airline_name;
                }
            }

            $airline = $airlines->has($codeKey) ? $airlines[$codeKey] : null;

            if ($airline && !empty($airline->airline_logo)) {
                $LogoURL = $airline->airline_logo;
                $logoName = basename($LogoURL);
                $path = 'public/airlineLogo/' . $logoName;
                $airlineLogo = URL::to('/') . Storage::url($path);
            } else {
                $airlineLogo = URL::to('/') . '/assets/images/airlineLogo/' . $codeKey . '.png';
            }

            $tempCodeArr = [
                'code' => $codeKey,
                'name' => $name,
                'logo' => $airlineLogo,
            ];

            $flightOffersData['airlineList'][] = $tempCodeArr;

            if (in_array($codeKey, $couponAirlines)) {
                $couponAirlineDetails[$codeKey] = $tempCodeArr;
                unset($couponAirlines[$codeKey]);
            }
        }


        if (count($couponAirlines) > 0) {
            $languageCode = strtolower($arrData->languageCode);

            //Fetch all airline translations
            $airlineTranslations = DB::table('airlines')
                ->join('airline_i18ns', 'airline_i18ns.airline_id', 'airlines.id')
                ->whereIn('airlines.airline_code', $couponAirlines)
                ->where('airline_i18ns.language_code', $languageCode)
                ->select('airlines.airline_code as code', 'airline_i18ns.airline_name')
                ->get()
                ->keyBy('code');

            //Fetch all airlines
            $airlines = Airline::whereIn('airline_code', $couponAirlines)
                ->where('airline_logo', '!=', '')
                ->get()
                ->keyBy('airline_code');

            foreach ($couponAirlines as $couponAirline) {
                $name = "";

                if ($airlineTranslations->has($couponAirline)) {
                    $name = $airlineTranslations[$couponAirline]->airline_name;
                }

                if ($airlines->has($couponAirline)) {
                    $LogoURL = $airlines[$couponAirline]->airline_logo;
                    $logoName = basename($LogoURL);
                    $path = 'public/airlineLogo/' . $logoName;
                    $airlineLogo = URL::to('/') . Storage::url($path);
                } else {
                    $airlineLogo = URL::to('/') . '/assets/images/airlineLogo/' . $couponAirline . '.png';
                }

                $tempCodeArr = [
                    'code' => $couponAirline,
                    'name' => $name,
                    'logo' => $airlineLogo
                ];

                $couponAirlineDetails[$couponAirline] = $tempCodeArr;
            }
        }

        //echo "<pre>"; print_r($couponAirlines); print_r($couponAirlineDetails); print_r($flightOffersData['airlineList']); print_r($coupons); die;
        if (count($coupons) > 0) {
            foreach ($coupons as &$coupon) {
                if ($coupon['airline_name'] != '') {
                    $airline = $couponAirlineDetails[$coupon['airline_name']];
                    $coupon['airline_name'] = $airline['name'];
                    $coupon['airline_logo'] = $airline['logo'];
                }
            }
        }
        $flightOffersData['coupons'] = $coupons;

        $timeLog['after_airline_list'] = date('Y-m-d H:i:s');

        //dynamically create array for layovers
        $flightOffersData['layover'] = [];

        $processedSegments = []; // Initialize an array to keep track of processed segments
        foreach ($flightOffersData['data'] as $key => $data) {
            foreach ($data['processed']['itineraries'] as $itineraries) {
                $segments = $itineraries['segments'];
                for ($i = 0; $i < count($segments) - 1; $i++) {
                    $currentFlight = $segments[$i];
                    $nextFlight = $segments[$i + 1];

                    // Check if the current segment has already been processed
                    $currentSegmentId = $currentFlight['id'];
                    if (!in_array($currentSegmentId, $processedSegments)) {
                        $arrivalTime = $currentFlight['arrival']['at'];
                        $departureTime = $nextFlight['departure']['at'];
                        $difference = getTimeDifference($departureTime, $arrivalTime);

                        $layoverArray = [
                            'fromSegmentId' => $currentFlight['id'],
                            'toSegmentId' => $nextFlight['id'],
                            'layoverDurationText' => $difference['hours'] . 'h ' . $difference['minutes'] . 'm',
                            'layoverDuration' => (($difference['hours'] < 9) ? str_pad($difference['hours'], 2, '0', STR_PAD_LEFT) : $difference['hours']) . ':' . $difference['minutes']
                        ];
                        array_push($flightOffersData['layover'], $layoverArray);

                        // Add the current segment to the processedSegments array to avoid duplicates
                        $processedSegments[] = $currentSegmentId;
                    }
                }
            }
        }
        foreach ($flightOffersData['data'] as $key => &$data) {
            unset($data['original']);
        }

        $flightOffersData['status'] = true;

        $flightOffersData['log_id'] = $logId;

        $flightOffersData['ama_client_ref'] = $amaClientRef;

        $flightOffersData['service_provider'] = 'AMADEUS';

        $timeLog['end_of_search'] = date('Y-m-d H:i:s');

        $flightOffersData['time_log'] = $timeLog;

        return $flightOffersData;
    }

    /**
     * Search flight offer api with origin and processed response
     */
    public function flightOfferSearchGet($requestData)
    {
        $this->checkTokenHealth();

        // Build query string for GET params
        $queryString = http_build_query($requestData);

        $appURL = $this->amadeusAPIEndPoint . "/v2/shopping/flight-offers?" . $queryString;

        $response = CurlRequestService::sendRequest([
            'method' => 'GET',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/x-www-form-urlencoded',
                "Authorization: Bearer $this->amadeusAPISecret",
            ],

        ]);

        $flightOffersData = json_decode($response, true);
        $flightOffersData['from_name'] = DB::table('geo_airport_lists')->where('airport_code', $requestData['originLocationCode'])->value('airport_name');
        $flightOffersData['to_name'] = DB::table('geo_airport_lists')->where('airport_code', $requestData['destinationLocationCode'])->value('airport_name');
        $flightOffersData['departure_date'] = $requestData['departureDate'];
        $flightOffersData['flight_count'] = count($flightOffersData['data'] ?? []);

        return $flightOffersData;
    }


    /**
     * fetch flight offers price data
     */
    public function flightOfferPriceGet($requestData)
    {
        $requestTemp = [
            'data' => $requestData
        ];

        $postfields = json_encode($requestTemp, JSON_PRETTY_PRINT);
        // $arrData = json_decode($requestData);

        $this->checkTokenHealth();

        $appURL = $this->amadeusAPIEndPoint . "/v1/shopping/flight-offers/pricing?include=detailed-fare-rules%2Cbags";
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/json',
                "Authorization: Bearer $this->amadeusAPISecret"
            ],
            'body' =>  $postfields,
        ]);


        $flightOffersData = json_decode($response, true);
        if (isset($flightOffersData['errors'])) {
            return $flightOffersData;
        }

        // Add a new key 'originalFlightOffers' with a copy of 'flightOffers' into response

        $flightOffersData['data']['originalFlightOffers'] = array_map(function ($flightOffer) {
            return $flightOffer;
        }, $flightOffersData['data']['flightOffers']);



        // $flightOffersData['data']['flightOffers'] = array_map(function($flightPrice) use($arrData){

        //     // Replace values in the price array using the replacePriceValues function
        //     $flightPrice['price'] = replacePriceValues($flightPrice['price'],$arrData);

        //     return $flightPrice;
        // },$flightOffersData['data']['flightOffers']);



        // Use array_map to transform the data
        /*$flightOffersData['data']['flightOffers'] = array_map(function ($flightOffer) use($requestData){

            $flightOffer = $this->getPricesWithMarkup($flightOffer,$requestData);

            return $flightOffer;
        }, $flightOffersData['data']['flightOffers']); */

        $flightOffersData['processedPrice'] = [
            'currency' => "SAR",
            'serviceFee' => number_format("0", '2', '.', ''),
            'vat' => number_format("0", '2', '.', ''),
            'totalPrice' => number_format("0", '2', '.', ''),
            'grandTotal' => number_format("0", '2', '.', ''),
            'markupType' => "",
            'markupValue' => "",
            'totalTaxAmount' => ""

        ];
        //dynamically create array for airportList
        $flightOffersData['airportList'] = [];
        foreach ($flightOffersData['dictionaries']['locations'] as $key => $dictionaries) {


            $airport_name = DB::table('airports')->join('airport_i18ns', 'airport_i18ns.airport_id', 'airports.id')->select('airport_i18ns.airport_name')->where('airports.iata_code', $key)->where('airport_i18ns.language_code', strtolower($requestData['languageCode']))->first();
            $city_name_en = DB::table('cities')->join('city_i18ns', 'city_i18ns.city_id', 'cities.id')->select('city_i18ns.city_name')->where('cities.iso_code', $key)->where('city_i18ns.language_code', strtolower($requestData['languageCode']))->first();
            $country_name_en = DB::table('countries')->join('country_i18ns', 'country_i18ns.country_id', 'countries.id')->select('country_i18ns.country_name')->where('countries.iso_code', $dictionaries['countryCode'])->where('country_i18ns.language_code', strtolower($requestData['languageCode']))->first();
            $country_code = Airport::where('iata_code', $key)->value('country_code');
            $country_name_en = DB::table('countries')
                ->join('country_i18ns', 'country_i18ns.country_id', 'countries.id')
                ->where('country_i18ns.language_code', strtolower($requestData['languageCode']))
                ->where('iso_code', $country_code)
                ->first();

            $tempArr = [
                'code' => $key,
                'airport_name' => ($airport_name != "") ? $airport_name->airport_name : "",
                'city_name' => ($city_name_en != "") ? $city_name_en->city_name : "",
                'country_name' => ($country_name_en != "") ? $country_name_en->country_name : "",

            ];
            array_push($flightOffersData['airportList'], $tempArr);
        }


        // Dynamically create array for airlineList
        $flightOffersData['airlineList'] = [];
        foreach ($flightOffersData['data']['flightOffers'] as $flightOffer) {
            foreach ($flightOffer['itineraries'] as $itinerary) {
                foreach ($itinerary['segments'] as $segment) {
                    $carrierCode = $segment['carrierCode'];
                    $airlineName_en = DB::table('airlines')
                        ->join('airline_i18ns', 'airline_i18ns.airline_id', 'airlines.id')
                        ->select('airline_i18ns.airline_name')
                        ->where('airlines.airline_code', $carrierCode)
                        ->where('airline_i18ns.language_code', strtolower($requestData['languageCode']))
                        ->first();

                    $isExistAirlineCode = Airline::where('airline_code', $carrierCode)
                        ->where('airline_logo', '!=', '')
                        ->first();

                    if (!empty(json_decode($isExistAirlineCode, true))) {
                        $LogoURL = $isExistAirlineCode->airline_logo;
                        $logoName = basename($LogoURL);
                        $path = 'public/airlineLogo/' . $logoName; // The path to your image file in the storage
                        $airlineLogo = URL::to('/') . Storage::url($path);
                    } else {
                        $airlineLogo = URL::to('/') . '/assets/images/airlineLogo/' . $carrierCode . '.png';
                    }

                    $tempCodeArr = [
                        'code' => $carrierCode,
                        'name' => ($airlineName_en != "") ? $airlineName_en->airline_name : "",
                        'logo' => $airlineLogo
                    ];

                    // Check if the carrierCode is not already in the array
                    if (!in_array($tempCodeArr['code'], array_column($flightOffersData['airlineList'], 'code'))) {
                        $flightOffersData['airlineList'][] = $tempCodeArr;
                    }
                }
            }
        }

        // Check if flightOffers key exists and it is an array
        if (isset($flightOffersData['data']['flightOffers']) && is_array($flightOffersData['data']['flightOffers'])) {
            //dynamically create array for layovers
            $flightOffersData['layover'] = [];

            $processedSegments = []; // Initialize an array to keep track of processed segments
            // Iterate through each flight offer
            foreach ($flightOffersData['data']['flightOffers'] as $flightOffer) {
                // Check if itineraries key exists and it is an array
                if (isset($flightOffer['itineraries']) && is_array($flightOffer['itineraries'])) {
                    // Iterate through each itinerary
                    foreach ($flightOffer['itineraries'] as $itineraries) {
                        $segments = $itineraries['segments'];
                        for ($i = 0; $i < count($segments) - 1; $i++) {
                            $currentFlight = $segments[$i];
                            $nextFlight = $segments[$i + 1];

                            // Check if the current segment has already been processed
                            $currentSegmentId = $currentFlight['id'];
                            if (!in_array($currentSegmentId, $processedSegments)) {
                                $arrivalTime = $currentFlight['arrival']['at'];
                                $departureTime = $nextFlight['departure']['at'];
                                $difference = getTimeDifference($departureTime, $arrivalTime);

                                $layoverArray = [
                                    'fromSegmentId' => $currentFlight['id'],
                                    'toSegmentId' => $nextFlight['id'],
                                    'layoverDurationText' => $difference['hours'] . 'h ' . $difference['minutes'] . 'm',
                                    'layoverDuration' => (($difference['hours'] < 9) ? str_pad($difference['hours'], 2, '0', STR_PAD_LEFT) : $difference['hours']) . ':' . $difference['minutes']
                                ];
                                array_push($flightOffersData['layover'], $layoverArray);

                                // Add the current segment to the processedSegments array to avoid duplicates
                                $processedSegments[] = $currentSegmentId;
                            }
                        }
                    }
                }
            }
        }
        return $flightOffersData;
    }

    public function flightOfferPriceGetWithMarkup($requestData, $parentLogId = '', $customerId = '', $amaClientRef = '')
    {
        $requestTemp = ['data' => $requestData];
        $postfields = json_encode($requestTemp, JSON_PRETTY_PRINT);

        $this->checkTokenHealth();

        $appURL = $this->amadeusAPIEndPoint . "/v1/shopping/flight-offers/pricing?include=detailed-fare-rules%2Cbags";
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/json',
                "Authorization: Bearer $this->amadeusAPISecret",
                "ama-client-ref: $amaClientRef",
            ],
            'body' => $postfields,
        ]);

        $flightOffersData = json_decode($response, true);
        if (isset($flightOffersData['errors'])) {
            return $flightOffersData;
        } elseif ($parentLogId != '') {
            $this->createSupplierLog('Flight', 'Amadeus', 'flight-offer-price', $postfields, $response, $customerId, $parentLogId, $amaClientRef);
        }

        // Preserve original
        $flightOffersData['data']['originalFlightOffers'] = $flightOffersData['data']['flightOffers'];

        /*
         * Bulk fetch Airports / Cities / Countries
         */
        $locationKeys = array_keys($flightOffersData['dictionaries']['locations']);
        $language = strtolower($requestData['languageCode']);

        // Airports
        $airports = DB::table('airports')
            ->join('airport_i18ns', 'airport_i18ns.airport_id', '=', 'airports.id')
            ->whereIn('airports.iata_code', $locationKeys)
            ->where('airport_i18ns.language_code', $language)
            ->pluck('airport_i18ns.airport_name', 'airports.iata_code');

        // Cities
        $cities = DB::table('cities')
            ->join('city_i18ns', 'city_i18ns.city_id', '=', 'cities.id')
            ->whereIn('cities.iso_code', $locationKeys)
            ->where('city_i18ns.language_code', $language)
            ->pluck('city_i18ns.city_name', 'cities.iso_code');

        // Countries
        $countryCodes = collect($flightOffersData['dictionaries']['locations'])
            ->pluck('countryCode')
            ->unique()
            ->toArray();

        $countries = DB::table('countries')
            ->join('country_i18ns', 'country_i18ns.country_id', '=', 'countries.id')
            ->whereIn('countries.iso_code', $countryCodes)
            ->where('country_i18ns.language_code', $language)
            ->pluck('country_i18ns.country_name', 'countries.iso_code');

        // Airport country codes
        $airportCountryCodes = Airport::whereIn('iata_code', $locationKeys)
            ->pluck('country_code', 'iata_code');

        // Build airportList
        $flightOffersData['airportList'] = [];
        foreach ($flightOffersData['dictionaries']['locations'] as $key => $dictionaries) {
            $countryCode = $airportCountryCodes[$key] ?? null;
            $tempArr = [
                'code'         => $key,
                'airport_name' => $airports[$key] ?? "",
                'city_name'    => $cities[$key] ?? "",
                'country_name' => $countries[$countryCode] ?? "",
            ];
            $flightOffersData['airportList'][] = $tempArr;
        }

        // Domestic check
        $isDomestic = collect($flightOffersData['airportList'])
            ->every(fn($airport) => trim($airport['country_name']) === 'Saudi Arabia');

        // Apply Markup
        $flightOffersData['data']['flightOffers'] = array_map(function ($flightOffer) use ($requestData, $customerId, $isDomestic) {
            return $this->getPricesWithMarkup($flightOffer, $requestData, $customerId, $isDomestic);
        }, $flightOffersData['data']['flightOffers']);

        /**
         * Bulk fetch Airlines 
         */
        $carrierCodes = collect($flightOffersData['data']['flightOffers'])
            ->flatMap(
                fn($offer) =>
                collect($offer['itineraries'])->flatMap(
                    fn($itinerary) =>
                    collect($itinerary['segments'])->pluck('carrierCode')
                )
            )
            ->unique()
            ->toArray();

        $airlineNames = DB::table('airlines')
            ->join('airline_i18ns', 'airline_i18ns.airline_id', '=', 'airlines.id')
            ->whereIn('airlines.airline_code', $carrierCodes)
            ->where('airline_i18ns.language_code', $language)
            ->pluck('airline_i18ns.airline_name', 'airlines.airline_code');

        $airlineLogos = Airline::whereIn('airline_code', $carrierCodes)
            ->where('airline_logo', '!=', '')
            ->pluck('airline_logo', 'airline_code');

        $flightOffersData['airlineList'] = [];
        foreach ($carrierCodes as $carrierCode) {
            if (isset($airlineLogos[$carrierCode])) {
                $logoUrl = $airlineLogos[$carrierCode];
                $logoName = basename($logoUrl);
                $path = 'public/airlineLogo/' . $logoName;
                $airlineLogo = URL::to('/') . Storage::url($path);
            } else {
                $airlineLogo = URL::to('/') . '/assets/images/airlineLogo/' . $carrierCode . '.png';
            }

            $flightOffersData['airlineList'][] = [
                'code' => $carrierCode,
                'name' => $airlineNames[$carrierCode] ?? "",
                'logo' => $airlineLogo
            ];
        }

        /**
         * 
         * Layovers 
         */
        $flightOffersData['layover'] = [];
        $processedSegments = [];
        foreach ($flightOffersData['data']['flightOffers'] as $flightOffer) {
            foreach ($flightOffer['itineraries'] as $itineraries) {
                $segments = $itineraries['segments'];
                for ($i = 0; $i < count($segments) - 1; $i++) {
                    $currentFlight = $segments[$i];
                    $nextFlight = $segments[$i + 1];
                    $currentSegmentId = $currentFlight['id'];

                    if (!in_array($currentSegmentId, $processedSegments)) {
                        $arrivalTime = $currentFlight['arrival']['at'];
                        $departureTime = $nextFlight['departure']['at'];
                        $difference = getTimeDifference($departureTime, $arrivalTime);

                        $flightOffersData['layover'][] = [
                            'fromSegmentId'       => $currentFlight['id'],
                            'toSegmentId'         => $nextFlight['id'],
                            'layoverDurationText' => $difference['hours'] . 'h ' . $difference['minutes'] . 'm',
                            'layoverDuration'     => (($difference['hours'] < 9) ? str_pad($difference['hours'], 2, '0', STR_PAD_LEFT) : $difference['hours']) . ':' . $difference['minutes']
                        ];

                        $processedSegments[] = $currentSegmentId;
                    }
                }
            }
        }

        return $flightOffersData;
    }


    public function flightOfferPriceBags($requestPayload, $parentLogId = '', $customerId = '', $amaClientRef = '')
    {
        $result = [
            'success' => false,
            'data' => [],
            'message' => ''
        ];

        $postfields = json_encode($requestPayload, JSON_PRETTY_PRINT);

        $this->checkTokenHealth();

        $appURL = $this->amadeusAPIEndPoint . "/v1/shopping/flight-offers/pricing?include=bags";
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/json',
                "Authorization: Bearer $this->amadeusAPISecret",
                "ama-client-ref: $amaClientRef",
            ],
            'body' =>  $postfields,
        ]);

        $flightOffersData = json_decode($response, true);
        if (isset($flightOffersData['errors'])) {
            $result['message'] = 'Unable to get bag details';
        } else {
            if ($parentLogId != '') {
                $this->createSupplierLog('Flight', 'Amadeus', 'flight-offer-price-with-bags', $postfields, $response, $customerId, $parentLogId, $amaClientRef);
            }
            $result = $flightOffersData;
        }
        return $result;
    }

    /**
     * fetch flight offers price details with amenities
     */
    public function flightOfferPriceUpsellingGet($requestData)
    {
        $this->checkTokenHealth();

        $appURL = $this->amadeusAPIEndPoint . "/v1/shopping/flight-offers/upselling";
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/json',
                "Authorization: Bearer $this->amadeusAPISecret"
            ],
            'body' =>  $requestData,
        ]);

        $flightOffersData = json_decode($response, true);

        if (array_key_exists('data', $flightOffersData) == false) {
            $requestData = json_decode($requestData, true);
            $flightOffersData = [
                "meta" => [
                    "count" => 1,
                ],
                "data" => [$requestData['data']['flightOffers'][0]]
            ];
        }

        $brandedFares = [];
        $brandedFareBySegment = [];
        if (count($flightOffersData['data']) > 0) {
            foreach ($flightOffersData['data'] as $key => $flightOffer) {
                $brandedFares[$flightOffer['id']] = $this->createBrandedFare($flightOffer, true);
                $brandedFareBySegment[] = BrandedFareService::createBrandedFare($flightOffer);
            }
        }
        // $flightOffersData['brandedFares'] = $brandedFares;
        $flightOffersData['brandedFares'] = self::checkBaggageBySegment($brandedFares, $brandedFareBySegment);
        $flightOffersData['brandedFaresBySegment'] = $brandedFareBySegment;
        return $flightOffersData;
    }

    private static function checkBaggageBySegment(&$brandedFares, $brandedFareBySegment)
    {
        $segmentIndex = 0;
        foreach ($brandedFares as $index => &$fare) {

            if (!isset($brandedFareBySegment[$segmentIndex]))
            {
                $segmentIndex++;
                continue;            
            } 

            $segmentGroup = $brandedFareBySegment[$segmentIndex];

            $cabinQty = [];
            $checkedQty = [];

            foreach ($segmentGroup as $segment) {
                if (!empty($segment['baggageDetails']['cabin_quantity'])) {
                    $cabinQty[] = (int) $segment['baggageDetails']['cabin_quantity'];
                }
                if (isset($segment['baggageDetails']['checked_quantity']) && $segment['baggageDetails']['checked_quantity'] !== null) {
                    $checkedQty[] = (int) $segment['baggageDetails']['checked_quantity'];
                }
            }
            // Cabin Baggage Logic
            $cabinDescEn = $cabinDescAr = '';
            if (!empty($cabinQty)) {
                $zeroCount = count(array_filter($cabinQty, fn($qty) => $qty === 0));
                $totalCount = count($cabinQty);
                $count = abs($zeroCount - $totalCount);
                if ($count  == 0 && in_array(0, $cabinQty, true)) {
                    $cabinStatus = false;
                    $minCabinQty = '';
                    $cabinDescEn =  ' No Cabin Baggage';
                    $cabinDescAr =  ' لا يُسمح بأمتعة المقصورة';
                } elseif (($count  != 0 && $count  < $totalCount) && in_array(0, $cabinQty, true)) {
                    $cabinStatus = 'partial';
                    $minCabinQty = '';
                    $cabinDescEn =  ' Partial Cabin Baggage';
                    $cabinDescAr =  ' أمتعة المقصورة الجزئية';
                } else {
                    $cabinSame = count(array_unique($cabinQty)) === 1;
                    $cabinStatus = $cabinSame ? 'same' : 'partial';
                    $minCabinQty = min($cabinQty);
                    $cabinDescEn =  ' Cabin Baggage';
                    $cabinDescAr =  ' أمتعة المقصورة';
                }
            } else {
                $cabinStatus = false;
                $minCabinQty = '';
                $cabinDescEn =  ' No Cabin Baggage';
                $cabinDescAr =  ' لا يُسمح بأمتعة المقصورة';
            }

            // Checked Baggage Logic
            $checkedDescEn = $checkedDescAr = '';
            if (!empty($checkedQty)) {
                $zeroCount = count(array_filter($checkedQty, fn($qty) => $qty === 0));
                $totalCount = count($checkedQty);
                $count = abs($zeroCount - $totalCount);
                if ($count  == 0 && in_array(0, $checkedQty, true)) {
                    $checkedStatus = false;
                    $minCheckedQty = '';
                    $checkedDescEn =  ' No Checked Baggage';
                    $checkedDescAr =  ' لا يُسمح بتسجيل الأمتعة';
                } elseif (($count  != 0 && $count  < $totalCount) && in_array(0, $checkedQty, true)) {
                    $checkedStatus = 'partial';
                    $minCheckedQty = '';
                    $checkedDescEn =  ' Partial Checked Baggage';
                    $checkedDescAr =  ' الأمتعة المسجلة جزئيًا';
                } else {
                    $checkedSame = count(array_unique($checkedQty)) === 1;
                    $checkedStatus = $checkedSame ? 'same' : 'partial';
                    $minCheckedQty = min($checkedQty);
                    $checkedDescEn =  ' Checked Baggage';
                    $checkedDescAr =  ' الأمتعة المسجلة';
                }
            } else {
                $checkedStatus = false;
                $minCheckedQty = '';
                $checkedDescEn =  ' No Checked Baggage';
                $checkedDescAr =  ' لا يُسمح بتسجيل الأمتعة';
            }


            // $status = ($cabinStatus === 'same' && $checkedStatus === 'same') ? 'same' : 'partial';

            // Cabin Baggage
            $minCabinQty = $minCabinQty ? ($minCabinQty <= 5 ? ($minCabinQty ? $minCabinQty . ' ' : '') : ($minCabinQty ? $minCabinQty . ' ' : 'Kg'))  : '';
            $fare['cabinBaggage']['baggageExist'] = $cabinStatus;
            $fare['cabinBaggage']['description_en'] = $cabinStatus == 'partial' ? $minCabinQty . $cabinDescEn : $fare['cabinBaggage']['description_en'];
            $fare['cabinBaggage']['description_ar'] = $cabinStatus == 'partial' ? $minCabinQty .  $cabinDescAr : $fare['cabinBaggage']['description_ar'];
            // $fare['cabinBaggage']['finalQty'] = $cabinQty;

            // Checked Baggage
            if ($minCheckedQty) {
                if ($minCheckedQty <= 5) {
                    $minCheckedQty = $minCheckedQty . ' ';
                } else {
                    $minCheckedQty = $minCheckedQty . 'Kg';
                }
            } else {
                $minCheckedQty = '';
            }

            $fare['checkedBaggage'][0]['baggageExist'] = $checkedStatus;
            $fare['checkedBaggage'][0]['description_en'] = $checkedStatus == 'partial' ? $minCheckedQty .  $checkedDescEn : $fare['checkedBaggage'][0]['description_en'];
            $fare['checkedBaggage'][0]['description_ar'] = $checkedStatus == 'partial' ? $minCheckedQty .  $checkedDescAr : $fare['checkedBaggage'][0]['description_ar'];
            // $fare['checkedBaggage'][0]['finalQty'] = $checkedQty;
            $segmentIndex++;
        }

        return $brandedFares;
    }



    /**
     * fetch flight offers seatmap details with amenities
     */
    public function flightOfferSeatmapAmenities($requestData, $parentLogId = '', $customerId = '', $amaClientRef = '')
    {
        $this->checkTokenHealth();

        $appURL = $this->amadeusAPIEndPoint . "/v1/shopping/seatmaps";
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/json',
                "Authorization: Bearer $this->amadeusAPISecret",
                "ama-client-ref: $amaClientRef",
            ],
            'body' =>  $requestData,
        ]);
        if ($parentLogId != '') {
            $this->createSupplierLog('Flight', 'Amadeus', 'flight-seatmap', $requestData, $response, $customerId, $parentLogId, $amaClientRef);
        }

        $flightOffersData = json_decode($response, true);
        if (empty($flightOffersData['data'])) {
            $flightOffersData['success'] = "No amenities found in this flight";
            return $flightOffersData;
        }
        //dynamically create array for airlineList
        $flightOffersData['amenities'] = [];
        foreach ($flightOffersData['data'] as $data) {

            $Airline[$data['id']] = [];
            if (array_key_exists('aircraftCabinAmenities', $data) && is_array($data['aircraftCabinAmenities']) && count($data['aircraftCabinAmenities']) > 0) {
                foreach ($data['aircraftCabinAmenities'] as $amenitiesKey => $amenities) {

                    $tempAmenitiesArr = [
                        'amenities' => $amenitiesKey,
                        'icon' => URL::to('/') . '/assets/images/amenities/' . $amenitiesKey . '.svg',
                        'segmentId' => $data['segmentId']
                    ];
                    array_push($Airline[$data['id']], $tempAmenitiesArr);
                }
            }
            array_push($flightOffersData['amenities'], $Airline[$data['id']]);
        }

        return $flightOffersData;
    }
    /**
     * create flight order using traveler's details
     * created date 19-12-2023
     */
    public function flightNewOrderCreate($requestData, $amaClientRef)
    {
        $result = [
            'status' => 'false',
            'data' => '',
            'message' => ''
        ];
        $this->checkTokenHealth();
        //convert array into json request for amadeus api
        $postFields = json_encode($requestData, JSON_PRETTY_PRINT);
        //echo $postFields; die;
        $appURL = $this->amadeusAPIEndPoint . "/v1/booking/flight-orders";
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/vnd.amadeus+json',
                "Authorization: Bearer $this->amadeusAPISecret",
                "ama-client-ref: $amaClientRef",
            ],
            'body' => $postFields,
        ]);
        $response = json_decode($response, true);
        if (isset($response['errors']) && is_array($response['errors'])) {
            $result['message'] = json_encode($response['errors']);
        } else {
            $result['status'] = 'true';
            $result['data'] = $response['data'];
        }
        $result['response'] = $response;
        return $result;
    }

    /**
     * create flight order using traveler's details
     * created date 19-12-2023
     */
    public function flightNewOrderIssuanceCreate($supplierReference, $amaClientRef)
    {
        $result = [
            'status' => 'false',
            'data' => '',
            'message' => ''
        ];
        $this->checkTokenHealth();
        //convert array into json request for amadeus api
        $postfields = [];
        $appURL = $this->amadeusAPIEndPoint . "/v1/booking/flight-orders/" . $supplierReference . "/issuance";
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/vnd.amadeus+json',
                "Authorization: Bearer $this->amadeusAPISecret",
                "ama-client-ref: $amaClientRef",
            ],
            'body' =>  $postfields,
        ]);
        $response = json_decode($response, true);
        if (isset($response['errors']) && is_array($response['errors'])) {
            $result['message'] = json_encode($response['errors']);
        } else {
            $result['status'] = 'true';
            $result['data'] = '';
        }
        $result['response'] = $response;
        return $result;
    }

    public function addOriginalFlightOffersToBooking($requestData, $flightNewOrderResponse)
    {
        // Add a new key 'originalFlightOffers' with a copy of 'flightOffers' into response
        $flightNewOrderResponse['data']['originalFlightOffers'] = array_map(function ($flightOffer) {
            return $flightOffer;
        }, $flightNewOrderResponse['data']['flightOffers']);



        // Use array_map to transform the data
        $flightNewOrderResponse['data']['flightOffers'] = array_map(function ($flightOffer) use ($requestData) {
            // Replace values in the nested arrays as needed
            $flightOffer = $this->getPricesWithMarkup($flightOffer, $requestData);
            return $flightOffer;
        }, $flightNewOrderResponse['data']['flightOffers']);

        return $flightNewOrderResponse;
    }
    /**
     * use flightoffers obejct to replace and modify and return new array with added markup and tax
     * use arrData which is json request of flight-offer-pricing api
     */
    public function getPricesWithMarkup($flightOffersData, $requestData = [], $customerId = '', $isDomestic = false)
    {
        $query = Markups::query();
        $query->with(['getServiceType', 'getChannel', 'getOriginCountry.countryCode', 'getOriginCity.cityCode', 'getOriginAirport.AirportName', 'getDestinationCountry.countryCode', 'getDestinationCity.cityCode', 'getDestinationAirport.AirportName', 'getAirline.getMarkupsAirline', 'getSupplier.getMarkupsSupplier', 'getAgent']);
        $query->select(
            'markups.*'
        );
        $query->where('priority', '1');
        $result = $query->first();

        if ($result) {
            //get airline codes into array which we need to add markup
            $airlineCodeArr = collect($result['getAirline'])->map(function ($item) {
                return $item['airline_code'];
            })->all();

            //get supplier ids on which we need to add markup
            $supplierArr = collect($result['getSupplier'])->map(function ($item) {
                $supplierCode = Suppliers::where('id', $item['supplier_id'])->get('code')[0]['code'];
                return $supplierCode;
            })->all();

            $markupIDArr = [];

            foreach ($flightOffersData['itineraries'] as $itinerary) {
                foreach ($itinerary['segments'] as $segment) {

                    //get only date from departure['at'] format
                    $dateString = $segment['departure']['at'];
                    $dateTime = new DateTime($dateString);
                    $departureTime = $dateTime->format('Y-m-d');

                    $carrierCode = $segment['carrierCode'];

                    // Check conditions: departure time and carrier code
                    if (
                        $departureTime >= $result['from_travel_date'] &&
                        $departureTime <= $result['to_travel_date'] &&
                        date('Y-m-d') >= $result['from_booking_date'] &&
                        date('Y-m-d') <= $result['to_booking_date'] &&
                        in_array($carrierCode, $airlineCodeArr)
                    ) {

                        // Add the 'id' value to the temparr array
                        $markupIDArr[] = $segment['id'];
                    }
                }
            }

            if (!empty($markupIDArr)) {
                foreach ($flightOffersData['travelerPricings'] as &$travelerPricing) {
                    // Check traveler type and initiate the condition
                    if ($travelerPricing['travelerType'] === strtoupper($result['pax_type'])) {
                        foreach ($travelerPricing['fareDetailsBySegment'] as $fareDetail) {
                            // Check conditions: cabin, class, segmentId
                            if (
                                $fareDetail['cabin'] === strtoupper($result['cabin_class']) &&
                                $fareDetail['class'] === $result['booking_class'] &&
                                in_array($fareDetail['segmentId'], $markupIDArr) &&
                                in_array($flightOffersData['price']['supplier'], $supplierArr)
                            ) {
                                $sumBase = 0;
                                $sumTotal = 0;
                                if ($result['comm_markup_on'] == 'base_fare') {
                                    if ($travelerPricing['price']['base'] >= $result['from_price_range'] && $travelerPricing['price']['base'] <= $result['to_price_range']) {
                                        $travelerPricing['price']['base'] = priceWithMarkup($result['b2c_markup_type'], $result['b2c_markup'], $travelerPricing['price']['base']);
                                        $sumBase += $travelerPricing['price']['base'];
                                    }
                                } else if ($result['comm_markup_on'] == 'total_fare') {
                                    if ($travelerPricing['price']['total'] >= $result['from_price_range'] && $travelerPricing['price']['total'] <= $result['to_price_range']) {
                                        $travelerPricing['price']['total'] = priceWithMarkup($result['b2c_markup_type'], $result['b2c_markup'], $travelerPricing['price']['total']);
                                    }
                                } else if ($result['comm_markup_on'] == 'net_fare') {
                                    //logic for net fare
                                } else {
                                    //logic for base_fare+YQ
                                }
                            }
                        }
                    }
                }
            }
        } else {
            if (isset($requestData['bookingDetails'])) {
                $supplier = $requestData['bookingDetails']['supplier'];
                $agencyId = $requestData['bookingDetails']['agencyId'];
                $languageCode = $requestData['bookingDetails']['customer_language_code'];
                $currency = $requestData['bookingDetails']['customer_currency'];
            } else {
                $supplier = $requestData['supplier'];
                $agencyId = $requestData['agencyId'];
                $languageCode = $requestData['languageCode'];
                $currency = $requestData['currencyCode'];
            }
            $getSupplierID = Suppliers::where('code', $supplier)->value('id');

            //get default markup details
            $query = DefaultMarkup::with(['getSupplier' => function ($q) use ($getSupplierID) {
                $q->where('supplier_id', $getSupplierID);
            }])->first();

            if ($agencyId == '0') {
                $markupType = $query['b2c_markup_type'];
                $markupValue = $query['b2c_markup'];
            } else {
                $markupType = $query['b2b_markup_type'];
                $markupValue = $query['b2b_markup'];
            }

            $chargeableSeatPriceTotal = 0;
            if (isset($flightOffersData['price']['additionalServices']) && is_array($flightOffersData['price']['additionalServices'])) {
                foreach ($flightOffersData['price']['additionalServices'] as $additionalService) {
                    if ($additionalService['type'] == 'SEATS') {
                        $chargeableSeatPriceTotal += $additionalService['amount'];
                    }
                }
            }

            $travelersTotalPriceSum = $chargeableSeatPriceTotal;
            $travelersBasePriceSum = $totalTaxAmount = 0;
            $generalVATPercentage = Setting::where('config_key', 'general|site|defaultVatPercentage')->get('value')[0]['value'];
            $generalServiceFee = Setting::where('config_key', 'general|site|flightServiceFee')->get('value')[0]['value'];
            foreach ($flightOffersData['travelerPricings'] as &$travelerPricing) {
                // Initialize the sum variable
                $sumOfAllAmounts = 0;

                //set travelers wise converted currency rate start
                foreach ($travelerPricing['price']['taxes'] as $tax) {
                    $sumOfAllAmounts += $tax['amount'];
                }

                $convertedTotalPrice = convertCurrencyExchangeRate($travelerPricing['price']['total'], Currency::SAR->value, $currency, $requestData);

                $convertedBasePrice = convertCurrencyExchangeRate($travelerPricing['price']['base'], Currency::SAR->value, $currency, $requestData);

                $convertedTaxAmount = convertCurrencyExchangeRate($sumOfAllAmounts, Currency::SAR->value, $currency, $requestData);


                $displayCurrency = ($languageCode == strtoupper(Locale::Arabic->value)) ? $convertedTotalPrice['data']['symbol'] : $convertedTotalPrice['data']['toCurrencyCode'];
                //remove vat from traveler's base price
                $baseRemovedVAT = $convertedBasePrice['data']['convertedRate'] / (1 + ($generalVATPercentage / 100));

                $travelerVAT = ($isDomestic == true) ? $baseRemovedVAT * $generalVATPercentage / 100 : $convertedBasePrice['data']['convertedRate'] * $generalVATPercentage / 100;
                $travelerPricing['processedPrice'] = [
                    'currency' => $displayCurrency,
                    'totalConverted' => strval($convertedTotalPrice['data']['convertedRate']),
                    'totalFormatted' => $convertedTotalPrice['data']['formattedPrice'],
                    'baseFareConverted' => strval($convertedBasePrice['data']['convertedRate']),
                    'baseFareFormatted' => $convertedBasePrice['data']['formattedPrice'],
                    'taxTotalConverted' => strval($convertedTaxAmount['data']['convertedRate']),
                    'taxTotalFormatted' => $convertedTaxAmount['data']['formattedPrice'],
                    'travelerVAT' => number_format($travelerVAT, '2', '.', ''),


                ];
                /*set travelers wise converted currency rate end*/



                //get sum of total and base of all travelers
                $travelersTotalPriceSum += $convertedTotalPrice['data']['convertedRate'];
                $travelersBasePriceSum += $baseRemovedVAT;
                $totalTaxAmount += $sumOfAllAmounts;
            }
            $domesticPrice = $travelersBasePriceSum * $generalVATPercentage / 100;

            $markupFee = ($markupType == 'percentage') ? $travelersTotalPriceSum * $markupValue / 100 : $markupValue;
            $markupServiceFee = $markupFee + $generalServiceFee;
            if ($isDomestic) {
                $vatIn = ($travelersTotalPriceSum * $generalVATPercentage) / 100;
            } else {
                $vatIn = 0;
            }
            $vatOut = ($markupServiceFee * $generalVATPercentage) / 100;
            //$vat = ($isDomestic == true) ?  $domesticPrice + ($markupFee * $generalVATPercentage / 100) : $markupFee * $generalVATPercentage / 100;
            $vat = $vatIn + $vatOut;
            $totalInclusiveTax = $travelersTotalPriceSum  + $markupServiceFee + $vat;

            $payload = [
                'service' => 'Hotel Booking',
                'amount' => $totalInclusiveTax,
                'fromCurrency' => Currency::SAR->value,
                'toCurrency' => $currency
            ];
            $loyaltyPoints = $this->getLoyaltyPoints($payload);
            $redeemDetails = [];
            if ($customerId != '') {
                $redeemDetails = $this->getCustomerMaxRedeemPoints($customerId, 'Flight', $currency);
            }

            $travelersTotalPriceSum = $travelersTotalPriceSum - $chargeableSeatPriceTotal;

            //get grandTotal of all kind of prices with separate key
            $flightOffersData['processedPrice'] = [
                'currency' => $displayCurrency,
                'markupFee' => number_format($markupFee, '2', '.', ''),
                'serviceFee' => number_format($generalServiceFee, '2', '.', ''),
                'markupServiceFee' => number_format($markupServiceFee, '2', '.', ''),
                'isDomestic' => $isDomestic,
                'vatIn' => number_format($vatIn, '2', '.', ''),
                'vatOut' => number_format($vatOut, '2', '.', ''),
                'vat' => number_format($vat, '2', '.', ''),
                'travelersPrice' => number_format($travelersTotalPriceSum, '2', '.', ''),
                'totalPrice' => number_format($travelersTotalPriceSum, '2', '.', ''),
                'grandTotal' => number_format($totalInclusiveTax, '2', '.', ''),
                'markupType' => $markupType,
                'markupValue' => $markupValue,
                'totalTaxAmount' => $totalTaxAmount,
                'chargeableSeatPrice' => number_format($chargeableSeatPriceTotal, '2', '.', ''),
                'loyaltyPoints' => $loyaltyPoints,
                'redeemDetails' => $redeemDetails
            ];
        }

        return $flightOffersData;
    }
    /**
     * create flight order using traveler's details
     * created date 19-12-2023
     */
    public function flightNewOrderCreateOld($requestData)
    {
        $result = [
            'status' => 'false',
            'data' => '',
            'message' => ''
        ];
        $this->checkTokenHealth();
        //convert array into json request for amadeus api
        $postfields = json_encode($requestData, JSON_PRETTY_PRINT);
        $appURL = $this->amadeusAPIEndPoint . "/v1/booking/flight-orders";
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/vnd.amadeus+json',
                "Authorization: Bearer $this->amadeusAPISecret",
            ],
            'body' =>  $postfields,
        ]);
        $response = json_decode($response, true);
        if (isset($response['errors']) && is_array($response['errors'])) {
            $result['message'] = json_encode($response['errors']);
        } else {
            $result['status'] = 'true';

            // Add a new key 'originalFlightOffers' with a copy of 'flightOffers' into response
            $response['data']['originalFlightOffers'] = array_map(function ($flightOffer) {
                return $flightOffer;
            }, $response['data']['flightOffers']);

            // Use array_map to transform the data
            $response['data']['flightOffers'] = array_map(function ($flightOffer) use ($requestData) {
                // Replace values in the nested arrays as needed
                $flightOffer = $this->getPricesWithMarkup($flightOffer, $requestData);

                return $flightOffer;
            }, $response['data']['flightOffers']);

            $result['data'] = $response['data'];
        }

        return $result;
    }

    public function flightBookingCancel($bookingReference)
    {
        $result = [
            'status' => 'false',
            'data' => '',
            'http_code' => '',
            'message' => ''
        ];
        $this->checkTokenHealth();

        $appURL = $this->amadeusAPIEndPoint . "/v1/booking/flight-orders/" . $bookingReference;
        $headers = [
            'Content-Type: application/vnd.amadeus+json',
            "Authorization: Bearer $this->amadeusAPISecret"
        ];
        $cancelResponse = $this->libCurl($appURL, 'DELETE', $headers);
        if ($cancelResponse['status'] == 'false') {
            $result['message'] = 'Request failed: ' . $cancelResponse['message'];
        } else {
            $cancelResponseData = $cancelResponse['data'];
            if ($cancelResponse['http_code'] == 204) {
                $result['status'] = 'true';
                $result['data'] = $cancelResponseData['booking'];
                $result['http_code'] = $cancelResponseData['http_code'];
                $result['response'] = $cancelResponseData;
                $result['message'] = 'Flight booking have cancelled successfully';
            } else {
                $errorMessages = [];
                if (array_key_exists('errors', $cancelResponseData) && is_array($cancelResponseData['errors'])) {
                    foreach ($cancelResponseData['errors'] as $key => $value) {
                        $tempError = [];
                        if (array_key_exists('status', $value)) {
                            array_push($tempError, ('Status: ' . $value['status']));
                        }
                        if (array_key_exists('code', $value)) {
                            array_push($tempError, ('Code: ' . $value['code']));
                        }
                        if (array_key_exists('title', $value)) {
                            array_push($tempError, ('Title: ' . $value['title']));
                        }
                        if (array_key_exists('detail', $value)) {
                            array_push($tempError, ('Detail: ' . $value['detail']));
                        }
                        array_push($errorMessages, implode(', ', $tempError));
                    }
                }
                $result['message'] = (count($errorMessages) > 0) ? implode('<br/>', $errorMessages) : '';
            }
        }
        return $result;
    }

    public function flightCalenderSearch($requestData)
    {
        $this->checkTokenHealth();

        $appURL = $this->amadeusAPIEndPoint . "/v2/shopping/flight-offers";

        $amaClientRef = $this->getUuid();
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/json',
                "Authorization: Bearer $this->amadeusAPISecret",
                "ama-client-ref: $amaClientRef",
            ],
            'body' =>  $requestData,
        ]);
        $flightOffersData = json_decode($response, true);
        return $flightOffersData;
    }


    public function priceDetails($requestData)
    {
        $this->checkTokenHealth();

        $appURL = $this->amadeusAPIEndPoint . "/v1/shopping/flight-offers/pricing?include=detailed-fare-rules,credit-card-fees,other-services";
        $amaClientRef = $this->getUuid();
        $response = CurlRequestService::sendRequest([
            'method' => 'POST',
            'url' => $appURL,
            'headers' => [
                'Content-Type: application/json',
                "Authorization: Bearer $this->amadeusAPISecret",
                "ama-client-ref: $amaClientRef",
            ],
            'body' =>  $requestData,
        ]);

        $priceDetailsData = json_decode($response, true);

        return $priceDetailsData;
    }

    public function getSeatmapFlightOffers2($bookingDetails, $amaClientRef)
    {
        if (
            array_key_exists('seatDetails', $bookingDetails['flightDetails']) &&
            $bookingDetails['flightDetails']['seatDetails']['status'] == true
        ) {
            $seatMapAdded = false;
            $seatDetailsByTravelerAndSegment = [];
            $seatDetails = $bookingDetails['flightDetails']['seatDetails']['seatDetails'];
            /*foreach ($seatDetails as $seatDetail) {
                foreach ($seatDetail as $seatDetailChild) {
                    //if ($seatDetailChild['selected'] == 1) {
                        if ($seatDetailChild['segmentId'] == 1) {
                            $seatDetailChild['segmentId'] = 55;
                        }
                        if ($seatDetailChild['segmentId'] == 2) {
                            $seatDetailChild['segmentId'] = 56;
                        }
                        $seatNumber = $seatDetailChild['ticket_details']['seat_number'];
                        if ($seatNumber != '' && $seatNumber != 'null') {
                            $seatDetailsByTravelerAndSegment[$seatDetailChild['travelerId']][$seatDetailChild['segmentId']] = $seatNumber;
                        }
                    //}
                }
            }*/
            $segmentIds = [];
            $i = 1;
            foreach ($bookingDetails['flightDetails']['flightInfo'] as $flightOffer) {
                foreach ($flightOffer['travelerPricings'] as $travelerPricing) {
                    foreach ($travelerPricing['fareDetailsBySegment'] as $fareDetailsBySegment) {
                        if (in_array($fareDetailsBySegment['segmentId'], $segmentIds) == false) {
                            $segmentIds[$i++] = $fareDetailsBySegment['segmentId'];
                        }
                    }
                }
            }

            foreach ($seatDetails as $seatDetail) {
                foreach ($seatDetail as $seatDetailChild) {
                    //if ($seatDetailChild['selected'] == 1) {
                    $segmentId = $segmentIds[$seatDetailChild['segmentId']];
                    $seatNumber = $seatDetailChild['ticket_details']['seat_number'];
                    if ($seatNumber != '' && $seatNumber != 'null') {
                        $seatDetailsByTravelerAndSegment[$seatDetailChild['travelerId']][$seatDetailChild['segmentId']] = $seatNumber;
                    }
                    //}
                }
            }
            foreach ($bookingDetails['flightDetails']['flightInfo'] as &$flightOffer) {
                foreach ($flightOffer['travelerPricings'] as &$travelerPricing) {
                    foreach ($travelerPricing['fareDetailsBySegment'] as &$fareDetailsBySegment) {
                        if (array_key_exists($travelerPricing['travelerId'], $seatDetailsByTravelerAndSegment)) {
                            if (array_key_exists($fareDetailsBySegment['segmentId'], $seatDetailsByTravelerAndSegment[$travelerPricing['travelerId']])) {
                                if (array_key_exists('additionalServices', $fareDetailsBySegment) == false) {
                                    $fareDetailsBySegment['additionalServices'] = [];
                                }
                                $seatNumber = $seatDetailsByTravelerAndSegment[$travelerPricing['travelerId']][$fareDetailsBySegment['segmentId']];
                                $fareDetailsBySegment['additionalServices']['chargeableSeatNumber'] = $seatNumber;
                                $seatMapAdded = true;
                            }
                        }
                    }
                }
            }
            $payload = [
                "agencyId" => "0",
                "type" => "flight-offers-pricing",
                "currencyCode" => "SAR",
                "supplier" => "AMADUES",
                "languageCode" => "en",
                "flightOffers" => $bookingDetails['flightDetails']['flightInfo']
            ];
            //echo "<pre>"; echo json_encode($payload); die;
            $response = $this->flightOfferPriceGetWithMarkup($payload, '', '', $amaClientRef);
            //echo json_encode($response['data']['flightOffers']);
            return $response;
        }
    }
}
