<?php

namespace App\Services;

use App\Enums\Locale;
use App\Models\HotelDescriptionAr;
use App\Models\HotelDescriptionEn;
use Illuminate\Support\Facades\Cache;

class HotelDescriptionService
{
    /**
     * Get Arabic descriptions for hotels with optional caching
     */
    public function getDescriptionsAr(array $hotelIds, bool $useCache = true)
    {
        if (!$useCache) {
            return HotelDescriptionAr::getByHotelIds($hotelIds);
        }

        sort($hotelIds);
        $hotelIdsStr = implode(',', $hotelIds);
        $cacheKey = 'hotel_descriptions_ar_' . md5($hotelIdsStr);
        
        return Cache::remember($cacheKey, 3600, function () use ($hotelIds) {
            return HotelDescriptionAr::getByHotelIds($hotelIds);
        });
    }

    /**
     * Get English descriptions for hotels with optional caching
     */
    public function getDescriptionsEn(array $hotelIds, bool $useCache = true)
    {
        if (!$useCache) {
            return HotelDescriptionEn::getByHotelIds($hotelIds);
        }

        sort($hotelIds);
        $hotelIdsStr = implode(',', $hotelIds);
        $cacheKey = 'hotel_descriptions_en_' . md5($hotelIdsStr);
        
        return Cache::remember($cacheKey, 3600, function () use ($hotelIds) {
            return HotelDescriptionEn::getByHotelIds($hotelIds);
        });
    }

    /**
     * Get descriptions in both languages
     */
    public function getDescriptionsBoth(array $hotelIds, bool $useCache = true)
    {
        return [
            Locale::Arabic->value => $this->getDescriptionsAr($hotelIds, $useCache),
            Locale::English->value => $this->getDescriptionsEn($hotelIds, $useCache)
        ];
    }

    /**
     * Get descriptions by language code
     */
    public function getDescriptionsByLanguage(array $hotelIds, string $language = Locale::English->value, bool $useCache = true)
    {
        switch ($language) {
            case Locale::Arabic->value:
                return $this->getDescriptionsAr($hotelIds, $useCache);
            case Locale::English->value:
            default:
                return $this->getDescriptionsEn($hotelIds, $useCache);
        }
    }
}