<?php

namespace App\Services;

class FareBasisService
{
    /**
     * Extract penalties (before/after departure + noShow)
     */
    public function extractPenaltyAmounts(string $text, string $sectionType = 'changes'): array
    {
        $result = [
            'beforeDeparture' => null,
            'afterDeparture'  => null,
            'noShow'          => null,
        ];

        $text = strtoupper($text);
        $lines = preg_split('/\r?\n/', $text);

        $currencyPattern = '(USD|GBP|EUR|AED|SAR|INR|CAD|AUD|CHF|CNY|JPY)';
        $amountPattern   = '/' . $currencyPattern . '\s*([\d]+(?:\.\d{1,2})?)/';

        $foundFirstAmount = false;

        foreach ($lines as $line) {
            // 1) NO-SHOW / FAIL TO BOARD
            if (preg_match('/(NO-SHOW|FAIL TO BOARD)/', $line)) {
                if (preg_match($amountPattern, $line, $m)) {
                    $result['noShow'] = [
                        'currency' => $m[1],
                        'amount'   => (float)$m[2],
                    ];
                }
                continue;
            }

            // 2) AFTER DEPARTURE
            if (strpos($line, 'AFTER DEPARTURE') !== false) {
                if (preg_match($amountPattern, $line, $m)) {
                    $result['afterDeparture'] = [
                        'currency' => $m[1],
                        'amount'   => (float)$m[2],
                    ];
                }
                continue;
            }

            // 3) BEFORE DEPARTURE (explicit keywords)
            if (preg_match('/BEFORE DEPARTURE|ANY TIME|REISSUE|AMEND/', $line)) {
                if (preg_match($amountPattern, $line, $m)) {
                    $result['beforeDeparture'] = [
                        'currency' => $m[1],
                        'amount'   => (float)$m[2],
                    ];
                }
                continue;
            }

            // 4) First amount fallback (if no explicit keyword yet)
            if (!$foundFirstAmount && preg_match($amountPattern, $line, $m)) {
                $result['beforeDeparture'] = [
                    'currency' => $m[1],
                    'amount'   => (float)$m[2],
                ];
                $foundFirstAmount = true;
            }
        }

        // 5) Apply defaults
        foreach ($result as $key => $val) {
            if ($val === null) {
                $result[$key] = $sectionType === 'changes'
                    ? 'Non-Amendable'
                    : 'Non-Refundable';
            }
        }

        return $result;
    }



    /**
     * Main method to parse penalties from fare rules
     */
    public function formatFarePenalties(array $fare, $requestedCurrency = 'SAR'): array
    {
        $penalties = array_filter(
            $fare['fareNotes']['descriptions'] ?? [],
            fn($d) => strtoupper($d['descriptionType']) === 'PENALTIES'
        );

        $penaltyText = '';
        if (!empty($penalties)) {
            $penaltyText = array_values($penalties)[0]['text'];
        }

        $sections = [
            'changes'       => '',
            'cancellations' => '',
        ];

        $lines = preg_split('/\r?\n/', $penaltyText);
        $currentSection = null;
        foreach ($lines as $line) {
            $upper = strtoupper(trim($line));

            // Detect section headers
            if (strpos($upper, 'CHANGES/CANCELLATIONS') === 0) {
                // Assign based on last seen section
                if ($currentSection === 'changes') {
                    $sections['changes'] .= $line . "\n";
                    $currentSection = 'changes';
                } elseif ($currentSection === 'cancellations') {
                    $sections['cancellations'] .= $line . "\n";
                    $currentSection = 'cancellations';
                } else {
                    // fallback: assign to both if no context yet
                    $sections['changes'] .= $line . "\n";
                    $sections['cancellations'] .= $line . "\n";
                }
                continue;
            }

            if (strpos($upper, 'CHANGES') === 0) {
                $currentSection = 'changes';
                continue;
            }

            if (strpos($upper, 'CANCELLATIONS') === 0) {
                $currentSection = 'cancellations';
                continue;
            }

            // Append lines into current section
            if ($currentSection && $line !== '') {
                $sections[$currentSection] .= $line . "\n";
            }
        }

        $changes       = $this->extractPenaltyAmounts($sections['changes'], 'changes');
        $cancellations = $this->extractPenaltyAmounts($sections['cancellations'], 'cancellations');

        $changes = array_map(fn($v) => is_array($v) ? $this->convertAmountIfNeeded($v, $requestedCurrency) : $v, $changes);
        $cancellations = array_map(fn($v) => is_array($v) ? $this->convertAmountIfNeeded($v, $requestedCurrency) : $v, $cancellations);

        $finalResult =  [
            'changes'       => $changes,
            'cancellations' => $cancellations,
            'originalText'  => $penaltyText,
        ];

        return $finalResult;
    }


    private function convertAmountIfNeeded(array $amountData, string $requestedCurrency): array
    {
        if (!isset($amountData['currency']) || !isset($amountData['amount'])) {
            return $amountData;
        }

        if (strtoupper($amountData['currency']) === strtoupper($requestedCurrency)) {
            return $amountData;
        }

        $currencyExchangeService = new CurrencyApiClientService(env('CURRENCY_EXCHANGE_KEY'));
        $exchangeRate = $currencyExchangeService->convertAmount($amountData['amount'], $amountData['currency'], $requestedCurrency);

        return [
            'currency' => strtoupper($requestedCurrency),
            'amount'   => $exchangeRate
        ];
    }
}
