<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class GoogleMapLog extends Model
{
    use HasFactory;
    protected $fillable = [
        'created_by',
        'log_name',
        'view_count',
    ];

    /**
     * Logs a view for the Google Map feature.
     * 
     * If the user is authenticated, increments the view count for the current day.
     * Otherwise, creates a new log entry with an initial view count.
     *
     * @param int|string|null $userId   The ID of the authenticated user, or 'Guest' if not logged in.
     * @param string $logName           The name of the log event. Defaults to 'Google Place API - Around this Hotel'.
     * @return void
     */

    public static function createGoogleMapLog($userId, $logName = 'Google Place API - Around this Hotel')
    {
        $userId = $userId ?? 'Guest';

        $todayStart = Carbon::today();
        $todayEnd = Carbon::today()->endOfDay();

        $existingLog = self::where('created_by', $userId)
            ->where('log_name', $logName)
            ->whereBetween('created_at', [$todayStart, $todayEnd])
            ->first();

        if ($existingLog) {
            $existingLog->increment('view_count');
        } else {
            return self::create([
                'created_by' => $userId,
                'log_name' => $logName,
                'view_count' => 1
            ]);
        }
    }
}
