<?php

namespace App\Jobs;

use App\Models\HotelAddress;
use App\Models\MostBookedHotel;
use Illuminate\Bus\Queueable;
use Illuminate\Support\Facades\DB;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class MostBookedHotelJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $hotelId;
    protected $hotelName;
    protected $bookingId;

    /**
     * Create a new job instance.
     */
    public function __construct($hotelId, $hotelName, $bookingId)
    {
        $this->hotelId = $hotelId;
        $this->hotelName = $hotelName;
        $this->bookingId = $bookingId;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        if (empty($this->hotelId) || empty($this->bookingId)) {
            return; // Required data missing
        }

        $cityName = HotelAddress::where('hotelid', $this->hotelId)->value('cityname');

        DB::transaction(function () use ($cityName) {
            $hotel = MostBookedHotel::lockForUpdate()->where('hotel_id', $this->hotelId)->first();

            if ($hotel) {
                $existingBookingIds = array_filter(explode(',', $hotel->booking_ids));

                if (!in_array($this->bookingId, $existingBookingIds)) {
                    $existingBookingIds[] = $this->bookingId;

                    $hotel->booking_ids = implode(',', $existingBookingIds);
                    $hotel->booked_count = $hotel->booked_count + 1;
                    $hotel->save();
                }
            } else {
                MostBookedHotel::create([
                    'hotel_id' => $this->hotelId,
                    'hotel_name' => $this->hotelName,
                    'city_name' => $cityName,
                    'booked_count' => 1,
                    'booking_ids' => $this->bookingId,
                ]);
            }
        });
    }
}
