<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use App\Models\Role;
use App\Models\User;
use App\Models\Agency;
use App\Models\Module;
use App\Models\Wallet;
use App\Models\Country;
use App\Models\Setting;
use App\Models\AppUsers;
use App\Models\Bookings;
use App\Models\Customer;
use App\Models\WalletLog;
use App\Traits\ActiveLog;
use App\Models\ActivityLog;
use Illuminate\Support\Str;
use App\Traits\EmailService;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;


class WalletController extends Controller
{
    //

    public function index()
    {
        if (!hasPermission('USERS_LIST', 'read')) {
            return view('admin/401');
        }

        $header['title'] = 'Wallet';
        $header['heading'] = 'Wallet';

        $queryStringConcat = '?';
        if (isset($_GET['per_page'])) {
            $queryStringConcat .= '&per_page=' . $_GET['per_page'];
        }
        if (isset($_GET['page'])) {
            $queryStringConcat .= '&page=' . $_GET['page'];
        }

        $filter = [
            'per_page' => request()->input('per_page', '10'),
            'order_by' => request()->input('order_by', 'created_at'),
            'sorting' => request()->input('sorting', 'desc'),
            'status' => request()->input('status', ''),
            'email' => request()->input('email', ''),
        ];

        $WalletListData = Wallet::getFilteredWallets($filter);
        $CustomerEmails = Customer::all();

        return view('admin.wallet.index', [
            'header' => $header,
            'WalletData' => $WalletListData,
            'CustomerEmails' => $CustomerEmails,
            'WalletCountData' => $WalletListData->total(),
            'queryStringConcat' => $queryStringConcat,
            'appliedFilter' => $filter,
            'i' => ($WalletListData->currentPage() - 1) * $filter['per_page'],
        ]);
    }




    public function create()
    {
        if (!hasPermission('USERS_LIST', 'create')) {
            return view('admin/401');
        }
        $getAgencyName = Agency::get();
        $getIsdCode = Country::with('countryCode')->get();

        $customers = Customer::select('id', 'email')
            ->whereNotNull('email')
            ->where('email', '!=', '')
            ->get();

        $bookings = [];

        $header['title'] = "Wallet - Add";
        $header['heading'] = "Wallet - Add";
        $header['breadcrumb'] = "Update";

        return view('admin/wallet/add_update')->with([
            'header' => $header,
            'getIsdCode' => $getIsdCode,
            'customers' => $customers,
            'bookings' => $bookings,
        ]);
    }



    public function getBookingsByCustomer($customerId)
    {
        $bookings = Bookings::where('customer_id', $customerId)->where('booking_status', 'confirmed')->get(['id', 'email', 'booking_ref', 'booking_type']);
        return response()->json($bookings);
    }


    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer' => 'required|exists:customers,id',
            'booking_id' => 'required|exists:bookings,id',
            'balance'  => 'required|numeric|min:0',
            'wallet_currency' => 'required|in:SAR,USD',
            'status'   => 'required|in:active,inactive',
            'expiry_date' => 'required|date',
            'attachment' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ], [
            'wallet_currency.in' => 'The selected wallet currency is invalid. Please enter SAR/USD'
        ]);

        $userId = auth()->id();

        $wallet = new Wallet();
        $wallet->customer_id = $request->customer;
        $wallet->booking_id = $request->booking_id;
        $wallet->balance = $request->balance;
        $wallet->currency = $request->wallet_currency;
        $wallet->type = 'credit';
        $wallet->status = $request->status;
        $wallet->expiry_date = $request->expiry_date;

        if ($request->hasFile('attachment')) {
            $imagePath = $request->file('attachment')->store('wallet_attachments', 'public');
            $wallet->attachment = $imagePath;
        }

        $wallet->save();
        return redirect()->route('wallet.index')->with('success', 'Wallet has been successfully created.');
    }


    public function edit($id)
    {
        $getAgencyName = Agency::get();
        $getIsdCode = Country::with('countryCode')->get();
        $wallet = Wallet::findOrFail($id);
        $customers = Customer::all();

        $bookings = Bookings::where('customer_id', $wallet->customer_id)->get(['id', 'email', 'booking_ref', 'booking_type']);
        $header['title'] = "Wallet - Edit";
        $header['heading'] = "Wallet - Edit";
        $header['breadcrumb'] = "Update";
        $walletLog = WalletLog::where('wallet_id', $id)->orderBy('id', 'desc')->paginate(5);

        return view('admin/wallet/add_update')->with([
            'header' => $header,
            'getIsdCode' => $getIsdCode,
            'customers' => $customers,
            'wallet' => $wallet,
            'bookings' => $bookings,
            'walletLog' => $walletLog
        ]);
    }




    /**
     * Update the specified user in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function update(Request $request, $id)
    {
        if (!hasPermission('USERS_LIST', 'update')) {
            return view('admin/401');
        }

        $request->validate([
            'customer' => 'required|exists:customers,id',
            'booking_id' => 'required|exists:bookings,id',
            'balance'  => 'required|numeric|min:0',
            'wallet_currency' => 'required',
            'status'   => 'required|in:active,inactive',
            'expiry_date' => 'required|date',
            'attachment' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $wallet = Wallet::findOrFail($id);

        if($wallet->wallet_used_status == 'used'){
            return redirect()->route('wallet.index')->with('error', 'Selected Wallet is already used.');
        }

        $is_updated = 'No Field Updated';
        if ($wallet->balance != $request->balance && $wallet->status != $request->status) {
            $is_updated = 'Wallet Balance and Wallet Status Updated';
        } elseif ($wallet->balance != $request->balance) {
            $is_updated = 'Wallet Balance Updated';
        } elseif ($wallet->booking_id != $request->booking_id) {
            $is_updated = 'Booking Updated';
        } elseif ($wallet->status != $request->status) {
            $is_updated = 'Wallet Status Updated';
        }

        $wallet->customer_id = $request->customer;
        $wallet->booking_id = $request->booking_id;
        $wallet->balance = $request->balance;
        $wallet->currency = $request->wallet_currency;
        $wallet->type = 'credit';
        $wallet->status = $request->status;
        $wallet->expiry_date = $request->expiry_date;

        if ($request->hasFile('attachment')) {
            if ($wallet->attachment) {
                Storage::disk('public')->delete($wallet->attachment);
            }

            $imagePath = $request->file('attachment')->store('wallet_attachments', 'public');
            $wallet->attachment = $imagePath;
        }



        $wallet->save();

        return redirect()->route('wallet.index')->with('success', 'Wallet has been successfully updated.');
    }


    public function deleteWallet(Request $request)
    {
        $url = URL::previous();

        if (!hasPermission('USERS_LIST', 'delete')) {
            return view('admin/401');
        }

        $walletIDs = explode(',', $request->input('wallet_ids'));

        // dd($walletIDs);
        $userId = auth()->id();
        $message = "";
        $success = true;
        foreach ($walletIDs as $wallet_id) {
            $wallet = Wallet::find($wallet_id);

            if ($wallet) {
                $response = $wallet->delete();
                $message .= $response ? "Wallet with ID {$wallet_id} deleted successfully.<br>" : "Failed to delete wallet with ID {$wallet_id}.<br>";
            } else {
                $message .= "Wallet with ID {$wallet_id} not found.<br>";
                $success = false;
            }
        }
        if ($success) {
            return redirect()->route('wallet.index')->with('success', 'Selected wallets have been deleted successfully.');
        } else {
            return redirect()->route('wallet.index')->with('error', 'Some wallets could not be deleted. ' . $message);
        }
    }

    public function getCurrencyByBookingId($booking_id)
    {
        $booking = Bookings::find($booking_id);

        if ($booking) {
            return response()->json([
                'status' => true,
                'currency' => $booking->supplier_currency
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Booking not found'
            ]);
        }
    }
}
