<?php

/**
 * @package     Markups
 * @subpackage   Markups
 * @Author     Rehlte Travel and Tourism
 * @Copyright(C) 2025 [Travel Portal].
 * @Version 1.0.0
 * module of the  Markups.
 */

namespace App\Http\Controllers\Admin\Markups;

use DB;
use App\Models\City;
use App\Models\Agency;
use App\Models\Airline;
use App\Models\Airport;
use App\Models\Country;
use App\Models\Markups;
use App\Models\Setting;
use App\Models\MarkupLog;
use App\Models\Suppliers;
use App\Models\ServiceType;
use Illuminate\Http\Request;
use App\Models\DefaultMarkup;
use App\Models\MarkupSetting;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\URL;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Yajra\DataTables\Facades\DataTables;

class FlightMarkupsController extends Controller
{
    /**
     * Display a listing of the markups.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $header['title'] = "Markups List";
        $header['heading'] = "Markups List";
        $appliedFilter = "";
        return view('admin/markups/index')->with(['header' => $header, 'appliedFilter' => $appliedFilter]);
    }


    /**
     * Show the form for creating a new markups.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $header['title'] = "Flight - Add";
        $header['heading'] = "Flight - Add";

        return view('admin/markups/add')->with(['header' => $header]);;
    }

    /**
     * Store a newly created markups in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */

    public function store(Request $request)
    {


        $requestData = $request->only([
            'ruleName',
            'priority',
            'service_type_id',
            'channel',
            'supplier',
            'originCriteria',
            'originName',
            'destinationCriteria',
            'destinationName',
            'airlines',
            'fromBookingDate',
            'toBookingDate',
            'fromTravelDate',
            'toTravelDate',

            'tripType',
            'paxType',
            'from_price_range',
            'to_price_range',
            'fareType',
            'b2c_markup_type',
            'b2c_markup',
            'b2b_markup_type',
            'b2b_markup',
            'service_b2b_markup_type',
            'service_b2b_markup',
            'service_b2c_markup_type',
            'service_b2c_markup',
            'ancillary_service_markup_type',
            'ancillary_service_markup',
            'commMarkupOn',
            'agency',
            'agencyGroup',
            'calculationMethod',
        ]);

        $rules = [
            'ruleName' => 'required',
            'priority' => 'required',


            'originCriteria' => 'required',
            'originName' => 'required',
            'destinationCriteria' => 'required',
            'destinationName' => 'required',

            'fromBookingDate' => 'required',
            'toBookingDate' => 'required',
            'fromTravelDate' => 'required',
            'toTravelDate' => 'required',

            'tripType' => 'required',
            'paxType' => 'required',
            'from_price_range' => 'required',
            'to_price_range' => 'required',
            'fareType' => 'required',
            'b2c_markup_type' => 'required',
            'b2c_markup' => 'required',
            'service_b2c_markup_type' => 'required',
            'service_b2c_markup' => 'required',
            'commMarkupOn' => 'required',
            'calculationMethod' => 'required'

        ];

        $customMessages = [];

        $niceNames = array();

        $this->validate($request, $rules, $customMessages, $niceNames);

        $response = Markups::createMarkups($requestData);

        MarkupLog::create([
            'markup_id' => $response['data']['id'],
            'service_type_id' => $response['data']['service_type_id'],
            'action' => 'Markup Created',
            'created_by' => Auth::id(),
        ]);

        $service_type_id = $response['data']['service_type_id'];
        $getServiceTypeData = ServiceType::select('id', 'name')->where('id', $service_type_id)->first();
        $serviceTypeName = $getServiceTypeData->name ?? '';
        if ($response['status'] == 1 && !empty($response['data'])) {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('success', $response['message']);
        } else {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('error', $response['message']);
        }
    }


    /**
     * Display the specified markups.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request, $id)
    {
        $header['title'] = "Flight - View";
        $header['heading'] = "Flight - View";

        $checkServiceType = ServiceType::where('name', $request->service_type)->first();
        $service_type_id = $checkServiceType->id ?? '';

        $filter = array(
            'id' => $id,
            'service_type_id' => $service_type_id,
        );

        $response = Markups::getMarkupsData($filter);
        $markupsDetail = $response['data'];
        $service_type = $markupsDetail['getServiceType']['name'] ?? '';

        if ($response['status'] == 1 && !empty($response['data'])) {
            return view('admin/markups/flight-markup/view')->with(['header' => $header, 'service_type' => $service_type, 'markupsDetail' => $markupsDetail]);
        } else {
            return redirect()->to('markups/manage?service_type=' . $service_type)->with('error', $response['message']);
        }
    }

    /**
     * Show the form for editing the specified markups.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, $id)
    {
        $header['title'] = "Flight - Edit";
        $header['heading'] = "Flight - Edit";

        $checkServiceType = ServiceType::where('name', $request->service_type)->first();
        $service_type_id = $checkServiceType->id ?? '';

        $filter = array(
            'id' => $id,
            'service_type_id' => $service_type_id,
        );

        $response = Markups::getMarkupsData($filter);
        $markupsDetail = $response['data'];

        $service_type = $request->service_type;
        $getServiceTypeData = ServiceType::select('id')->where('name', $service_type)->first();
        $serviceTypeId = $getServiceTypeData->id ?? '';
        $getAirports = Airport::with('airportName')->where('status', 'active')->get()->toArray();
        $getAirlines = Airline::with('airlineCodeName')->where('status', 'active')->get()->toArray();
        $serviceType = ServiceType::where('name', 'Flight')->value('id');
        $getSupplier = Suppliers::where('core_service_type_id', $serviceType)->get()->toArray();
        $getAgency = Agency::where('status', 'active')->get()->toArray();

        $getCountry = Country::with('countryCode')->where('status', 'active')->get()->toArray();
        $getCities = City::with('cityCode')->where('status', 'active')->get()->toArray();

        if ($response['status'] == 1 && !empty($response['data'])) {
            return view('admin/markups/flight-markup/editFlight')->with(['header' => $header, 'serviceTypeId' => $serviceTypeId, 'service_type' => $service_type, 'markupsDetail' => $markupsDetail, 'getAirports' => $getAirports, 'getAirlines' => $getAirlines, 'getSupplier' => $getSupplier, 'getAgency' => $getAgency, 'getCountry' => $getCountry, 'getCities' => $getCities]);
        } else {
            return redirect()->to('markups/manage?service_type=' . $service_type)->with('error', $response['message']);
        }
    }

    /**
     * Update the specified markups in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateMarkupData(Request $request, $id)
    {
        // echo "<pre>";print_r($request->all());die;
        $requestData = $request->only([
            'markups_id',
            'ruleName',
            'priority',
            'service_type_id',
            'channel',
            'supplier',
            'originCriteria',
            'originName',
            'destinationCriteria',
            'destinationName',
            'airlines',
            'fromBookingDate',
            'toBookingDate',
            'fromTravelDate',
            'toTravelDate',

            'tripType',
            'paxType',
            'from_price_range',
            'to_price_range',
            'fareType',
            'b2c_markup_type',
            'b2c_markup',
            'b2b_markup_type',
            'b2b_markup',
            'service_b2b_markup_type',
            'service_b2b_markup',
            'service_b2c_markup_type',
            'service_b2c_markup',
            'ancillary_service_markup_type',
            'ancillary_service_markup',
            'commMarkupOn',
            'agency',
            'agencyGroup',
            'service_type',
            'calculationMethod',
            'starCategory'
        ]);

        $rules = [];

        $customMessages = [];

        $niceNames = array();

        $this->validate($request, $rules, $customMessages, $niceNames);
        $response = Markups::updateMarkups($requestData);
        MarkupLog::create([
            'markup_id' => $response['data']['id'],
            'service_type_id' => $response['data']['service_type_id'],
            'action' => 'Markup Updated',
            'created_by' => Auth::id(),
        ]);

        if ($requestData['service_type'] == 'hotel' || $requestData['service_type'] == 'Hotel') {
            $response['data']['service_type_id'] = $requestData['service_type_id'];
        }

        $service_type_id = $response['data']['service_type_id'];
        $getServiceTypeData = ServiceType::select('id', 'name')->where('id', $service_type_id)->first();
        $serviceTypeName = $getServiceTypeData->name ?? '';
        if ($response['status'] == 1 &&  !empty($response['data'])) {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('success', $response['message']);
        } else {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('error', $response['message']);
        }
    }

    /**
     * Remove the specified markups from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    /**
     * Display a listing of the markups.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function getListData(Request $request)
    {
        $serviceTypes = "";
        if (isset($_GET['service_type'])) {
            $serviceTypes = str_replace(' ', '_', $_GET['service_type']);
        }
        $header['title'] = "Markup Rules List";
        $header['heading'] = "Markups Rules List";
        $queryStringConcat = '?';
        if (isset($_GET['per_page'])) {
            $queryStringConcat .= ($queryStringConcat == '') ? '?per_page=' . $_GET['per_page'] : '&per_page=' . $_GET['per_page'];
        }
        if (isset($_GET['page'])) {
            $queryStringConcat .= ($queryStringConcat == '') ? '?page=' . $_GET['page'] : '&page=' . $_GET['page'];
        }

        $appliedFilter = array(
            'per_page' => (request()->input('per_page') != NULL) ? request()->input('per_page') : Setting::where('config_key', 'general|setting|pagePerAdminRecords')->get('value')[0]['value'],
            'order_by' => (request()->input('order_by') != NULL) ? request()->input('order_by') : 'created_at',
            'sorting' => (request()->input('sorting') != NULL) ? request()->input('sorting') : 'desc',
            'origin' => (request()->input('origin') != NULL) ? request()->input('origin') : '',
            'rule_name' => (request()->input('rule_name') != NULL) ? request()->input('rule_name') : '',
            'channel' => (request()->input('channel') != NULL) ? request()->input('channel') : '',
            'originCriteria' => (request()->input('originCriteria') != NULL) ? request()->input('originCriteria') : '',
            'destinationCriteria' => (request()->input('destinationCriteria') != NULL) ? request()->input('destinationCriteria') : '',
            'from_booking_date' => (request()->input('from_booking_date') != NULL) ? request()->input('from_booking_date') : '',
            'to_booking_date' => (request()->input('to_booking_date') != NULL) ? request()->input('to_booking_date') : '',
            'cabinClass' => (request()->input('cabinClass') != NULL) ? request()->input('cabinClass') : '',
            'commMarkupOn' => (request()->input('commMarkupOn') != NULL) ? request()->input('commMarkupOn') : '',
            'service_type' => $_GET['service_type'] ?? '',
        );

        if (request()->input('rule_name') != NULL) {
            $appliedFilter['where'][] = ['rule_name', 'LIKE', '%' .  request()->input('rule_name') . '%'];
        }

        if (request()->input('channel') != NULL) {
            $appliedFilter['whereHas'] = ['channel', '=', request()->input('channel')];
        }
        if (request()->input('originCriteria') != NULL) {
            $appliedFilter['where'][] = ['origin_criteria', '=', request()->input('originCriteria')];
        }
        if (request()->input('destinationCriteria') != NULL) {
            $appliedFilter['where'][] = ['destination_criteria', '=', request()->input('destinationCriteria')];
        }
        if (request()->input('from_booking_date') != NULL) {
            $appliedFilter['where'][] = ['from_booking_date', '=', request()->input('from_booking_date')];
        }
        if (request()->input('to_booking_date') != NULL) {
            $appliedFilter['where'][] = ['to_booking_date', '=', request()->input('to_booking_date')];
        }
        if (request()->input('cabinClass') != NULL) {
            $appliedFilter['where'][] = ['cabin_class', '=', request()->input('cabinClass')];
        }
        if (request()->input('commMarkupOn') != NULL) {
            $appliedFilter['where'][] = ['comm_markup_on', '=', request()->input('commMarkupOn')];
        }
        $coreServiceType = ServiceType::select('id')->where('name', 'LIKE',"%$serviceTypes%")->first();
        $default_markup_id = false;
        if($coreServiceType){
            $default_markup_id = DefaultMarkup::where('service_type_id', $coreServiceType->id)->value('id'); 
        }
        $markupsDataList = Markups::getMarkupsData($appliedFilter);
        $markupsDataCount = Markups::count();
        $markupsData = $markupsDataList['data'];

        $markupsSettingData = MarkupSetting::first();

        if ($markupsData['status'] == 1) {
            return view('admin/markups/flight-markup/index')->with(['header' => $header, 'appliedFilter' => $appliedFilter, 'serviceType' => $serviceTypes, 'markupsDataCount' => $markupsDataCount, 'markupsData' => $markupsData, 'markupSetting' => $markupsSettingData, 'queryStringConcat' => $queryStringConcat, 'i' => (request()->input('page', 1) - 1) * $appliedFilter['per_page'], 'default_markup_id' => $default_markup_id]);
        } else {
            return view('admin/markups/flight-markup/index')->with(['header' => $header, 'appliedFilter' => $appliedFilter, 'serviceType' => $serviceTypes, 'markupsData' => $markupsData, 'markupsDataCount' => $markupsDataCount, 'markupSetting' => $markupsSettingData, 'queryStringConcat' => $queryStringConcat, 'i' => (request()->input('page', 1) - 1) * $appliedFilter['per_page'],'default_markup_id' => $default_markup_id]);
        }
    }

    /**
     * Fetch airport for origin.
     *
     * 
     * @return \Illuminate\Http\Response
     */
    public function fetchOriginList(Request $request)
    {

        $requestData = $request->all();
        $term = $request->input('term');
        $page = $request->input('page');
        $resultsPerPage = 10;
        $offset = ($page - 1) * $resultsPerPage;

        $query = Airport::with('airportName');

        if ($term) {
            $query->whereHas('airportName', function ($query) use ($term) {
                $query->where('airport_name', 'like', '%' . $term . '%');
            });
        }
        $query->offset($offset)->limit($resultsPerPage);
        $query->orderBy('iata_code', 'asc');
        $airports = $query->get();

        $data = [];
        foreach ($airports as $airport) {
            $airname = [];
            if (!empty($airport['airportName'])) {
                foreach ($airport['airportName'] as $airport_name) {
                    $airname[] = $airport_name['airport_name'] . ' ';
                }
            }
            $data[] = ['id' => $airport['id'], 'airname' => $airname, 'first_page' => $page];
        }

        return response()->json($data);
    }

    /**
     * Fetch airline code.
     *
     * 
     * @return \Illuminate\Http\Response
     */
    public function fetchAirlines(Request $request)
    {
        $term = $request->input('term');
        $page = $request->input('page');

        $resultsPerPage = 10;
        $offset = ($page - 1) * $resultsPerPage;
        $query = Airline::with('airlineCodeName');

        if ($term) {
            $query->whereHas('airlineCodeName', function ($query) use ($term) {
                $query->where('airline_name', 'like', '%' . $term . '%');
            });
        }
        $query->offset($offset)->limit($resultsPerPage);
        $query->orderBy('airline_code', 'asc');
        $airlines = $query->get();

        $data = [];
        foreach ($airlines as $airline) {
            $airname = [];
            if (!empty($airline['airlineCodeName'])) {
                foreach ($airline['airlineCodeName'] as $airline_name) {
                    $airname[] = $airline_name['airline_name'] . ' ';
                }
            }
            $data[] = ['id' => $airline['id'], 'airname' => $airname];
        }

        return response()->json($data);
    }

    /**
     * Fetch supplier code.
     *
     * 
     * @return \Illuminate\Http\Response
     */
    public function fetchSupplier(Request $request)
    {
        $requestData = $request->all();
        $page = $request->input('page');
        $resultsPerPage = 10;
        $offset = ($page - 1) * $resultsPerPage;
        $serviceType = ServiceType::whereId($requestData['service_id'])->get('id')[0]['id'];
        $getSupplier = Suppliers::where('core_service_type_id', $serviceType);
        $getSupplier->offset($offset) // For par page 10 records
            ->limit($resultsPerPage);
        if ($request->input('q') != "") {
            $getSupplier->where('name', 'LIKE', '%' . $request->input('q') . '%');
        }

        $data = $getSupplier->get()->toArray();
        return response()->json($data);
    }

    /**
     *  Store a newly created markups in storage.
     *
     * 
     * @return \Illuminate\Http\Response
     */
    public function addData(Request $request)
    {

        $service_type = $request->service_type;
        $header['title'] = $service_type . " - Add";
        $header['heading'] = $service_type . " - Add";

        $getServiceTypeData = ServiceType::select('id')->where('name', $service_type)->first();
        $serviceTypeId = $getServiceTypeData->id ?? '';
        $getAgency = Agency::where('status', 'active')->get()->toArray();

        return view('admin/markups/flight-markup/add')->with(['header' => $header, 'serviceTypeId' => $serviceTypeId, 'getAgency' => $getAgency]);
    }

    /**
     * Remove the specified markups from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function deleteMarkups(Request $request)
    {
        if (!hasPermission('Markups', 'delete')) {
            return view('admin/401');
        }
        $url = URL::previous();
        $markupsIDs = explode(',', $request->input('delete_markups_id'));

        $message = "";
        foreach ($markupsIDs as $delete_markups_id) {
            $response = Markups::deleteMarkups($delete_markups_id);
            $message .= $response['message'] . '</br>';
        }

        if ($response['status'] == 1) {

            return redirect()->to($url)->with('success', $message);
        } else {
            return redirect()->to($url)->with('error', $response['message']);
        }
    }



    public function addDefaultMarkupData(Request $request)
    {

        $service_type = $request->service_type;
        $header['title'] = $service_type . " - Add";
        $header['heading'] = $service_type . " - Add";

        $getServiceTypeData = ServiceType::select('id')->where('name', $service_type)->first();
        $serviceTypeId = $getServiceTypeData->id ?? '';
        $getAgency = Agency::where('status', 'active')->get()->toArray();

        return view('admin/markups/addDefaultMarkup')->with(['header' => $header, 'serviceTypes' => $service_type, 'serviceTypeId' => $serviceTypeId, 'getAgency' => $getAgency]);
    }
    public function storeDefaultMarkup(Request $request)
    {
        $requestData = $request->all();

        $rules = [
            'supplier' => 'required',
            'b2c_markup_type' => 'required',
            'b2c_markup' => 'required',
            'calculationMethod' => 'required',
        ];

        $customMessages = [];

        $niceNames = array();

        $this->validate($request, $rules, $customMessages, $niceNames);

        $response = DefaultMarkup::createDefaultFlightMarkups($requestData);

        $service_type_id = $response['data']['service_type_id'];
        $getServiceTypeData = ServiceType::select('id', 'name')->where('id', $service_type_id)->first();
        $serviceTypeName = $getServiceTypeData->name ?? '';
        if ($response['status'] == 1 && !empty($response['data'])) {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('success', $response['message']);
        } else {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('error', $response['message']);
        }
    }



    public function editDefaultMarkup(Request $request, $id)
    {
        $header['title'] = $request->service_type ? ucfirst($request->service_type) ." - Edit" : 'Default Markup Edit';
        $header['heading'] = $request->service_type ? ucfirst($request->service_type) ." - Edit" : 'Default Markup Edit';;

        $checkServiceType = ServiceType::where('name', $request->service_type)->first();
        $service_type_id = $checkServiceType->id ?? '';

        $filter = array(
            'id' => $id,
            'service_type_id' => $service_type_id,
        );

        $response = DefaultMarkup::getDefaultMarkupsData($filter);
        $markupsDetail = $response['data'];
        $service_type = $request->service_type;
        $getServiceTypeData = ServiceType::select('id')->where('name', $service_type)->first();
        $serviceTypeId = $getServiceTypeData->id ?? '';
        $serviceType = ServiceType::whereId($service_type_id)->value('id');
        $getSupplier = Suppliers::where('core_service_type_id', $serviceType)->get()->toArray();
        if ($response['status'] == 1 && !empty($response['data'])) {
            return view('admin/markups/editDefaultMarkup')->with(['header' => $header, 'serviceTypeId' => $serviceTypeId, 'service_type' => $service_type, 'markupsDetail' => $markupsDetail, 'getSupplier' => $getSupplier]);
        } else {
            $response = DefaultMarkup::where('service_type_id', 2)->first();
            $markupsDetail = $response;
            $service_type = $request->service_type;
            $getServiceTypeData = ServiceType::select('id')->where('name', $service_type)->first();
            $serviceTypeId = $getServiceTypeData->id ?? '';
            $serviceType = ServiceType::where('name', 'Hotel')->value('id');
            $getSupplier = Suppliers::where('core_service_type_id', $serviceType)->get()->toArray();

            return view('admin/markups/addDefaultMarkup')->with(['header' => $header, 'serviceTypeId' => $serviceTypeId, 'service_type' => $service_type, 'markupsDetail' => $markupsDetail, 'getSupplier' => $getSupplier]);

            // return redirect()->to('markups/flight-markup/manage?service_type=' . $service_type)->with('error', $response['message']);
        }
    }

    public function updateDefaultMarkup(Request $request, $id)
    {
        $requestData = $request->only([
            'default_markups_id',
            'service_type_id',
            'supplier',
            'b2c_markup_type',
            'b2c_markup',
            'b2b_markup_type',
            'b2b_markup',
            'calculationMethod',
        ]);
       
        $rules = [];

        $customMessages = [];

        $niceNames = array();

        $this->validate($request, $rules, $customMessages, $niceNames);
        $response = DefaultMarkup::updateDefaultMarkups($requestData);
   
        $service_type_id = $response['data']->service_type_id;
        $getServiceTypeData = ServiceType::select('id', 'name')->where('id', $service_type_id)->first();
        $serviceTypeName = $getServiceTypeData->name ?? '';
        if ($response['status'] == 1 &&  !empty($response['data'])) {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('success', $response['message']);
        } else {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('error', $response['message']);
        }
    }

    /**
     * Remove the specified markups from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function deleteDefaultMarkups(Request $request)
    {
        if (!hasPermission('Markups', 'delete')) {
            return view('admin/401');
        }
        $url = URL::previous();
        $markupsIDs = explode(',', $request->input('delete_markups_id'));

        $message = "";
        foreach ($markupsIDs as $delete_markups_id) {
            $response = DefaultMarkup::deleteDefaultMarkups($delete_markups_id);
            $message .= $response['message'] . '</br>';
        }

        if ($response['status'] == 1) {
            return redirect()->to($url)->with('success', $message);
        } else {
            return redirect()->to($url)->with('error', $response['message']);
        }
    }





    public function storeHotel(Request $request)
    {

        $requestData = $request->only([

            'priority',
            'service_type_id',
            'channel',
            'supplier',

            'destinationCriteria',
            'destinationName',

            'fromBookingDate',
            'toBookingDate',
            'fromCheckInDate',
            'toCheckInDate',


            'fromBaseFare',
            'toBaseFare',
            'fareType',
            'b2c_markup_type',
            'b2c_markup',
            'b2b_markup_type',
            'b2b_markup',
            'service_b2b_markup_type',
            'service_b2b_markup',
            'service_b2c_markup_type',
            'service_b2c_markup',

            'agency',
            'agencyGroup',
            'calculationMethod',
            'starCategory'
        ]);

        $rules = [

            'priority' => 'required',


            'destinationCriteria' => 'required',
            'destinationName' => 'required',

            'fromBookingDate' => 'required',
            'toBookingDate' => 'required',





            'fromBaseFare' => 'required',
            'toBaseFare' => 'required',
            'fareType' => 'required',
            'b2c_markup_type' => 'required',
            'b2c_markup' => 'required',
            'service_b2c_markup_type' => 'required',
            'service_b2c_markup' => 'required',
            'calculationMethod' => 'required',
            'starCategory' => 'required',


        ];

        $customMessages = [];

        $niceNames = array();

        $this->validate($request, $rules, $customMessages, $niceNames);

        $response = Markups::createMarkupsHotel($requestData);
        MarkupLog::create([
            'markup_id' => $response['data']['id'],
            'service_type_id' => $response['data']['service_type_id'],
            'action' => 'Markup Created',
            'created_by' => Auth::id(),
        ]);

        $service_type_id = 2;
        $getServiceTypeData = ServiceType::select('id', 'name')->where('id', $service_type_id)->first();
        $serviceTypeName = $getServiceTypeData->name ?? '';
        if ($response['status'] == 1 && !empty($response['data'])) {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('success', $response['message']);
        } else {
            return redirect()->to('markups/manage?service_type=' . $serviceTypeName)->with('error', $response['message']);
        }
    }

    public function showMarkupLogs(Request $request)
    {
        $header['title'] = "Markup Logs";
        $header['heading'] = "Markup Logs";

        $service_type = $request->service_type;

        return view('admin/markups/logs')->with(['header' => $header, 'serviceType' => $service_type]);
    }
    public function getMarkupLogs(Request $request)
    {
        return DataTables::of(MarkupLog::with('user')->select('*')->orderByDesc('created_at'))
            ->addIndexColumn()
            ->editColumn('service_type_id', function ($row) {
                return $row->service_type_id == 1 ? 'Flight' : ($row->service_type_id == 2 ? 'Hotel' : 'N/A');
            })
            ->editColumn('created_at', function ($row) {
                return optional($row->created_at)->format('d M Y, h:i A');
            })
            ->editColumn('updated_at', function ($row) {
                return optional($row->updated_at)->format('d M Y, h:i A');
            })
            ->editColumn('created_by', function ($row) {
                return $row?->user?->name;
            })
            ->addColumn('action_button', function ($row) {
                if (Markups::where('id',$row->markup_id)->exists() === false) {
                    return '';
                }

                $serviceType = $row->service_type_id == 1 ? 'Flight' : 'Hotel';

               return '<a href="' . route('flight-markups.edit', $row->markup_id) . '?service_type=' . $serviceType . '" class="btn btn-primary btn-sm" title="Edit">View</a>';
            })
            ->rawColumns(['action_button'])
            ->make(true);
    }
}
