<?php

namespace App\Http\Controllers\Admin\Geography;

use App\Enums\Locale;
use App\Models\City;
use App\Models\Country;
use App\Models\Setting;
use App\Models\CityI18n;
use App\Models\CityImage;
use App\Models\CountryI18ns;
use Illuminate\Http\Request;
use App\Models\TopDestination;
use App\Models\TopDestinationI18s;
use Illuminate\Support\Facades\URL;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class TopDestinationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if (!hasPermission('CITY', 'read')) {
            return view('admin/401');
        }
    
        // Set header information
        $header = [
            'title'   => "Top Destinations",
            'heading' => "Top Destinations"
        ];
    
        // Build query string for pagination
        $queryStringConcat = http_build_query(request()->only(['per_page', 'page']));
    
        // Define filters with default values
        $perPage = request()->input('per_page', Setting::where('config_key', 'general|setting|pagePerAdminRecords')->value('value'));
        $orderBy = request()->input('order_by', 'created_at');
        $sorting = request()->input('sorting', 'desc');
    
        $filter = [
            'per_page'     => $perPage,
            'order_by'     => $orderBy,
            'sorting'      => $sorting,
            'city_name'    => request()->input('city_name', ''),
            'city_code'    => request()->input('city_code', ''),
            'country_code' => request()->input('country_code', ''),
            'status'       => request()->input('status', ''),
        ];
    
        // Query the TopDestination and related TopDestinationI18s data with filters
        $query = TopDestination::query();
    
        if (!empty($filter['city_code'])) {
            $query->whereHas('i18ns', function ($q) use ($filter) {
                $q->where('city_code', 'like', '%' . $filter['city_code'] . '%');
            });
        }
        if (!empty($filter['city_name'])) {
            $query->whereHas('i18ns', function ($q) use ($filter) {
                $q->where('city_name', 'like', '%' . $filter['city_name'] . '%');
            });
        }
        if (!empty($filter['country_code'])) {
            $query->where('country_code', $filter['country_code']);
        }
        if (!empty($filter['status'])) {
            $query->where('status', $filter['status']);
        }
    
        // Apply sorting and pagination
        $cityData = $query->orderBy($orderBy, $sorting)->paginate($perPage);
        $cityDataCount = TopDestination::count();
        $getCountries = Country::with('countryCode')->where('status','active')->get();
    
        return view('admin/geography/top_destination/index', [
            'header'           => $header,
            'cityData'         => $cityData,
            'queryStringConcat' => '?' . $queryStringConcat,
            'appliedFilter'    => $filter,
            'i'                => ($cityData->currentPage() - 1) * $perPage,
            'cityDataCount'    => $cityDataCount,
            'getCountries'     => $getCountries,
        ]);
    }
    


    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        if (!hasPermission('CITIES', 'create')) {
            return view('admin/401');
        }
        $header['title'] = @trans('Cities - Add');
        $header['heading'] = @trans('city.moduleHeading');
        $cities = City::get()->toArray();
        $countries = Country::get()->toArray();

        return view('admin/geography/top_destination/add')->with(['header' => $header, 'cities' => $cities, 'countries' => $countries]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
{
    $validator = Validator::make($request->all(), [
        'country_code'   => 'required',
        'city'           => 'required|array',
        'city_thumbnail' => 'required|array',  // Ensure this is an array
        'status'         => 'required'
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    // Check if the city already exists
    $checkExist = TopDestination::where('country_code', $request->country_code)->exists();  
    if ($checkExist) {
        return redirect()->back()->withErrors(['message' => 'The City already Exist']);
    }

    // Fetch country details
    $country = Country::where('iso_code', $request->country_code)->firstOrFail();
    $country_names = CountryI18ns::where('country_id', $country->id)
        ->whereIn('language_code', [Locale::English->value, Locale::Arabic->value])
        ->pluck('country_name', 'language_code');

    // Create the city details record
    $city_details = TopDestination::create([
        'country_code'        => $request->country_code,
        'country_name'        => $country_names[Locale::English->value] ?? '',
        'country_name_arabic' => $country_names[Locale::Arabic->value] ?? '',
        'status'              => $request->status,
    ]);

    // Handle file uploads for city thumbnails
    $thumbnails = $request->file('city_thumbnail') ?? []; // Ensure it's always an array

    // Loop through the cities and handle file uploads for each one
    foreach ($request->city as $index => $cityCode) {
        // Fetch city details
        $city = City::where('iso_code', $cityCode)->firstOrFail();
        $city_names = CityI18n::where('city_id', $city->id)
            ->whereIn('language_code', [Locale::English->value, Locale::Arabic->value])
            ->pluck('city_name', 'language_code');

        // Handle thumbnail for the current city
        $cityImagePath = null;

        // Check if a file was uploaded for the current city and upload it
        if (isset($thumbnails[$index])) {
            $cityImagePath = $thumbnails[$index]->store('cities_thumbnail', 'public');
        }

        // Store city-related information in the `TopDestinationI18s` table
        TopDestinationI18s::create([
            'top_destination_id' => $city_details->id,
            'city_code'          => $cityCode,
            'city_name'          => $city_names[Locale::English->value] ?? '',
            'city_name_arabic'   => $city_names[Locale::Arabic->value] ?? '',
            'thumbnail'          => $cityImagePath, // Only set if a file is uploaded
        ]);
    }

    // Redirect with success message
    if ($city_details) {
        return redirect()->route('city-images.index')->with('success', 'Thumbnail Created Successfully for the Selected City');
    } else {
        return redirect()->back()->with('error', 'Error Creating City Thumbnail');
    }
}

    
    


    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        if (!hasPermission('CITIES', 'update')) {
            return view('admin/401');
        }
        $header['title'] = 'Top Destination - Edit';
        $header['method'] = 'Edit';
        $filter = array(
            'id' => $id,
        );

        $cityDetail = TopDestination::with('i18ns')->findOrFail($id);


        // print_r($airportDetail->iata_code);exit;

        if ($cityDetail) {
            return view('admin/geography/top_destination/update')->with(['header' => $header, 'cityDetail' => $cityDetail]);
        } else {
            return redirect()->route('cities.index')->with('error', 'Something Went Wrong');
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
{
    $validator = Validator::make($request->all(), [
        'country_code'   => 'required',
        'city'           => 'required|array',
        'city_thumbnail' => 'nullable|array',
        'status'         => 'required',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    $city_details = TopDestination::findOrFail($id);
    $checkExist = TopDestination::where('country_code', $request->country_code)
        ->where('id', '!=', $id)
        ->exists();

    if ($checkExist) {
        return redirect()->back()->withErrors(['message' => 'The City already Exists']);
    }

    // Update top destination details
    $city_details->update([
        'country_code' => $request->country_code,
        'status'       => $request->status,
    ]);

    // **Handle Removed Cities**
    if ($request->filled('removed_cities')) {
        $removedCityCodes = explode(',', $request->removed_cities);
        
        // Delete cities from `TopDestinationI18s`
        TopDestinationI18s::where('top_destination_id', $city_details->id)
            ->whereIn('city_code', $removedCityCodes)
            ->delete();
    }

    // Handle file uploads and city updates
    $thumbnails = $request->file('city_thumbnail') ?? [];

    foreach ($request->city as $index => $cityCode) {
        $city = City::where('iso_code', $cityCode)->firstOrFail();
        $city_names = CityI18n::where('city_id', $city->id)
            ->whereIn('language_code', [Locale::English->value, Locale::Arabic->value])
            ->pluck('city_name', 'language_code');

        $cityImagePath = null;

        if (isset($thumbnails[$index])) {
            $cityImagePath = $thumbnails[$index]->store('cities_thumbnail', 'public');
        }

        $cityImage = TopDestinationI18s::where('top_destination_id', $city_details->id)
            ->where('city_code', $cityCode)
            ->first();

        if ($cityImage) {
            $cityImage->update([
                'city_name'        => $city_names[Locale::English->value] ?? '',
                'city_name_arabic' => $city_names[Locale::Arabic->value] ?? '',
                'thumbnail'        => $cityImagePath ?? $cityImage->thumbnail,
            ]);
        } else {
            TopDestinationI18s::create([
                'top_destination_id' => $city_details->id,
                'city_code'          => $cityCode,
                'city_name'          => $city_names[Locale::English->value] ?? '',
                'city_name_arabic'   => $city_names[Locale::Arabic->value] ?? '',
                'thumbnail'          => $cityImagePath,
            ]);
        }
    }

    return redirect()->route('city-images.index')->with('success', 'City Thumbnail Updated Successfully');
}



    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }



    public function fetchCitiesByCountry(Request $request)
    {
        $countryCode = $request->input('country_code');
        $term = $request->input('term', '');


        $cities = City::with(['cityCode' => function ($q) use ($term) {
            if ($term) {
                $q->where('city_name', 'like', '%' . $term . '%');
            }
        }])
            ->where('country_code', $countryCode)
            ->get();

        $data = [];
        foreach ($cities as $city) {
            $cityName = [];


            if ($city->cityCode->isNotEmpty()) {
                foreach ($city->cityCode as $cityI18n) {
                    $cityName[] = $cityI18n->city_name;
                }
            }


            $data[] = [
                'iso_code' => $city->iso_code,
                'city_name' => implode(', ', $cityName),
                'country_code' => $city->country_code,
            ];
        }

        return response()->json($data);
    }


    public function  deleteCityImage($id)
    {

        $cityDetail = TopDestination::find($id);
        $cityDetaili18ns= TopDestinationI18s::where('top_destination_id', $id);



        if (!$cityDetail) {
            return redirect()->route('city-images.index')->with('error', 'City Thumbnail not found');
        }

        $url = URL::previous();
        if ($cityDetail->delete() && $cityDetaili18ns->delete()) {

            return redirect()->to($url)->with('success', 'City Thumbnail deleted successfully');
        } else {
            return redirect()->to($url)->with('error', 'Error deleting Multimedia');
        }
    }
}
