<?php

/**
 * @package     Dashboard
 * @subpackage  Dashboard
 * @Author      Amar Technolabs Pvt. ltd(info@amarinfotech.com)
 * @Copyright(C) 2023 [NAME OF THE ORGANISATION THAT ON BEHALF OF THE CODE WE ARE WORKING].
 * @Version 1.0.0
 * module of the Dashboard.
 */

namespace App\Http\Controllers\Admin;

use DateTime;
use ZipArchive;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Agency;
use App\Models\Coupon;
use App\Models\Payment;
use App\Models\Setting;
use App\Models\Bookings;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Traits\ActiveLog;
use App\Traits\HotelBeds;
use Illuminate\Http\File;
use App\Models\UserOtpLog;
use App\Traits\SmsService;
use App\Models\ServiceType;
use App\Traits\EmailService;
use Illuminate\Http\Request;
use App\Models\TrendingHotel;
use App\Traits\CommonService;
use App\Traits\TraacsService;
use App\Models\TrendingFlight;
use App\Traits\AmadeusService;
use App\Traits\BookingService;
use App\Traits\PaymentService;
use App\Models\SeoTrendingHotel;
use App\Models\SeoTrendingFlight;
use Illuminate\Support\Facades\DB;
use App\Models\CustomerActivityLog;




use Illuminate\Support\Facades\URL;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Facades\DataTables;

class DashboardController extends Controller
{
    /**
     * Display a listing of the dashboard.
     *
     * @return \Illuminate\Http\Response
     */

    public function index(Request $request)
    {

        $supplierDataList = Suppliers::get()->toArray();
        $bookDate = now()->format('Y-m-d');
        $totalCustomers = Customer::count();
        $totalCoupons = Coupon::count();
        $totalBookings = Bookings::count();
        $totalUsers = User::count();


        $data['totalCustomers'] = $totalCustomers;
        $data['totalCoupons'] = $totalCoupons;
        $data['totalBookings'] = $totalBookings;
        $data['totalUsers'] = $totalUsers;


        $data['title'] = @trans('dashboard.title');
        $data['heading'] = @trans('dashboard.moduleHeading');
        $data['supplierDataList'] = $supplierDataList;
        $data['getServiceType'] = ServiceType::get()->toArray();
        $data['getAgency'] = Agency::get()->toArray();


        //  bar chart data for Total Customers
        $customers = Customer::select(
            DB::raw("COUNT(*) as count"),
            DB::raw("MONTHNAME(created_at) as month_name")
        )
            ->whereYear('created_at', date('Y'))
            ->groupBy(DB::raw("MONTH(created_at)"))
            ->orderBy(DB::raw("MONTH(created_at)"))
            ->pluck('count', 'month_name');


        $data['customerLabels'] = $customers->keys();
        $data['customerData'] = $customers->values();

        //  bar chart data for Total Bookings

        $bookings = Bookings::select(
            DB::raw("COUNT(*) as count"),
            DB::raw("MONTHNAME(created_at) as month_name")
        )
            ->whereYear('created_at', date('Y'))
            ->groupBy(DB::raw("MONTH(created_at)"))
            ->orderBy(DB::raw("MONTH(created_at)"))
            ->pluck('count', 'month_name');


        $data['bookingLabels'] = $bookings->keys();
        $data['bookingData'] = $bookings->values();


        //  bar chart data for Total Coupons

        $coupons = Coupon::select(
            DB::raw("COUNT(*) as count"),
            DB::raw("MONTHNAME(created_at) as month_name")
        )
            ->whereYear('created_at', date('Y'))
            ->groupBy(DB::raw("MONTH(created_at)"))
            ->orderBy(DB::raw("MONTH(created_at)"))
            ->pluck('count', 'month_name');


        $data['couponLabels'] = $coupons->keys();
        $data['couponData'] = $coupons->values();


        // Pie chart
        $data['totalCustomers'] = Customer::count();
        $bookingsQuery = Bookings::whereDate('created_at', now()->toDateString());

        $data['totalBookings'] = (clone $bookingsQuery)->count();
        $data['totalFlightBookings'] = (clone $bookingsQuery)->where('booking_type', 'Flight')->count();
        $data['totalHotelBookings'] = (clone $bookingsQuery)->where('booking_type', 'Hotel')->count();
        $data['confirmedBookings'] = (clone $bookingsQuery)->where('booking_status', 'confirmed')->count();
        $data['failedBookings'] = (clone $bookingsQuery)->where('booking_status', 'failed')->count();

        $data['flightConfirmedBookings'] = (clone $bookingsQuery)->where('booking_status', 'confirmed')->where('booking_type', 'Flight')->count();
        $data['hotelConfirmedBookings'] = (clone $bookingsQuery)->where('booking_status', 'confirmed')->where('booking_type', 'Hotel')->count();

        $data['flightFailedBookings'] = (clone $bookingsQuery)->where('booking_status', 'failed')->where('booking_type', 'Flight')->count();
        $data['hotelFailedBookings'] = (clone $bookingsQuery)->where('booking_status', 'failed')->where('booking_type', 'Hotel')->count();

        $data['totalCoupons'] = Coupon::count();

        $data['bookings'] = [
            ['name' => 'Hotel', 'value' => Bookings::where('booking_type', 'Hotel')->count()],
            ['name' => 'Flight', 'value' => Bookings::where('booking_type', 'Flight')->count()],
            ['name' => 'Confirmed', 'value' => Bookings::where('booking_status', 'confirmed')->count()],
            ['name' => 'Failed', 'value' => Bookings::where('booking_status', 'failed')->count()],
            ['name' => 'Processing', 'value' => Bookings::where('booking_status', 'processing')->count()],
        ];


        $data['customers'] = Customer::orderBy('created_at', 'desc')->limit(12)->get();
        $data['newCustomerCount'] = Customer::where('created_at', '>', date('Y-m-d', strtotime('-1 week')))->count();
        $data['recentBookings'] = Bookings::where('created_at', '>=', Carbon::now()->today())->limit(10)->get();

        $data['trendingFlights'] = TrendingFlight::count();
        $data['trendingHotels'] = TrendingHotel::count();
        $data['seoTrendingFlights'] = SeoTrendingFlight::count();
        $data['seoTrendingHotels'] = SeoTrendingHotel::count();

        // Monthly Recap ChartData
        $data['customerLabels'] = $customers->keys();
        $data['customerData'] = $customers->values();

        $data['bookingLabels'] = $bookings->keys();
        $data['bookingData'] = $bookings->values();

        $data['couponLabels'] = $coupons->keys();
        $data['couponData'] = $coupons->values();

        // Goal Completion
        $data['goals'] = [
            [
                'title' => 'Total Customers',
                'current' => $totalCustomers,
                'target' => 1000,
                'color' => 'primary',
            ],
            [
                'title' => 'Total Bookings',
                'current' => $totalBookings,
                'target' => 10000,
                'color' => 'success',
            ],
            [
                'title' => 'Total Coupons',
                'current' => $totalCoupons,
                'target' => 200,
                'color' => 'warning',
            ],
            [
                'title' => 'Total Users',
                'current' => $totalUsers,
                'target' => 100,
                'color' => 'danger',
            ],
        ];
        $confirmedCount = Bookings::where('booking_status', 'confirmed')->count();
        $failedCount = Bookings::where('booking_status', 'failed')->count();
        $processingCount = Bookings::where('booking_status', 'processing')->count();

        // Total number of bookings considered as goal completions
        $totalBookingCount = $confirmedCount + $failedCount + $processingCount;


        $previousGoalCount = 1000;

        // Calculate percentage change
        if ($previousGoalCount > 0) {
            $goalCompletionChange = (($totalBookingCount - $previousGoalCount) / $previousGoalCount) * 100;
        } else {
            $goalCompletionChange = $totalBookingCount > 0 ? 100 : 0;
        }

        $data['dashboardStats'] = [
            [
                'label' => 'TOTAL REVENUE',
                'value' => Bookings::where('booking_status', 'confirmed')->sum('total'),
                'change' => 17,
                'direction' => 'up',
                'color' => 'success',
                'is_currency' => true,
            ],
            [
                'label' => 'TOTAL FAILED BOOKINGS',
                'value' => Bookings::where('booking_status', 'failed')->sum('total'),
                'change' => 0,
                'direction' => 'down',
                'color' => 'danger',
                'is_currency' => true,
            ],
            [
                'label' => 'TOTAL PROCESSED BOOKINGS',
                'value' => Bookings::where('booking_status', 'processing')->sum('total'),
                'change' => 20,
                'direction' => 'left',
                'color' => 'info',
                'is_currency' => true,
            ],
            [
                'label' => 'GOAL COMPLETIONS',
                'value' => $totalBookingCount,
                'change' => round($goalCompletionChange),
                'direction' => $goalCompletionChange >= 0 ? 'up' : 'down',
                'color' => $goalCompletionChange >= 0 ? 'success' : 'danger',
                'is_currency' => false,
            ],
        ];

        // User Login Attemts
        $data['userLogin'] = UserOtpLog::where('purpose', 'Login Successfull')->orderBy('created_at', 'desc')->limit(10)->get();




        return view("admin/dashboard/dashboard", $data);
    }

    /**
     * Get record.
     *
     * @return \Illuminate\Http\Response
     */
    public function getRecord(Request $request)
    {
        $requetData = $request->all();
    }

    /**
     * Get duration.
     *
     * @return \Illuminate\Http\Response
     */
    public function getDuration(Request $request)
    {
        if ($request->selected_day == 'today') {
            $fromDate = date('Y-m-d');
            $toDate = date('Y-m-d');
        } else if ($request->selected_day == 'yesterday') {
            $today = new DateTime();
            $yesterday = $today->modify('-1 day');
            $yesterdayDate = $yesterday->format('Y-m-d');
            $fromDate = $yesterdayDate;
            $toDate = $yesterdayDate;
        } else if ($request->selected_day == 'current_month') {
            $startOfMonth = new DateTime('first day of this month');
            $endOfMonth = new DateTime('last day of this month');
            $fromDate = $startOfMonth->format('Y-m-d');
            $toDate = $endOfMonth->format('Y-m-d');
        } else if ($request->selected_day == 'last_month') {
            // Get the current date
            $now = new DateTime();
            $startOfLastMonth = (new DateTime())->modify('first day of last month');
            $endOfLastMonth = (new DateTime())->modify('last day of last month');

            $fromDate = $startOfLastMonth->format('Y-m-d');
            $toDate = $endOfLastMonth->format('Y-m-d');
        } else if ($request->selected_day == 'last_six_month') {
            $now = new DateTime();
            $startOfLastSixMonths = (new DateTime())->modify('-6 months')->modify('first day of this month');
            $endOfLastSixMonths = (new DateTime())->modify('-1 month')->modify('last day of this month');
            $fromDate = $startOfLastSixMonths->format('Y-m-d');
            $toDate = $endOfLastSixMonths->format('Y-m-d');
        } else if ($request->selected_day == 'this_year') {
            $year = date('Y');
            $startOfYear = new DateTime("$year-01-01");
            $endOfYear = new DateTime("$year-12-31");
            $fromDate = $startOfYear->format('Y-m-d');
            $toDate = $endOfYear->format('Y-m-d');
        }
        $totalOrderCount = thousandsCurrencyFormat(Order::whereBetween(DB::raw('DATE(created_at)'), [$fromDate, $toDate])->count());
        $totalSaleGet = Order::whereBetWeen(DB::raw('DATE(created_at)'), [$fromDate, $toDate])->sum('total_amount');
        $totalProduct = Product::whereBetween(DB::raw('DATE(created_at)'), [$fromDate, $toDate])->count();
        $totalUser = User::whereBetween(DB::raw('DATE(created_at)'), [$fromDate, $toDate])->count();
        $data = [];
        $totalSaleNum = number_format($totalSaleGet, 2, '.', '');
        $totalSale = "Rs " . $totalSaleNum;

        $data = [
            'total_order' => $totalOrderCount,
            'total_sale' => $totalSale,
            'total_product' => $totalProduct,
            'total_user' => $totalUser
        ];
        // return json_encode($data);
        return response()->json($data);
    }

    public function getActiveCustomers(Request $request)
    {
        $fiveMinutesAgo = Carbon::now()->subMinutes(5);

        $query = CustomerActivityLog::select(['customer_id', 'device_id', 'browser_name', 'request_url', 'created_at'])
            ->with('customer:id,first_name,last_name,email')
            ->where('created_at', '>=', $fiveMinutesAgo)
            ->orderByDesc('created_at')
            ->limit(100)
            ->get();

        return DataTables::of($query)
            ->addIndexColumn()
            ->editColumn('name', function ($customer) {
                $firstName = optional($customer->customer)->first_name ?? 'Guest';
                $lastName = optional($customer->customer)->last_name ?? '';
                if ($firstName == 'Guest') {
                    return '<p style="font-size:12px;"">' .
                        $firstName . ' ' . $lastName . '</p>';
                }
                return '<a style="font-size:12px;" href="' . route('customers.show', $customer->customer_id) . '">' .
                    $firstName . ' ' . $lastName . '</a>';
            })
            ->editColumn('email', function ($customer) {
                return '<a style="font-size:12px; color:black" href="mailto:' . optional($customer->customer)->email . '">' .
                    optional($customer->customer)->email . '</a>';
            })
            ->editColumn('created_at', function ($customer) {
                return Carbon::parse($customer->created_at)->format('Y-m-d H:i:s');
            })
            ->rawColumns(['name', 'email', 'created_at'])
            ->make(true);
    }


    public function getFailedBookings(Request $request)
    {
        $failedBookings = Bookings::where('booking_status', 'failed')
            ->orderBy('created_at', 'desc');

        return DataTables::of($failedBookings)
            ->addIndexColumn()
            ->editColumn('booking_ref', function ($bookings) {
                return '<a style="font-size:12px;" href="' . route('booking.edit', $bookings->id) . '">
                        <i class="fa fa-booking me-1 text-primary"></i>' .
                    $bookings->booking_ref .
                    '</a>';
            })
            ->editColumn('booking_status', function ($bookings) {
                return '<button style="font-size:12px" class="btn btn-sm btn-danger" style="font-size:14px; color:white"' . '">
                       ' .
                    ucwords($bookings->booking_status) .
                    '</button>';
            })

            ->editColumn('created_at', function ($bookings) {
                return '<p style="font-size:12px">' .
                    ($bookings->created_at ? with(new \DateTime($bookings->created_at))->format('Y-m-d H:i:s') : '') .
                    '</p>';
            })
            ->rawColumns(['booking_ref', 'booking_status', 'created_at'])
            ->make(true);
    }
    public function getDashboardFilteredData(Request $request)
    {
        $query = Bookings::query();

        if ($request->filter) {
            switch ($request->filter) {
                case 'custom':
                    if ($request->from_date && $request->to_date) {
                        $fromDate = Carbon::parse($request->from_date)->startOfDay();
                        $toDate = Carbon::parse($request->to_date)->endOfDay();
                        if ($fromDate > $toDate) {
                            return response()->json(['error' => 'Invalid date range'], 400);
                        }
                        $query->whereBetween('created_at', [$fromDate, $toDate]);
                    }
                    break;
            }
        }

        $data['totalBookings'] = $query->count();
        $data['totalFlightBookings'] = (clone $query)->where('booking_type', 'Flight')->count();
        $data['totalHotelBookings'] = (clone $query)->where('booking_type', 'Hotel')->count();
        $data['totalConfirmedBookings'] = (clone $query)->where('booking_status', 'confirmed')->count();
        $data['totalFailedBookings'] = (clone $query)->where('booking_status', 'failed')->count();
        $data['flightConfirmedBookings'] = (clone $query)->where('booking_type', 'Flight')->where('booking_status', 'confirmed')->count();
        $data['flightFailedBookings'] = (clone $query)->where('booking_type', 'Flight')->where('booking_status', 'failed')->count();
        $data['hotelConfirmedBookings'] = (clone $query)->where('booking_type', 'Hotel')->where('booking_status', 'confirmed')->count();
        $data['hotelFailedBookings'] = (clone $query)->where('booking_type', 'Hotel')->where('booking_status', 'failed')->count();

        return response()->json($data);
    }
}
