<?php

namespace App\Http\Controllers\Admin\Auth;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\CustomerLoginToken;
use App\Models\LoginAsCustomerLog;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Yajra\DataTables\Facades\DataTables;

class AdminLoginAsCustomerController extends Controller
{
    public function generateLoginToken(Request $request, $customerId)
    {
        if (!auth()->check()) {
            abort(403);
        }

        $admin = auth()->user();
        $rawToken = Str::random(60);
        $hashedToken = hash('sha256', $rawToken);
        $expiresAt = now()->addMinutes(10);
        $customer = \App\Models\Customer::find($customerId);
        if (!$customer) {
            return redirect()->back()->with('error', 'Something went wrong. Please try again');
        }
        CustomerLoginToken::create([
            'customer_id' => $customer->id,
            'token' => $hashedToken,
            'expires_at' => $expiresAt,
            'used' => false,
        ]);
        // Log::info("Admin {$admin->id} generated token for customer {$customerId}");
        // return response()->json([
        //     'url' => env('WEBSITE_URL'). 'en?as=' .  $customer->id . '&by-this=' . urlencode($hashedToken),
        // ]);

        return redirect()->away(env('WEBSITE_URL').'en?as=' .  $customer->id . '&by-this=' . urlencode($hashedToken));
    }

    public function adminAsCustomerLogList()
    {
        return view('admin.customers.login-as-customer-log-list');
    }

    public function getAdminAsCustomerLogList()
    {
        $logs = LoginAsCustomerLog::with(['customer', 'createdBy'])->latest();

        return DataTables::of($logs)
            ->addIndexColumn()
            ->addColumn('customer_name', function ($log) {
                if ($log->customer) {
                    $customerName = $log->customer->first_name ?? 'N/A';
                    $customerId = $log->customer->id ?? '';
                    $url = url("customers/{$customerId}");
                    return '<a href="' . $url . '" target="_blank">' . e($customerName) . '</a>';
                }
                return 'N/A';
            })
            ->addColumn('admin_name', function ($log) {
                if ($log->createdBy) {
                    $adminName = $log->createdBy->name ?? 'N/A';
                    $adminId = $log->createdBy->id ?? '';
                    $url = url("user/{$adminId}");
                    return '<a href="' . $url . '" target="_blank">' . e($adminName) . '</a>';
                }
                return 'N/A';
            })

            ->editColumn('created_at', function ($log) {

                return \Carbon\Carbon::parse($log->created_at)->format('F d, Y h:i A');
            })
            ->rawColumns(['customer_name','admin_name'])
            ->make(true);
    }
}
