<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Intervention\Image\ImageManagerStatic as Image;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\ApiData;
use App\Models\Setting;

use Carbon\Carbon;

class ApiController extends Controller
{
    /**
     * Display a listing of the API List.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $header = [
            'title' => "API List",
            'heading' => "API List",
        ];

        $perPage = $request->input('per_page') ??
            Setting::where('config_key', 'general|setting|pagePerAdminRecords')->value('value') ?? 20;

        $filter = [
            'per_page'   => $perPage,
            'order_by'   => $request->input('order_by') ?? 'created_at',
            'sorting'    => $request->input('sorting') ?? 'desc',
            'status'     => $request->input('status') ?? '',
            'end_point'  => $request->input('end_point') ?? '',
            'api_title'  => $request->input('api_title') ?? '',
            'category'   => $request->input('category') ?? '',
            'api_method' => $request->input('api_method') ?? '',
            'where'      => [],
        ];

        if ($filter['api_title']) {
            $filter['where'][] = ['api_title', 'LIKE', '%' . $filter['api_title'] . '%'];
        }
        if ($filter['end_point']) {
            $filter['where'][] = ['api_url', 'LIKE', '%' . $filter['end_point'] . '%'];
        }
        if ($filter['category']) {
            $filter['where'][] = ['api_category', 'LIKE', '%' . $filter['category'] . '%'];
        }
        if ($filter['api_method']) {
            $filter['where'][] = ['api_method', 'LIKE', '%' . $filter['api_method'] . '%'];
        }

        $queryStringConcat = http_build_query($request->only(['per_page', 'page']));
        $queryStringConcat = $queryStringConcat ? '?' . $queryStringConcat : '';

        $apiListDataList = ApiData::getApiList($filter);
        $apiListData = $apiListDataList['data'];

        return view('admin.api-list.index', [
            'header' => $header,
            'apiListData' => $apiListData,
            'queryStringConcat' => $queryStringConcat,
            'appliedFilter' => $filter,
            'i' => ($request->input('page', 1) - 1) * $filter['per_page'],
        ]);
    }


    /**
     * Show the form for creating a new best api
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        if (!hasPermission('CUSTOMERS_LIST', 'create')) {
            return view('admin/401');
        }
        $bestOfferId = $request->id;
        $header['title'] = 'API List - Add';

        return view('admin/api-list/add')->with(['header' => $header]);
    }

    /**
     * Store a newly created best apis in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $requestData = $request->all();
        $response = ApiData::createApi($requestData);
        if (!empty($response['data'])) {
            return redirect()->route('api-list.index', $response)->with('success', $response['message']);
        } else {
            return redirect()->back()->with('error', $response['message']);
        }
    }

    /**
     * Display the specified best api.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        if (!hasPermission('TRAVELLER_LIST', 'read')) {
            return view('admin/401');
        }
        $header['title'] = 'API List - View';
        $header['heading'] = 'API List - View';
        $filter = array(
            'id' => $id
        );
        $response = ApiData::getApiList($filter);

        $apiDetails = $response['data'];
        if ($response['status'] == 1 && !empty($response['data'])) {
            return view('admin/api-list/view')->with(['header' => $header, 'apiDetails' => $apiDetails]);
        } else {
            return redirect()->route('api-list.index')->with('error', $response['message']);
        }
    }

    /**
     * Show the form for editing the specified best api.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        if (!hasPermission('CUSTOMERS_LIST', 'update')) {
            return view('admin/401');
        }
        $header['title'] = 'API List - Edit';
        $header['heading'] = 'API - Edit';
        $header['method'] = 'Edit';

        $filter = array(
            'id' => $id
        );
        $response = ApiData::getApiList($filter);
        $apiDetails = $response['data'];
        if ($response['status'] == 1 && !empty($response['data'])) {
            return view('admin/api-list/update')->with(['header' => $header, 'apiDetails' => $apiDetails]);
        } else {
            return redirect()->route('api-list.index')->with('error', $response['message']);
        }
    }

    /**
     * Update the specified best api in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $url = $request->redirects_to;
        $requestData = $request->all();

        $rules = [];
        $customMessages = [];

        $niceNames = array();

        $this->validate($request, $rules, $customMessages, $niceNames);

        $response = ApiData::updateApi($requestData);

        if (!empty($response['data'])) {
            return redirect()->route('api-list.index', compact('id'))->with('success', $response['message']);
        } else {
            return redirect()->to($url)->with('error', $response['message']);
        }
    }

    /**
     * Remove the specified best api from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function deleteApi($apiId)
    {
        $url = URL::previous();
        $response = ApiData::deleteApi($apiId);
        $message = $response['message'] . '</br>';

        if ($response['status'] == 1) {
            return redirect()->to($url)->with('success', $message);
        } else {
            return redirect()->to($url)->with('error', $response['message']);
        }
    }
}
