<?php

namespace App\Http\Controllers\Admin\Action;

use App\Enums\Locale;
use App\Models\Setting;
use Illuminate\Http\Request;
use App\Models\BlockedFlight;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use App\Http\Controllers\Controller;
use App\Models\AirlineI18ns;

class BlockFlightController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $header['title'] = "Block Flights";
        $header['heading'] = "Block Flights";
        $queryStringConcat = '?';
        if (isset($_GET['per_page'])) {
            $queryStringConcat .= ($queryStringConcat == '') ? '?per_page=' . $_GET['per_page'] : '&per_page=' . $_GET['per_page'];
        }
        if (isset($_GET['page'])) {
            $queryStringConcat .= ($queryStringConcat == '') ? '?page=' . $_GET['page'] : '&page=' . $_GET['page'];
        }

        $appliedFilter = array(
            'per_page' => (request()->input('per_page') != NULL) ? request()->input('per_page') : Setting::where('config_key', 'general|setting|pagePerAdminRecords')->get('value')[0]['value'],
            'order_by' => (request()->input('order_by') != NULL) ? request()->input('order_by') : 'created_at',
            'sorting' => (request()->input('sorting') != NULL) ? request()->input('sorting') : 'desc',
            
            'airline_name' => (request()->input('airline_name') != NULL) ? request()->input('airline_name') : '',
            'status' => (request()->input('status') != NULL) ? request()->input('status') : '',
        );


        if (request()->input('airline_name') != NULL) {
            $appliedFilter['where'][] = ['airline_name', '=', request()->input('airline_name')];
        }
        if (request()->input('status') != NULL) {
            $appliedFilter['where'][] = ['status', '=', request()->input('status')];
        }

        $blockedFlightDataList = BlockedFlight::getBlockedFlights($appliedFilter);
        $blockedFlightDataCount = BlockedFlight::count();

        $blockedFlightData = $blockedFlightDataList['data'];
        if ($blockedFlightData['status'] == 1) {
            return view('admin/action.block-flight/index')->with(['header' => $header, 'appliedFilter' => $appliedFilter, 'blockedFlightCount' => $blockedFlightDataCount, 'blockedFlight' => $blockedFlightData, 'queryStringConcat' => $queryStringConcat, 'i' => (request()->input('page', 1) - 1) * $appliedFilter['per_page']]);
        } else {
            return view('admin/action.block-flight/index')->with(['header' => $header, 'appliedFilter' => $appliedFilter, 'blockedFlightData' => $blockedFlightData, 'blockedFlightCount' => $blockedFlightDataCount, 'queryStringConcat' => $queryStringConcat, 'i' => (request()->input('page', 1) - 1) * $appliedFilter['per_page']]);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $header['title'] = "Block Flights- Add";
        $header['heading'] = "Block Flights - Add";


        return view('admin/action.block-flight/add')->with(['header' => $header]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $requestData = $request->only(['airlines', 'status']);
    
        $rules = [
            'airlines' => 'required|string', 
            'status' => 'required|string',  
        ];
    
        $this->validate($request, $rules);

      
        $checkDuplicateData =  BlockedFlight::where('airline_id',$requestData['airlines'])->where('status',$requestData['status'])->exists();
        if($checkDuplicateData)
        {
            return redirect()->back()->with('error', 'The Status of Selected Flight is already a '. $requestData['status']);
        }
    
        try {
            DB::beginTransaction();
    
            $englishName = AirlineI18ns::where('airline_id', $requestData['airlines'])
                ->where('language_code', Locale::English->value)
                ->value('airline_name');
    
            $arabicName = AirlineI18ns::where('airline_id', $requestData['airlines'])
                ->where('language_code', Locale::Arabic->value)
                ->value('airline_name');
    
            BlockedFlight::create([
                'airline_id' => $requestData['airlines'],
                'airline_name' => $englishName ?? 'N/A',
                'airline_name_arabic' => $arabicName ?? 'N/A',
                'status' => $requestData['status'],
            ]);
    
            DB::commit();
            return redirect()->route('block-flight.index')->with('success', 'New Action Created Successfully');
        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            Log::error('Database Query Error', ['message' => $e->getMessage(), 'request' => $request->all()]);
            return redirect()->back()->with('error', 'Database error while creating blocked flight.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('General Error', ['message' => $e->getMessage(), 'request' => $request->all()]);
            return redirect()->back()->with('error', 'Error while creating the blocked flight.');
        }
    }
    
    
    
    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $header['title'] = "Block Flights - View";
        $header['heading'] = "Block Flights - View";
    
       
        $blockedFlightDetails = BlockedFlight::where('id', $id)->get();
    
       
        if (!$blockedFlightDetails->isEmpty()) {
           
            return view('admin/action.block-flight/view')->with([
                'header' => $header,
                'blockedFlightDetails' => $blockedFlightDetails->first(), 
            ]);
        } else {
            return redirect()->route('action.block-flight.index')->with('error', 'Error');
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $header['title'] = "Block Flights - Edit";
        $header['heading'] = "Block Flights - Edit";
    
      
        $blockedFlightDetails=BlockedFlight::where('id', $id)->get();
    
     
        if (!$blockedFlightDetails ->isEmpty()) {
       
            return view('admin/action.block-flight/update')->with([
                'header' => $header,
                'blockedFlightDetails' => $blockedFlightDetails ->first(), 
            ]);
        } else {
            return redirect()->route('action.block-flight.index')->with('error', 'Error');
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $requestData = $request->only(['airlines', 'status']);
    
        $rules = [
            'airlines' => 'required|string', 
            'status' => 'required|string',  
        ];
    
        $this->validate($request, $rules);

   
        try {
            DB::beginTransaction();
    
            $blockedFlight = BlockedFlight::findOrFail($id);
    
         
            $englishName = AirlineI18ns::where('airline_id', $requestData['airlines'])
                ->where('language_code', Locale::English->value)
                ->value('airline_name');
    
            $arabicName = AirlineI18ns::where('airline_id', $requestData['airlines'])
                ->where('language_code', Locale::Arabic->value)
                ->value('airline_name');
    
        
            $blockedFlight->update([
                'airline_id' => $requestData['airlines'],
                'airline_name' => $englishName ?? 'N/A',
                'airline_name_arabic' => $arabicName ?? 'N/A',
                'status' => $requestData['status'],
            ]);
    
            DB::commit();
            return redirect()->route('block-flight.index')->with('success', 'Blocked Flight updated successfully.');
        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            Log::error('Database Query Error', ['message' => $e->getMessage(), 'request' => $request->all()]);
            return redirect()->back()->with('error', 'Database error while updating blocked flight.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('General Error', ['message' => $e->getMessage(), 'request' => $request->all()]);
            return redirect()->back()->with('error', 'Error while updating the blocked flight.');
        }
    }
    

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }


    public function  deleteBlockedFlight($id)
    {
        
        $blockedFlight = BlockedFlight::find($id);
    
        
        if (!$blockedFlight) {
            return redirect()->route('action.block-flight.index')->with('error', 'BlockedFlight not found');
        }
    
        $url = URL::previous();
        if ($blockedFlight->delete()) {
         
            return redirect()->to($url)->with('success','Blocked Flights deleted successfully');
        } else {
            return redirect()->to($url)->with('error', 'Error deleting Blocked Flights');
   
        }
    }
}
