<?php

namespace App\Http\Controllers\API\V1;

use Exception;
use Carbon\Carbon;
use App\Enums\Locale;
use App\Models\Coupon;
use App\Models\Wallet;
use App\Enums\Currency;

use App\Models\Setting;
use App\Models\Bookings;
use App\Models\Suppliers;
use App\Traits\ActiveLog;

use App\Enums\ServiceType;
use App\Models\BookingLog;
use App\Models\StateI18ns;
use App\Traits\SmsService;
use App\Enums\BookingStatus;
use App\Traits\EmailService;
use Illuminate\Http\Request;
use App\Models\DefaultMarkup;
use App\Traits\BookingHelper;
use App\Traits\CommonService;
use App\Traits\TraacsService;
use App\Models\StoreReference;
use App\Traits\AmadeusService;

use App\Traits\BookingService;

use App\Traits\PaymentService;
use App\Models\BookingCheckout;
use App\Traits\WhatsappService;
use App\Traits\CheckFlightPrice;
use App\Traits\FlightListMarkup;
use App\Models\LoyaltyPointSetting;
use App\Models\PaymentCheckoutIdLog;
use App\Services\BrandedFareService;
use App\Traits\LoyaltyPointsService;
use Illuminate\Support\Facades\Auth;
use PhpParser\Node\Expr\Cast\Double;
use App\Models\CustomerLoyaltyPoints;
use App\Services\TranslateBookingDetails;

use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\API\V1\BaseController as BaseController;
use App\Models\Customer;

class PaymentController extends BaseController
{
    use AmadeusService, CommonService, EmailService, SmsService, WhatsappService, ActiveLog, BookingHelper, BookingService, LoyaltyPointsService, PaymentService, TraacsService, FlightListMarkup, CheckFlightPrice;

    public function __construct()
    {

        $this->perPage = count(Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')) > 0 ? Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')[0]['value'] : "20";

        //set AMADEUS API configuration from config key
        $this->amadeusAPIEnvironment = env('AMADEUS_API_CREDENTIALS');

        if ($this->amadeusAPIEnvironment == 'test') {
            $this->amadeusAPIEndPoint = env('AMADEUS_API_TEST_API_ENDPOINT');
            $this->amadeusAPIClientID = env('AMADEUS_API_TEST_CLIENT_ID');
            $this->amadeusAPIClientSecret = env('AMADEUS_API_TEST_CLIENT_SECRET');
            $this->amadeusAPIGrantType = env('AMADEUS_API_TEST_GRANT_TYPE');
        } else {
            $this->amadeusAPIEndPoint = env('AMADEUS_API_LIVE_API_ENDPOINT');
            $this->amadeusAPIClientID = env('AMADEUS_API_LIVE_CLIENT_ID');
            $this->amadeusAPIClientSecret = env('AMADEUS_API_LIVE_CLIENT_SECRET');
            $this->amadeusAPIGrantType = env('AMADEUS_API_LIVE_GRANT_TYPE');
        }

        $this->amadeusAPISecret = env('AMADEUS_API_SECRET');
    }

    public function createReference(Request $request)
    {
        // return $request->all();
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);
        $bookingVia = $request->input('booking_via', 'mobile-app');
        $bookingRefUrl = $request->input('booking_ref_url', 'https://demo.rehlte.com');
        $bookingRefUrl .= '/' . $requestData['locale'];

        $amaClientRef = $logId = '';
        $customerId = $requestData['customer_id'];
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];
        }
        $bookingDetails = [
            'userDetails' => [
                'userId' => $customerId
            ],
            'paymentDetails' => [
                'addon_details' => [
                    'status' => false,
                    'amount' => 0
                ],
                'bag_details' => [
                    'status' => false,
                    'amount' => 0
                ],
                'seatmap_details' => [
                    'status' => false,
                    'amount' => 0
                ],
                'coupon_details' => [
                    'status' => false,
                    'id' => '',
                    'code' => '',
                    'type' => 'amount',
                    'value' => 0,
                    'amount' => 0,
                ],
                'wallet_details' => [
                    'status' => false,
                    'amount' => 0
                ],
                'redeem_details' => [
                    'status' => false,
                    'amount' => 0
                ],
            ]
        ];
        $searchFrom = $request->input('searchFrom', '');
        if ($searchFrom && $searchFrom != 'b2c_desktop') {
            $requestData['search_details']['searchFrom'] = $request->input('searchFrom', '');
            $requestData['search_details']['originDevice'] = $request->input('searchFrom', 'web');
        }
        if ($requestData['type'] == ServiceType::Flight->value) {
            $hadAddBags = false;
            if (
                array_key_exists('additionalServices', $requestData['flight_details'][0]['price']) &&
                count($requestData['flight_details'][0]['price']['additionalServices']) > 0
            ) {
                foreach ($requestData['flight_details'][0]['price']['additionalServices'] as $additionalService) {
                    if ($additionalService['type'] == 'CHECKED_BAGS' && $additionalService['amount'] > 0) {
                        $hadAddBags = true;
                    }
                }
            }
            // Get Latest Flight Price
            $flightOfferPricingPayload = $this->getFlightOfferPricingPayload($requestData['currency'], $requestData['locale'], $requestData['flight_details'], $requestData['service_provider']);

            $flightOfferPricingResponse = $this->flightOfferPriceGetWithMarkup($flightOfferPricingPayload, $logId, $customerId, $amaClientRef);

            if (array_key_exists('data', $flightOfferPricingResponse)) {
                $requestData['flight_details'] = $flightOfferPricingResponse['data']['flightOffers'];
            } else {
                return $this->sendResponse([], 'Unable to proceed next. Please select different flight.', false);
            }

            foreach ($requestData['flight_details'] as &$flightDetail) {
                if (!isset($flightDetail['travelerPricings']) || !is_array($flightDetail['travelerPricings'])) {
                    continue;
                }

                foreach ($flightDetail['travelerPricings'] as &$travellerPricing) {
                    if (
                        isset($travellerPricing['price']) &&
                        is_array($travellerPricing['price']) &&
                        !isset($travellerPricing['price']['initialTotal'])
                    ) {
                        $travellerPricing['price']['initialTotal'] = $travellerPricing['price']['total'];
                    }
                }
                unset($travellerPricing);
            }
            unset($flightDetail);


            $addonPrice = Setting::where('config_key', 'general|site|flightBookingAddOn')->get('value')[0]['value'];
            $bookingDetails['paymentDetails']['addon_details']['amount'] = $addonPrice;
            $serviceBookingDetails = [
                'bookingType' => ServiceType::Flight->value,
                'customer_currency' => $requestData['currency'],
                'customer_language_code' => $requestData['locale'],
                'service_type_id' => 1,
                'supplier' => $requestData['service_provider'],
                'supplier_id' => 9,
                'agencyId' => 0
            ];
            $amaClientRef = $requestData['ama_client_ref'];
            $amount = 0;
            foreach ($requestData['flight_details'] as $key => $flightDetail) {
                $amount += $flightDetail['price']['grandTotal'];
            }
            $bookingDetails['bookingDetails'] = $serviceBookingDetails;

            $bookingDetails['searchDetails'] = $requestData['search_details'];
            $bookingDetails['flightDetails'] = [
                'additionalDetails' => [
                    'addBags' => [
                        'addedToFlightOffer' => false,
                        'status' => $hadAddBags,
                        'selectedBagDetails' => []
                    ]
                ],
                'airlines' => $requestData['airlines'],
                'brandedFares' => $requestData['branded_fares'],
                'flightInfo' => $requestData['flight_details'],
                'oneWayCombinable' => $requestData['one_way_combinable'],
                'packageDetails' => [],
                'passengerDetails' => [],
                'seatDetails' => []
            ];
            $flightInfo = $bookingDetails['flightDetails']['flightInfo'];

            foreach ($flightInfo as &$flight) {
                $bookingDetails['flightDetails']['brandedFaresBySegment'][]  = BrandedFareService::createBrandedFare($flight);
            }
        } elseif ($requestData['type'] == ServiceType::Hotel->value) {
            $addonPrice = Setting::where('config_key', 'general|site|hotelBookingAddOn')->get('value')[0]['value'];
            $bookingDetails['paymentDetails']['addon_details']['amount'] = $addonPrice;
            $serviceBookingDetails = [
                'bookingType' => ServiceType::Hotel->value,
                'customer_currency' => $requestData['currency'],
                'customer_language_code' => $requestData['locale'],
                'service_type_id' => 2,
                'supplier' => $requestData['service_provider'],
                'supplier_id' => 11,
                'agencyId' => 0
            ];
            $addressDetails = $this->getAddressDetails($requestData['locale'], $requestData['search_details']['geolocation']['latitude'], $requestData['search_details']['geolocation']['longitude']);
            $requestData['search_details']['address'] = $addressDetails;
            $requestData['search_details']['city'] = $requestData['city'] ?? '';
            $requestData['search_details']['nationality'] = $requestData['country'] ?? '';

            $amount = $requestData['room_details']['SimplePrice'];
            $bookingDetails['bookingDetails'] = $serviceBookingDetails;
            $bookingDetails['searchDetails'] = $requestData['search_details'];
            $bookingDetails['hotelDetails'] = [
                'hotelDetails' => $requestData['hotel_details'],
                'selectedRooms' => $requestData['room_details'],
                'cancellationPolicy' => $requestData['cancellation_policy'],
                'refundType' => ucfirst($requestData['refund_type'] ?? ''),
            ];
        }

        $processedPrice = $this->getProcessedPrice($requestData['currency'], $customerId, $bookingDetails, $initialLoad = false, $requestData);

        $bookingDetails = $processedPrice['booking_details'];
        unset($processedPrice['booking_details']);
        $bookingDetails['processedPrice'] = $processedPrice;

        $bookingPayload = [
            'ref_id' => getRandomString(),
            'log_id' => $logId,
            'customer_id' => $customerId,
            'loyalty_points' => $processedPrice['loyaltyPoints'],
            'checkout_ids' => '',
            'initial_amount' => $amount,
            'amount' => $amount,
            'booking_type' => $requestData['type'],
            'booking_via' => $bookingVia,
            'checkout_url' => $bookingRefUrl,
            'service_provider' => $requestData['service_provider'],
            'supplier_id' => $requestData['room_details']['SupplierId'] ?? '',
            'supplier_name' => $requestData['room_details']['SupplierName'] ?? '',
            'ama_client_ref' => $amaClientRef,
            'booking_details' => json_encode($bookingDetails),
        ];

        $savedCheckoutDetails = BookingCheckout::createBookingCheckout($bookingPayload);
        $refId = $savedCheckoutDetails['data']['ref_id'];

        StoreReference::create(
            [
                'reference_id' => $refId,
                'payload' => null,
                'initial_payload_result' => $savedCheckoutDetails['data']['booking_details']
            ]

        );

        return $this->sendResponse(['ref_id' => $refId], 'Booking Reference Created Successfully');
    }

    public function updateReference(Request $request)
    {

        $requestData = $request->all();

        $requestHeaders = $request->header();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);

        // $refId = $requestData['ref_id'];
        // StoreReference::updateOrCreate([
        //     'reference_id' => $refId,
        // ], [
        //     'payload' => json_encode($requestData)
        // ]);


        $refId = $requestData['ref_id'];

        $existingRecord = StoreReference::where('reference_id', $refId)->first();
        if ($existingRecord) {
            $existingPayload = json_decode($existingRecord->payload, true) ?? [];
            $mergedPayload = array_merge_recursive_distinct($existingPayload, $requestData);
        } else {

            $mergedPayload = $requestData;
        }
        StoreReference::updateOrCreate(
            ['reference_id' => $refId],
            ['payload' => json_encode($mergedPayload)]
        );

        $savedBookingCheckout = BookingCheckout::getBookingCheckout($refId);
        $bookingCheckoutId = $savedBookingCheckout['data']['id'];
        $bookingDetails = $savedBookingCheckout['data']['booking_details'];
        $bookingDetails = array_merge(json_decode($bookingDetails, true));



        $isAlreadyBooked = false;
        $isBooked = Bookings::where('ref_id', $refId)->first();
        if ($isBooked) {
            if ($isBooked->booking_status == BookingStatus::Confirmed->value) {
                $isAlreadyBooked = true;
                return $this->sendResponse([
                    'ref_id' => $refId,
                    'isAlreadyBooked' => $isAlreadyBooked,
                ], 'Booking already confirmed');
            } else {
                $isAlreadyBooked = false;
            }
        }
        $isError = false;

        if ($requestData['type'] == 'travelers-reset') {
            if (!empty($existingRecord['initial_payload_result'])) {
                $bookingPayload = [

                    'booking_details' => $existingRecord['initial_payload_result'],
                ];
                StoreReference::where('reference_id', $refId)->update(['payload' => null]);

                $savedCheckoutDetails = BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);
                if ($savedCheckoutDetails['status'] == 'true') {
                    return $this->sendResponse([$savedCheckoutDetails['data']], 'Reset Successfully');
                } else {
                    return $this->sendError('Unable to Reset. Please try again', [], 200);
                }
            } else {
                return $this->sendError('Something went wrong. Please try again', [], 200);
            }
        }
        if ($requestData['type'] == 'flight-continue-payment') {
            $billingDetails = [
                'address' => 'Abu Bakr As Siddiq Rd',
                'address1' => '',
                'city' => 'Buraydah',
                'state' => 'Al qassim',
                'country' => 'SA',
                'zipcode' => '52389',
                'state_id' => '',
                'country_id' => '98870058-a7c4-469c-8ef9-423cc74e59b3',
                'customer.email' => $requestData['userDetails']['email'],
                'customer.givenName' => $requestData['passengersDetails'][0]['first_name'] ?? '',
                'customer.surname' => $requestData['passengersDetails'][0]['last_name'] ?? '',
            ];

            // $isPriceUpdated = $this->getUpdatedFlightPriceDetails($savedBookingCheckout);
            // $bookingDetails['latestUpdates']['flightPrice'] = $isPriceUpdated ?? '';


            $bookingDetails['billingDetails'] = $billingDetails;
            $bookingDetails['userDetails'] = $requestData['userDetails'];
            $bookingDetails['passengerDetails'] = $requestData['passengersDetails'];
            $bookingDetails['cancellationPolicy'] = $requestData['cancellation_policy'] ?? [];
            // $bookingDetails['flightDetails']['seatDetails'][] = $requestData['seatmap'];

            $bookingDetails['flightDetails']['additionalDetails']['addBags']['status'] = isset($requestData['baggage']) && !empty($requestData['baggage']) ? true : false;
            $bookingDetails['flightDetails']['additionalDetails']['addBags']['selectedBagDetails'] = isset($requestData['baggage']) && !empty($requestData['baggage']) ? $requestData['baggage'] : [];
            $bookingDetails['flightDetails']['seatDetails']['status'] = isset($requestData['seatmap']) && !empty($requestData['seatmap']) ? true : false;
            $bookingDetails['flightDetails']['seatDetails']['seatDetails'] = isset($requestData['seatmap']) && !empty($requestData['seatmap'])  ? $requestData['seatmap'] : [];

            $amaClientRef = $customerId = $logId = '';
            if ($bookingDetails['userDetails']['userId'] != '') {
                $customerId = $bookingDetails['userDetails']['userId'];
            }
            if (isset($requestHeaders['log-id'])) {
                $logId = $requestHeaders['log-id'][0];
            }
            if (isset($requestHeaders['ama-client-ref'])) {
                $amaClientRef = $requestHeaders['ama-client-ref'][0];
            }
            $bookingDetails = $this->getSeatmapFlightOffers($bookingDetails);

            $flightOfferPricingPayload = $this->getFlightOfferPricingPayload($requestData['currency'], $requestData['locale'], $bookingDetails['flightDetails']['flightInfo'], $savedBookingCheckout['data']['service_provider']);

            $flightOfferPricingResponse = $this->flightOfferPriceGetWithMarkup($flightOfferPricingPayload, $logId, $customerId, $amaClientRef);


            //echo "<pre>"; print_r($flightOfferPricingPayload); print_r($flightOfferPricingResponse); die;
            if (array_key_exists('data', $flightOfferPricingResponse)) {
                // $bookingDetails['flightDetails']['flightInfo'] = $flightOfferPricingResponse['data']['flightOffers'];
                // $bookingDetails['flightDetails']['flightInfo'] = array_merge_recursive_distinct($bookingDetails['flightDetails']['flightInfo'], $flightOfferPricingResponse['data']['flightOffers']);;

                $processedPrice = $this->getProcessedPrice($requestData['currency'], $customerId, $bookingDetails, $initialLoad = false, $requestData);

                $bookingDetails = $processedPrice['booking_details'];

                unset($processedPrice['booking_details']);
                $bookingDetails['processedPrice'] = $processedPrice;

                $loyaltyPoints = $processedPrice['loyaltyPoints'];
                $bookingPayload = [
                    'customer_id' => $customerId ?? '',
                    'amount' => $processedPrice['grandTotal'],
                    'loyalty_points' => $loyaltyPoints,
                    'booking_details' => json_encode($bookingDetails)
                ];
            } else {
                $isError = true;
            }
        } elseif ($requestData['type'] == 'flight-add-bags') {
            //$bagPrice = $requestData['data']['price']['amount'];

            // $bookingDetails['processedPrice']['grandTotal'] -=   $bookingDetails['processedPrice']['markupValue'] ?? 0;
            $bookingDetails['userDetails'] = $requestData['userDetails'] ?? [];
            $bookingDetails['passengerDetails'] = $requestData['passengersDetails'] ?? [];
            $bookingDetails['flightDetails']['additionalDetails']['addBags']['status'] = true;
            $bookingDetails['flightDetails']['additionalDetails']['addBags']['selectedBagDetails'] = $requestData['baggage'];

            // dd($bookingDetails['flightDetails']['additionalDetails']['addBags']['selectedBagDetails']);

            $processedPrice = $this->getProcessedPrice($requestData['currency'], $savedBookingCheckout['data']['customer_id'], $bookingDetails, $initialLoad = false, $requestData);

            $bookingDetails = $processedPrice['booking_details'];
            unset($processedPrice['booking_details']);
            $bookingDetails['processedPrice'] = $processedPrice;

            $bookingPayload = [
                'amount' => $processedPrice['grandTotal'],
                'booking_details' => json_encode($bookingDetails),
            ];
        } else if ($requestData['type'] == 'flight-add-seat') {
            $bookingDetails['userDetails'] = isset($requestData['userDetails']) ? $requestData['userDetails'] :  [];
            $bookingDetails['passengerDetails'] = isset($requestData['passengersDetails']) ? $requestData['passengersDetails'] : [];
            $bookingDetails['flightDetails']['seatDetails']['status'] = true;
            $bookingDetails['flightDetails']['seatDetails']['seatDetails'] = $requestData['seatmap'];
            // dd($bookingDetails['flightDetails']['seatDetails']);
            $processedPrice = $this->getProcessedPrice($requestData['currency'], $savedBookingCheckout['data']['customer_id'], $bookingDetails, $initialLoad = false, $requestData);

            $bookingDetails = $processedPrice['booking_details'];
            unset($processedPrice['booking_details']);
            $bookingDetails['processedPrice'] = $processedPrice;
            $bookingPayload = [
                'amount' => $processedPrice['grandTotal'],
                'booking_details' => json_encode($bookingDetails),
            ];
        } else {
            $bookingPayload = [

                'booking_details' => json_encode($bookingDetails),
            ];
        }
        if ($isError) {
            return $this->sendResponse([], 'Unable to proceed next. Please select different flight.', false);
        } else {
            $savedCheckoutDetails = BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);

            BookingLog::create([
                'ref_id' => $refId,
                'booking_type' => $savedBookingCheckout['data']['booking_type'] ?? '',
                'customer_id' => $requestData['userDetails']['userId'] ?? '',
                'email' => $requestData['userDetails']['email'] ?? '',
                'phone_code' => $requestData['userDetails']['phoneCode'] ?? '',
                'phone_no' => $requestData['userDetails']['phoneNumber'] ?? '',
                'payload' => json_encode($requestData) ?? '',
                'amount' => $bookingDetails['processedPrice']['grandTotal'] ?? '',
                'currency' => $bookingDetails['processedPrice']['currency'] ?? '',
                'booking_details' => $savedCheckoutDetails['data']['booking_details']
            ]);
            if ($savedCheckoutDetails['status'] == 'true') {
                $result['success'] = true;
                if ($requestData['type'] == 'flight-add-bags') {
                    // $result['data'] = $bookingDetails['processedPrice'];
                    $result = $savedCheckoutDetails['data'];
                }
                if ($requestData['type'] == 'flight-add-seat') {
                    // $result['data'] = $bookingDetails['processedPrice'];
                    $result = $savedCheckoutDetails['data'];
                }
                return $this->sendResponse($result, 'Booking Reference updated successfully');
            } else {
                return $this->sendResponse([], 'Unable to proceed next. Please select different flight.', false);
            }
        }
    }

    public function addHotelGuestInfo(Request $request)
    {
        $requestData = $request->all();
        $refId = $requestData['ref_id'];
        $customer_id = $requestData['customer_id'] ?? '';
        $savedBookingCheckout = BookingCheckout::getBookingCheckout($refId);
        $bookingCheckoutId = $savedBookingCheckout['data']['id'];
        $bookingDetails = $savedBookingCheckout['data']['booking_details'];
        $tempBookingDetails = json_decode($bookingDetails, true);
        $userDetails = [];
        if (array_key_exists('userDetails', $tempBookingDetails)) {
            $userDetails = $tempBookingDetails['userDetails'];
        } else {
            $tempBookingDetails['userDetails']['userId'] = $savedBookingCheckout['data']['customer_id'];
        }
        $userDetails = array_merge($userDetails, $requestData['guest_details']);
        $bookingDetails = array_merge($tempBookingDetails, ['userDetails' => $userDetails]);



        $isAlreadyBooked = false;
        $isBooked = Bookings::where('ref_id', $refId)->first();
        if ($isBooked) {
            if ($isBooked->booking_status == BookingStatus::Confirmed->value) {
                $isAlreadyBooked = true;
                return $this->sendResponse([
                    'ref_id' => $refId,
                    'isAlreadyBooked' => $isAlreadyBooked,
                ], 'Booking already confirmed');
            } else {
                $isAlreadyBooked = false;
            }
        }
        $billingDetails = [
            'address' => 'Abu Bakr As Siddiq Rd',
            'address1' => '',
            'city' => 'Buraydah',
            'state' => 'Al qassim',
            'country' => 'SA',
            'zipcode' => '52389',
            'state_id' => '',
            'country_id' => '98870058-a7c4-469c-8ef9-423cc74e59b3',
            'customer.email' => $userDetails['email'],
            'customer.givenName' => $userDetails['firstName'],
            'customer.surname' => $userDetails['lastName'],
        ];
        $bookingDetails['billingDetails'] = $billingDetails;
        $finalAmount =  $bookingDetails['processedPrice']['grandTotal'];

        $bookingPayload = [
            'amount' => $finalAmount,
            'booking_details' => json_encode($bookingDetails),
        ];
        if (!$savedBookingCheckout['data']['customer_id'] && $customer_id) {
            $bookingPayload = [
                'customer_id' => $customer_id,
                'amount' => $finalAmount,
                'booking_details' => json_encode($bookingDetails),
            ];
        }
        $savedCheckoutDetails = BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);

        if ($savedCheckoutDetails['status'] == 'true') {

            BookingLog::create([
                'ref_id' => $refId,
                'booking_type' => $savedBookingCheckout['data']['booking_type'] ?? '',
                'customer_id' => $savedBookingCheckout['data']['customer_id'] ?? '',
                'email' => $requestData['guest_details']['email'] ?? $requestData['guest_details']['guestEmail'] ?? '',
                'phone_code' => $requestData['guest_details']['phoneCode'] ?? '',
                'phone_no' => $requestData['guest_details']['phoneNumber'] ?? '',
                'payload' => json_encode($requestData) ?? '',
                'amount' => $tempBookingDetails['processedPrice']['grandTotal'] ?? '',
                'currency' => $tempBookingDetails['processedPrice']['currency'] ?? '',
                'booking_details' => $savedCheckoutDetails['data']['booking_details'] ?? ''
            ]);

            return $this->sendResponse([], 'Booking Details updated Successfully');
        }
        return $this->sendResponse([], 'Unable to update reference details. Please try again.', false);
    }

    private function translatedBookingDetails($bookingDetails, $locale, $bookingType)
    {
        if ($bookingType == ServiceType::Flight->value) {
            $translatedBookingDetails = TranslateBookingDetails::translateFlightBookingDetails($bookingDetails, $locale);

            $translatedBookingDetails['booking_details']['booking_details']['searchDetails']['languageCode'] = $locale ?? $translatedBookingDetails['booking_details']['booking_details']['searchDetails']['languageCode'];
            $translatedBookingDetails['booking_details']['booking_details']['bookingDetails']['customer_language_code'] = $locale ?? $translatedBookingDetails['booking_details']['booking_details']['searchDetails']['languageCode'];
            $bookingDetails = json_encode($translatedBookingDetails['booking_details']['booking_details']) ?? [];
            return $bookingDetails;
        } else {
            $bookingDetails = [
                'booking_details' => $bookingDetails
            ];
            $translatedBookingDetails = TranslateBookingDetails::translateHotelBookingDetails($bookingDetails, $locale);
            $translatedBookingDetails['booking_details']['bookingDetails']['customer_language_code'] = $locale;
            return json_encode($translatedBookingDetails['booking_details']) ?? [];
        }
    }

    public function getReference(Request $request)
    {
        $refId = $request->input('ref_id', '');
        if (!$refId) {
            return $this->sendResponse([], 'Reference Id Missing', false);
        }
        $locale = $request->input('locale');
        $savedBookingCheckout = BookingCheckout::getBookingCheckout($refId);

        if (!$savedBookingCheckout || !isset($savedBookingCheckout['data'])) {
            return $this->sendResponse([], 'Booking not found');
        }

        $isAlreadyBooked = false;
        $isBooked = Bookings::where('ref_id', $refId)->first();
        if ($isBooked) {
            if ($isBooked->booking_status == BookingStatus::Confirmed->value) {
                $isAlreadyBooked = true;
            } else {
                $isAlreadyBooked = false;
            }
        }

        $storedReference = StoreReference::where('reference_id', $refId)->first();
        $savedBookingDetails = json_decode($savedBookingCheckout['data']['booking_details'], true);
        $bookingDetails = [
            'booking_details' => [
                'booking_details' => $savedBookingDetails
            ]
        ];

        $bookingType =  $savedBookingCheckout['data']['booking_type'];
        if ($bookingType == ServiceType::Flight->value) {
            $userSelectedLang = $savedBookingDetails['searchDetails']['languageCode'] ?? Locale::English->value;
            if ($userSelectedLang != $locale) {
                $bookingDetails = $this->translatedBookingDetails($bookingDetails, $locale, $bookingType);
                if (!empty($bookingDetails)) {
                    $bookingRefUrl = env('WEBSITE_URL') . $locale;
                    $bookingPayload = [
                        'checkout_url' => $bookingRefUrl,
                        'booking_details' => $bookingDetails
                    ];
                    $savedBookingCheckout =  BookingCheckout::updateBookingCheckout($savedBookingCheckout['data']['id'], $bookingPayload);
                    $savedBookingCheckout['data'] = json_decode($savedBookingCheckout['data'], true);
                }
            }
        } else {
            $userSelectedLang = $savedBookingDetails['bookingDetails']['customer_language_code'] ?? Locale::English->value;
            if ($userSelectedLang != $locale) {
                $bookingDetails = $this->translatedBookingDetails($savedBookingDetails, $locale, $bookingType);

                if (!empty($bookingDetails)) {
                    $bookingRefUrl = env('WEBSITE_URL') . $locale;
                    $bookingPayload = [
                        'checkout_url' => $bookingRefUrl,
                        'booking_details' => $bookingDetails
                    ];
                    $savedBookingCheckout =  BookingCheckout::updateBookingCheckout($savedBookingCheckout['data']['id'], $bookingPayload);
                    $savedBookingCheckout['data'] = json_decode($savedBookingCheckout['data'], true);
                }
            }
        }


        $data = array_merge(
            $savedBookingCheckout['data'],
            ['stored_data' => $storedReference['payload'] ?? []],
            ['isAlreadyBooked' =>  $isAlreadyBooked]
        );
        return $this->sendResponse($data, 'Reference details retrieved successfully');
    }


    public function bookAddOns(Request $request)
    {

        $requestData = $request->all();
        $currency = $request->input('currency', Currency::SAR->value);

        $savedBookingCheckout = BookingCheckout::getBookingCheckout($requestData['ref_id']);
        if ($savedBookingCheckout) {
            $customerId = $savedBookingCheckout['data']['customer_id'];
            $bookingDetails = json_decode($savedBookingCheckout['data']['booking_details'], true);
            $bookingAddAmount = $bookingDetails['paymentDetails']['addon_details']['amount'];
            if ($savedBookingCheckout['data']['booking_type'] == ServiceType::Flight->value) {
                // $amount = $bookingDetails['flightDetails']['flightInfo'][0]['price']['grandTotal'];
                // if ($bookingDetails['searchDetails']['oneWayCombinable']) {
                //     $amount += $bookingDetails['flightDetails']['flightInfo'][1]['price']['grandTotal'];
                // }
                $amount = $bookingDetails['processedPrice']['grandTotal'];
            } elseif ($savedBookingCheckout['data']['booking_type'] == ServiceType::Hotel->value) {
                $amount = $bookingDetails['hotelDetails']['selectedRooms']['SimplePrice'];
            }
            if ($requestData['add'] == true) {
                $amount = $amount + $bookingAddAmount;
                $bookingDetails['paymentDetails']['addon_details'] = [
                    'status' => true,
                    'amount' => $bookingAddAmount
                ];
            } else {

                $bookingDetails['paymentDetails']['addon_details'] = [
                    'status' => false,
                    'amount' => $bookingAddAmount
                ];
            }
            $processedPrice = $this->getProcessedPrice($requestData['currency'], $customerId, $bookingDetails, $initialLoad = false, $requestData);
            $bookingDetails = $processedPrice['booking_details'];
            unset($processedPrice['booking_details']);
            if ($savedBookingCheckout['data']['booking_type'] == ServiceType::Flight->value) {
                // if ($requestData['add'] === false) {
                //     $processedPrice['grandTotal'] -=   $processedPrice['markupValue'] ?? 0;
                // }
            }

            $bookingDetails = array_merge($bookingDetails, ['processedPrice' => $processedPrice]);
            $loyaltyPoints = $processedPrice['loyaltyPoints'];
            $payload = [
                'amount' => $amount,
                'loyalty_points' => $loyaltyPoints,
                'booking_details' => json_encode($bookingDetails)
            ];
            $savedCheckoutDetails = BookingCheckout::updateBookingCheckout($savedBookingCheckout['data']['id'], $payload);
            if ($savedCheckoutDetails['status'] == 'true') {
                return $this->sendResponse($savedCheckoutDetails['data'], 'Book Add-on ' . (($requestData['add'] == true) ? 'added' : 'removed') . ' successfully');
            }
        }

        return $this->sendResponse([], 'Something went wrong', false);
    }



    public function applyCoupon(Request $request)
    {
        $requestData = $request->all();
        $currency = $request->input('currency', Currency::SAR->value);
        $isCouponInvalid = false;

        $savedBookingCheckout = BookingCheckout::getBookingCheckout($requestData['ref_id']);
        if (!$savedBookingCheckout) {
            return $this->sendResponse([], 'Booking not found.', false);
        }

        $bookingDetails = json_decode($savedBookingCheckout['data']['booking_details'], true);
        $customerId = $savedBookingCheckout['data']['customer_id'];
        $now = Carbon::now();
        if($savedBookingCheckout['data']['booking_type'] == ServiceType::Flight->value){
            $fromDate =  $bookingDetails['searchDetails']['originDestinations'][0]['departureDate'] ?? [];
            if(!empty($fromDate)){
                $now = Carbon::parse($fromDate)->startOfDay();
            }
        }else{
            $fromDate =  $bookingDetails['searchDetails']['stay']['checkIn'] ?? [];
            if(!empty($fromDate)){
                $now = Carbon::parse($fromDate)->startOfDay();
            }
        }
        $couponCode = trim($requestData['coupon_code']);
        $couponDetails = Coupon::where('coupon_code', $couponCode)
            ->where('status', 'active')
            ->first();

        if (!$couponDetails) {
            $isCouponInvalid = true;
        } else {
            $fromDate = Carbon::parse($couponDetails->from_date)->startOfDay();
            $toDate = Carbon::parse($couponDetails->to_date)->endOfDay();

            if (!$now->between($fromDate, $toDate)) {
                $isCouponInvalid = true;
            }

            if ($couponDetails->limit_per_coupon == $couponDetails->redeem_count) {
                $isCouponInvalid = true;
            }

            if (
                in_array($couponDetails->coupon_type, ['discount coupon', 'cashback coupon']) &&
                Carbon::parse($couponDetails->expiry_date)->isPast()
            ) {
                $isCouponInvalid = true;
            }
        }
        $payload = ['booking_details' => json_encode($bookingDetails)];
        $isCouponUsed = false;
        $message = '';

        if (!$isCouponInvalid) {
            $bookingType = $savedBookingCheckout['data']['booking_type'];
            
            $amount = $bookingDetails['processedPrice']['grandTotal'] ?? 0;
 
            if (strtolower($couponDetails->product) != strtolower($bookingType)) {
                $bookingDetails['paymentDetails']['coupon_details'] = [
                    'status' => false,
                    'id' => '',
                    'code' => '',
                    'type' => 'amount',
                    'value' => 0,
                    'amount' => 0,
                    'additional_details' => '',
                ];
                $result['message'] = 'Invalid Coupon Code';
            } else {
                $couponAmount = 0;
                $cashbackAmount = null;
                $additionalDetails = 'amount';
                $couponType = '';

                if ($requestData['coupon_used'] == true) {
                    $minSpend = explode(',', $couponDetails->minimum_spend);
                    $maxSpend = explode(',', $couponDetails->maximum_spend);
                    $redeemAmounts = explode(',', $couponDetails->coupon_amount);
                    $matched = false;
                    
                    for ($i = 0; $i < max(count($minSpend), count($maxSpend), count($redeemAmounts)); $i++) {
                        $min = $minSpend[$i] ?? 0;
                        $max = $maxSpend[$i] ?? 0;
                        $redeem = $redeemAmounts[$i] ?? 0;
                        if ($amount >= $min && $amount <= $max) {
                            $couponAmount = $redeem;
                            $matched = true;
                            break;
                        }
                    }

                    if (!$matched) {
                        $isCouponUsed = false;
                    } else {
                        if ($couponDetails->discount_type == 'percentage') {
                            $couponAmount = ($amount * $couponAmount) / 100;
                            $additionalDetails = 'percentage';
                        }

                        if ($couponDetails->coupon_type == 'discount coupon') {
                            $amount -= $couponAmount;
                            $couponType = 'discount_coupon';
                            $cashbackAmount = null;
                        } elseif ($couponDetails->coupon_type == 'cashback coupon') {
                            $cashbackAmount = $couponAmount;
                            $couponAmount = 0;
                            $couponType = 'cashback_coupon';
                        }

                        $bookingDetails['paymentDetails']['coupon_details'] = [
                            'status' => true,
                            'id' => $couponDetails->id,
                            'code' => $couponDetails->coupon_code,
                            'type' => $couponDetails->discount_type,
                            'value' => $couponDetails->coupon_amount,
                            'amount' => $couponAmount,
                            'coupontype' => $couponType,
                            'cashbackAmount' => $cashbackAmount,
                            'additional_details' => $additionalDetails,
                        ];
                        $isCouponUsed = true;
                        $message = 'Coupon has been successfully redeemed';
                    }
                } else {
                    $bookingDetails['paymentDetails']['coupon_details'] = [
                        'status' => false,
                        'id' => '',
                        'code' => '',
                        'type' => 'amount',
                        'value' => 0,
                        'amount' => 0,
                        'additional_details' => '',
                    ];
                    $isCouponUsed = true;
                    $message = 'Coupon has been successfully removed';
                }

                $processedPrice = $this->getProcessedPrice(
                    $currency,
                    $customerId,
                    $bookingDetails,
                    $initialLoad = false,
                    $requestData
                );

                $bookingDetails = $processedPrice['booking_details'];
                unset($processedPrice['booking_details']);

                $bookingDetails = array_merge($bookingDetails, ['processedPrice' => $processedPrice]);
                $loyaltyPoints = $processedPrice['loyaltyPoints'];

                $payload = [
                    'amount' => $amount,
                    'loyalty_points' => $loyaltyPoints,
                    'booking_details' => json_encode($bookingDetails),
                ];
            }
        } else {
            return $this->sendResponse([], 'Invalid Coupon Code', false);
        }

        $savedCheckoutDetails = BookingCheckout::updateBookingCheckout($savedBookingCheckout['data']['id'], $payload);
        if ($savedCheckoutDetails['status'] == 'true') {
            return $this->sendResponse($savedCheckoutDetails['data'], $message, $isCouponUsed);
        }

        return $this->sendResponse([], 'Something went wrong. Please try again', false);
    }



    public function applyWalletBalance(Request $request)
    {
        $requestData = $request->all();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);

        $refId = $requestData['ref_id'];
        $paymentBrand = $requestData['payment_brand'];
        $walletAmount = $requestData['wallet_amount'];
        $walletUsed = $requestData['wallet_used'];

        $savedBookingCheckout = BookingCheckout::getBookingCheckout($refId);
        $bookingCheckoutId = $savedBookingCheckout['data']['id'];
        $customerId = $savedBookingCheckout['data']['customer_id'];
        $bookingDetails = json_decode($savedBookingCheckout['data']['booking_details'], true);
        $checkoutIds = json_decode($savedBookingCheckout['data']['checkout_ids'], true);

        // $initialAmount = $savedBookingCheckout['data']['initial_amount'];
        $initialAmount = $bookingDetails['processedPrice']['grandTotal'];
        $bookingAmount = $initialAmount;
        if ($walletUsed == true) {
            $bookingAmount = $bookingAmount - $walletAmount;
        }
        $bookingDetails['paymentDetails']['wallet_details'] = [
            'status' => $walletUsed,
            'amount' => ($walletUsed ? $walletAmount : 0)
        ];
        if ($walletUsed === false) {
            if (!empty($bookingDetails['processedPrice']['walletUsage'])) {
                $walletIds = array_filter(array_column($bookingDetails['processedPrice']['walletUsage'], 'wallet_id'));

                if (!empty($walletIds)) {
                    Wallet::whereIn('id', $walletIds)->update([
                        'wallet_used_status' => null,
                        'used_temp_amount' => null,
                        'temp_amount_currency' => null,
                        'wallet_used_amount' => null,
                        'converted_used_amount' => null,
                    ]);
                }
            }


            $bookingDetails['processedPrice']['walletUsage'] = [];
        }

        $maxAttempts = 3;
        $checkoutId = '';
        $paymentPayload = $this->getHotelCheckoutPayload($paymentBrand, $bookingAmount, $requestData['currency'], $bookingDetails, $refId);

        for ($currentAttempt = 1; $currentAttempt <= $maxAttempts; $currentAttempt++) {
            $checkoutDetails = $this->getPaymentCheckoutId($paymentPayload);
            if ($checkoutDetails['status'] == 'true') {
                $checkoutId = $checkoutDetails['data']['id'];
                break;
            }
        }

        if ($checkoutId != '') {
            $paymentCheckoutIdLog = PaymentCheckoutIdLog::createLog([
                'ref_id' => $refId,
                'checkout_id' => $checkoutId,
                'payment_brand' => $paymentBrand,
                'amount' => $bookingAmount,
                'request' => json_encode($checkoutDetails['request']),
                'response' => json_encode($checkoutDetails['response'])
            ]);

            $checkoutIds = $this->getCheckoutIdList();
            foreach ($checkoutIds as $key => $value) {
                $checkoutIds[$key] = '';
            }
            $checkoutIds[$paymentBrand] = $checkoutId;

            $processedPrice = $this->getProcessedPrice($requestData['currency'], $customerId, $bookingDetails, $initialLoad = false, $requestData);
            $bookingDetails = $processedPrice['booking_details'];
            unset($processedPrice['booking_details']);
            $bookingDetails = array_merge($bookingDetails, [
                'processedPrice' => $processedPrice
            ]);
            $bookingDetails = json_encode($bookingDetails);
            $bookingPayload = [
                'checkout_ids' => json_encode($checkoutIds),
                'amount' => $bookingAmount,
                'booking_details' => $bookingDetails
            ];
            $savedCheckoutDetails = BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);

            if ($savedCheckoutDetails['status'] == 'true') {
                $result = [
                    'checkout_ids' => $checkoutIds,
                    'processed_price' => $processedPrice,
                ];
                return $this->sendResponse($result, 'Wallet Applied successfully');
            } else {
                return $this->sendResponse('', 'Unable to apply wallet. Please try again.', false);
            }
        }
        return $this->sendResponse('', 'Unable to process payment. Please try again.', false);
    }

    public function applyRedeemPoints(Request $request)
    {
        $requestData = $request->all();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);

        $refId = $requestData['ref_id'];
        $paymentBrand = $requestData['payment_brand'];
        $redeemAmount = $requestData['redeem_amount'];
        $redeemPointsUsed = $requestData['redeem_points_used'];

        $savedBookingCheckout = BookingCheckout::getBookingCheckout($refId);
        $bookingCheckoutId = $savedBookingCheckout['data']['id'];
        $customerId = $savedBookingCheckout['data']['customer_id'];
        $bookingDetails = json_decode($savedBookingCheckout['data']['booking_details'], true);
        $checkoutIds = json_decode($savedBookingCheckout['data']['checkout_ids'], true);

        $initialAmount = $savedBookingCheckout['data']['initial_amount'];
        $bookingAmount = $initialAmount;
        if ($redeemPointsUsed == true) {
            $bookingAmount = $bookingAmount - $redeemAmount;
        }

        if ($redeemPointsUsed == false) {
            $existingReedeemDetails = $bookingDetails['processedPrice']['loyaltyPointsUsage'];

            if (!empty($existingReedeemDetails)) {
                $redeemIds = array_column($existingReedeemDetails, 'id');

                CustomerLoyaltyPoints::whereIn('id', $redeemIds)->update([
                    'points_redeemed' => 0,
                    'used_status' => 'not_used',
                    'redeem_amount' => 0,
                ]);
            }
        }

        $bookingDetails['paymentDetails']['redeem_details'] = [
            'status' => $redeemPointsUsed,
            'amount' => ($redeemPointsUsed ? $redeemAmount : 0)
        ];

        $maxAttempts = 3;
        $checkoutId = '';
        $paymentPayload = $this->getHotelCheckoutPayload($paymentBrand, $bookingAmount, $requestData['currency'], $bookingDetails, $refId);
        for ($currentAttempt = 1; $currentAttempt <= $maxAttempts; $currentAttempt++) {
            $checkoutDetails = $this->getPaymentCheckoutId($paymentPayload);
            if ($checkoutDetails['status'] == 'true') {
                $checkoutId = $checkoutDetails['data']['id'];
                break;
            }
        }
        if ($checkoutId != '') {
            $paymentCheckoutIdLog = PaymentCheckoutIdLog::createLog([
                'ref_id' => $refId,
                'checkout_id' => $checkoutId,
                'payment_brand' => $paymentBrand,
                'amount' => $bookingAmount,
                'request' => json_encode($checkoutDetails['request']),
                'response' => json_encode($checkoutDetails['response'])
            ]);

            $checkoutIds = $this->getCheckoutIdList();
            foreach ($checkoutIds as $key => $value) {
                $checkoutIds[$key] = '';
            }
            $checkoutIds[$paymentBrand] = $checkoutId;

            $processedPrice = $this->getProcessedPrice($requestData['currency'], $customerId, $bookingDetails, $initialLoad = false, $requestData);
            $bookingDetails = $processedPrice['booking_details'];
            unset($processedPrice['booking_details']);
            $bookingDetails = array_merge($bookingDetails, [
                'processedPrice' => $processedPrice
            ]);
            $bookingDetails = json_encode($bookingDetails);
            $bookingPayload = [
                'checkout_ids' => json_encode($checkoutIds),
                'amount' => $bookingAmount,
                'booking_details' => $bookingDetails
            ];
            $savedCheckoutDetails = BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);

            if ($savedCheckoutDetails['status'] == 'true') {
                $result = [
                    'checkout_ids' => $checkoutIds,
                    'processed_price' => $processedPrice,
                ];
                return $this->sendResponse($result, 'Points redeemed successfully');
            } else {
                return $this->sendResponse([], 'Unable to redeem your points. Please try again.', false);
            }
        }
        return $this->sendResponse([], 'Something Went wrong. Please try again.', false);
    }

    public function getCustomerWalletBalance($customerId)
    {
        $walletCredited = Wallet::getTotalWalletAmount($customerId);

        // $walletDebited = Wallet::where('customer_id', $customerId)
        //     ->where('type', 'debit')
        //     ->where('status', 'active')
        //     ->where('expiry_date', '>=', date('Y-m-d'))
        //     ->sum('balance');
        return $walletCredited;
    }
    public function getCustomerWallet($customerId)
    {
        $walletCredited = Wallet::where('customer_id', $customerId)
            ->where('type', 'credit')
            ->where('status', 'active')
            ->where('expiry_date', '>=', now()->toDateString())
            ->orderBy('expiry_date', 'asc')
            ->get();

        // $walletCredited = $walletCredited->map(function ($wallet) use ($currency) {
        //     $bookings = Bookings::where('id', $wallet->booking_id)->first();
        //     if ($bookings) {
        //         $convertedCurrency = convertCurrencyExchangeRate(
        //             $wallet->balance,
        //             $currency,
        //             $bookings->supplier_currency,

        //         );
        //         $wallet->balance = $convertedCurrency['data']['convertedRate'];
        //     }

        //     return $wallet;
        // });

        return $walletCredited->isEmpty() ? collect() : $walletCredited;
    }


    public function getProcessedPrice($currency, $customerId, $bookingDetails, $initialLoad = false, $requestData)
    {

        // dd($bookingDetails);
        $isAddOnUsed = $isCouponUsed = $isRedeemUsed = $isWalletUsed = $isBagUsed = false;
        $addOnAmount = $couponAmount = $redeemAmount = $walletAmount = $bagAmount  = 0;
        $initialPrice = $usedWalletAmount = $remainingWalletAmount = $travellerCount =  $markupFee = 0;
        $walletUsage = [];

        // $supplierTypeId = $bookingDetails['bookingDetails']['service_type_id'];
        // $defaultMarkup = DefaultMarkup::where('service_type_id', $supplierTypeId)->first();
        // $markupType = $defaultMarkup['b2c_markup_type'];
        // $markupValue = $defaultMarkup['b2c_markup'];
        $generalVATPercentage = Setting::where('config_key', 'general|site|defaultVatPercentage')->get('value')[0]['value'];
        $bookingType = $bookingDetails['bookingDetails']['bookingType'];
        $chargeableSeatPrice = 0;
        if ($bookingDetails['bookingDetails']['bookingType'] == ServiceType::Flight->value) {
            $loyaltyPointsService = 'Airline Ticket';

            $travelersTotalPriceSum = $travelersBasePriceSum = $totalTaxAmount = 0;
            if (isset($bookingDetails['flightDetails']['flightInfo'][0]['additionalServices']) && is_array($bookingDetails['flightDetails']['flightInfo'][0]['price']['additionalServices'])) {
                foreach ($bookingDetails['flightDetails']['flightInfo'] as $key => $flightDetail) {
                    foreach ($flightDetail['price']['additionalServices'] as $additionalService) {
                        if ($additionalService['type'] == 'SEATS') {
                            $chargeableSeatPrice += $additionalService['amount'];
                        }
                    }
                }
            }

            $travelersTotalPriceSum = $chargeableSeatPrice;
            // if ($bookingDetails['flightDetails']['additionalDetails']['addBags']['status'] == true) {
            //     //$bagPrice = $bookingDetails['flightDetails']['additionalDetails']['addBags']['selectedBagDetails']['price']['amount'];
            //     $bagPrice = 0;
            //     $travelersTotalPriceSum += $bagPrice;
            // }

            $processedTravelerIds = [];

            foreach ($bookingDetails['flightDetails']['flightInfo'] as $i => &$flightDetail) {
                foreach ($flightDetail['travelerPricings'] as $j => &$travelerPricing) {
                    $taxAmount = 0;

                    $travelerId = $travelerPricing['travelerId'] ?? null;

                    if ($travelerId && !in_array($travelerId, $processedTravelerIds)) {
                        $travellerCount++;
                        $processedTravelerIds[] = $travelerId;
                    }

                    if (isset($travelerPricing['price']['taxes'])) {
                        foreach ($travelerPricing['price']['taxes'] as $tax) {
                            $taxAmount += $tax['amount'];
                        }
                    }

                    $totalPrice = $travelerPricing['price']['initialTotal'] ?? 0;
                    $basePrice = $travelerPricing['price']['base'] ?? 0;

                    $travelersTotalPriceSum += $totalPrice;
                    $travelersBasePriceSum += $basePrice;
                    $totalTaxAmount += $taxAmount;

                    $bookingDetails['flightDetails']['flightInfo'][$i]['travelerPricings'][$j]['price']['totalTaxAmount'] = $taxAmount;
                }
            }


            $finalPrice = $travelersBasePriceSum;
        } elseif ($bookingDetails['bookingDetails']['bookingType'] == ServiceType::Hotel->value) {
            $loyaltyPointsService = 'Hotel Booking';
            $initialPrice = $bookingDetails['hotelDetails']['selectedRooms']['beforePrice'] ?? 0;

            // $travelersBasePriceSum = $finalPrice = $bookingDetails['hotelDetails']['selectedRooms']['Price']['FinalPrice'];
            $travelersBasePriceSum = $finalPrice = $bookingDetails['hotelDetails']['selectedRooms']['beforePrice'] ?? 0;

            if (array_key_exists('FinalTax', $bookingDetails['hotelDetails']['selectedRooms']['Price'])) {
                $totalTaxAmount = $bookingDetails['hotelDetails']['selectedRooms']['Price']['FinalTax'];
            } else {
                $totalTaxAmount = 0;
            }
            $totalTaxAmount = 0;
            $travelersTotalPriceSum = $travelersBasePriceSum;
        }

        $serviceFeeDetails = getServiceFee($bookingDetails['bookingDetails']['bookingType']);
        $vat = $serviceFeeDetails['vat'];
        $serviceFee = $serviceFeeDetails['serviceFee'];

        $totalInclusiveTax = (float)  $travelersTotalPriceSum + (float) $serviceFee;



        $markupPrice = $this->getMarkupProcessedPrice($bookingDetails, $travelersTotalPriceSum, $requestData);
        $beforePrice =  $bookingDetails['bookingDetails']['bookingType'] == ServiceType::Hotel->value ? $initialPrice : $travelersTotalPriceSum;
        $defaultMarkup = 'yes';
        $markupFee = 0;
        $service_calculation_method = '';
        $ancillary_service_markup_type = $ancillary_service_markup = 0;
        if (!empty($markupPrice)) {
            if ($bookingType == ServiceType::Hotel->value) {
                $finalPrice = $finalPrice;
                $markupValue = $markupPrice['markupValue'];
                $defaultMarkup = $markupPrice['defaultMarkup'];
                $beforePrice = $initialPrice;
                $markupType = $markupPrice['markupType'];
                $markupFee = $markupPrice['markupFee'];
                $service_calculation_method = $markupPrice['service_calculation_method'];
                $totalInclusiveTax += $markupFee;
            } elseif ($bookingType   == ServiceType::Flight->value) {
                $finalPrice = $markupPrice['totalPrice'];
                $markupValue = $markupPrice['markupValue'];
                $defaultMarkup = $markupPrice['defaultMarkup'];
                $beforePrice = $markupPrice['beforePrice'];
                $markupType = $markupPrice['markupType'];
                $markupFee = $markupPrice['markupFee'];
                $totalInclusiveTax += $markupFee;
                $service_calculation_method = $markupPrice['service_calculation_method'];
                $ancillary_service_markup_type = $markupPrice['ancillary_service_markup_type'];
                $ancillary_service_markup = $markupPrice['ancillary_service_markup'];
            }
        }


        if ($bookingType == ServiceType::Hotel->value) {
            $travelersTotalPriceSum += $markupFee;
            $finalPrice = round($travelersTotalPriceSum, 2);
            // $totalTaxAmount = $markupFee;
        }


        if (array_key_exists('addon_details', $bookingDetails['paymentDetails']) && $bookingDetails['paymentDetails']['addon_details']['status']) {
            $isAddOnUsed = true;
            $addOnAmount = $bookingDetails['paymentDetails']['addon_details']['amount'];
            $totalInclusiveTax = $totalInclusiveTax + $bookingDetails['paymentDetails']['addon_details']['amount'];
            // if ($bookingType == ServiceType::Flight->value) {
            //     $totalInclusiveTax -= $markupFee;
            // }
        }
        $selectedbagsCount = 0;
        $baggageMarkup = 0.0;
        $totalBaggageMarkup = 0.0;
        $baggageMarkupFee = 0.0;

        if (
            isset($bookingDetails['flightDetails']['additionalDetails']['addBags']['status']) &&
            $bookingDetails['flightDetails']['additionalDetails']['addBags']['status'] &&
            !empty($bookingDetails['flightDetails']['additionalDetails']['addBags']['selectedBagDetails']) &&
            is_array($bookingDetails['flightDetails']['additionalDetails']['addBags']['selectedBagDetails'])
        ) {
            $bagAmount = 0.0;
            $processedTravelerIds = []; // Track already processed travelerIds

            foreach ($bookingDetails['flightDetails']['additionalDetails']['addBags']['selectedBagDetails'] as $bagDetail) {

                if (
                    empty($bagDetail['bag']) ||
                    !isset($bagDetail['bag']['price']['amount']) ||
                    !isset($bagDetail['travelerId'])
                ) {
                    continue;
                }

                $travelerId = $bagDetail['travelerId'];

                if (in_array($travelerId, $processedTravelerIds)) {
                    continue; // Skip if already processed
                }

                $priceAmount = (float) $bagDetail['bag']['price']['amount'];
                $selectedbagsCount++;
                $bagAmount += $priceAmount;
                $processedTravelerIds[] = $travelerId;
            }

            $baggageMarkupFee = $ancillary_service_markup_type == 'fixed_amount'
                ? $ancillary_service_markup
                : (($bagAmount * $ancillary_service_markup) / 100);

            // $baggageMarkup = ($selectedbagsCount > 0)
            //     ? (($selectedbagsCount == 1) ? $baggageMarkupFee : ((float) $baggageMarkupFee / $selectedbagsCount))
            //     : 0.0;

            $totalBaggageMarkup = ($selectedbagsCount > 0) ? $baggageMarkupFee : 0.0;

            $totalInclusiveTax += $bagAmount;
        }




        $selectedSeatsTotalPrice = 0;
        $isSeatUsed = false;
        $selectedSeatsCount = 0;
        $seatMarkup = $seatMarkupFee = 0.0;
        $totalSeatMarkup = 0.0;

        $markupFee = $markupFee ?? 0.0;
        $totalInclusiveTax = $totalInclusiveTax ?? 0.0;

        $seatDetails = $bookingDetails['flightDetails']['seatDetails'] ?? [];


        if (!empty($seatDetails) && is_array($seatDetails) && !empty($seatDetails['status'])) {

            $isSeatUsed = true;
            $seatSegments = $seatDetails['seatDetails'] ?? [];

            if (is_array($seatSegments)) {
                foreach ($seatSegments as $traveler) {
                    $price = $traveler['price_details']['allPriceDetails'][0]['price']['total'] ?? '';

                    if ($price !== '' && is_numeric($price)) {
                        $selectedSeatsCount++;
                        $selectedSeatsTotalPrice += (float) $price;
                    }
                }

                if ($selectedSeatsCount > 0) {
                    $seatMarkupFee = $ancillary_service_markup_type === 'fixed_amount'
                        ? $ancillary_service_markup
                        : (($selectedSeatsTotalPrice * $ancillary_service_markup) / 100);

                    $totalSeatMarkup = $seatMarkupFee;
                    $totalInclusiveTax += $selectedSeatsTotalPrice;
                }
            }
        }




        if (array_key_exists('coupon_details', $bookingDetails['paymentDetails']) && $bookingDetails['paymentDetails']['coupon_details']['status']) {
            $isCouponUsed = true;
            $couponAmount = $bookingDetails['paymentDetails']['coupon_details']['amount'];
            $totalInclusiveTax = $totalInclusiveTax - $bookingDetails['paymentDetails']['coupon_details']['amount'];
        }
        $redeemPoint = $expiryDate = $expiryFromArrival = 0;
        $redeemUsageDetails = [];

        if (
            array_key_exists('redeem_details', $bookingDetails['paymentDetails']) &&
            $bookingDetails['paymentDetails']['redeem_details']['status']
        ) {
            $isRedeemUsed = true;
            $redeemAmount = $bookingDetails['paymentDetails']['redeem_details']['amount'];
            $convertedRedeemAmount = convertCurrencyExchangeRate($redeemAmount, $currency, Currency::SAR->value)['data']['convertedRate'];

            $pointToAmountRate = LoyaltyPointSetting::first()?->point_to_amount ?? 1;
            $totalPointsToRedeem = $convertedRedeemAmount * $pointToAmountRate;

            CustomerLoyaltyPoints::where('customer_id', $customerId)
                ->where('used_status', '=', 'inprogress')
                ->update([
                    'used_status' => 'not_used',
                    'points_redeemed' => 0,
                    'redeem_amount' => 0
                ]);

            $customerLoyaltyPoints = CustomerLoyaltyPoints::getLoyaltyPoints($customerId);
            $updates = [];
            $redeemUsageDetails = [];

            foreach ($customerLoyaltyPoints as $record) {
                $availablePoints = $record->points_earned - $record->points_redeemed;
                if ($availablePoints <= 0) continue;

                $pointsToUse = min($availablePoints, $totalPointsToRedeem);
                $redeemedThisTime = round($pointsToUse / $pointToAmountRate, 2);

                $updates[] = [
                    'id' => $record->id,
                    'points_redeemed' => $record->points_redeemed + $pointsToUse,
                    'redeem_amount' => $redeemedThisTime,
                    'used_status' => ($record->points_earned - ($record->points_redeemed + $pointsToUse) <= 0) ? 'inprogress' : 'not_used'
                ];

                $redeemUsageDetails[] = [
                    'id' => $record->id,
                    'points_used' => $pointsToUse,
                    'redeem_amount' => $redeemedThisTime,
                    'service_end_date' => $record->service_end_date,
                    'expiry_date' => $record->expiry_date
                ];

                $totalPointsToRedeem -= $pointsToUse;
                if ($totalPointsToRedeem <= 0) break;
            }

            foreach ($updates as $update) {
                CustomerLoyaltyPoints::where('id', $update['id'])->update([
                    'points_redeemed' => $update['points_redeemed'],
                    'redeem_amount' => $update['redeem_amount'],
                    'used_status' => $update['used_status']
                ]);
            }

            $totalInclusiveTax -= $redeemAmount;
        }



        if (array_key_exists('wallet_details', $bookingDetails['paymentDetails']) && $bookingDetails['paymentDetails']['wallet_details']['status']) {
            $isWalletUsed = true;
            $targetCurrency = $currency;
            $walletAmount = $bookingDetails['paymentDetails']['wallet_details']['amount'];

            $existingWallets = $this->getCustomerWallet($customerId);
            $remainingWalletAmount = $walletAmount;
            $walletUsage = [];

            foreach ($existingWallets as $wallet) {
                if ($remainingWalletAmount <= 0) {
                    break;
                }

                $converted = convertCurrencyExchangeRate(
                    $wallet->balance,
                    $wallet->currency,
                    $targetCurrency
                );

                $convertedBalance = $converted['data']['convertedRate'];

                $convertedUsedAmount = min($convertedBalance, $remainingWalletAmount);


                $originalUsedAmount = convertCurrencyExchangeRate(
                    $convertedUsedAmount,
                    $targetCurrency,
                    $wallet->currency
                )['data']['convertedRate'];

                // Update wallet
                Wallet::where('id', $wallet->id)->update([
                    'wallet_used_status' => 'inprogress',
                    'used_temp_amount' => $originalUsedAmount,
                    'converted_used_amount' => $convertedUsedAmount,
                    'temp_amount_currency' => $targetCurrency,
                ]);

                $walletUsage[] = [
                    'wallet_id' => $wallet->id,
                    'used_amount' => $convertedUsedAmount,
                    'original_used' => $originalUsedAmount,
                    'wallet_currency' => $wallet->currency,
                    'converted_currency' => $targetCurrency
                ];

                $remainingWalletAmount -= $convertedUsedAmount;
            }

            if ($remainingWalletAmount > 0) {
                return $this->sendResponse([], 'Insufficient wallet balance');
            }


            $totalUsedFromWallet = array_sum(array_column($walletUsage, 'used_amount'));
            $totalInclusiveTax = $totalInclusiveTax - $totalUsedFromWallet;
        }

        // $travelersBasePriceSum = (float)$travelersBasePriceSum / (float)(1 + ($generalVATPercentage / 100));

        $payload = [
            'service' => $loyaltyPointsService,
            'amount' =>  $bookingType == ServiceType::Flight->value ? round($travelersTotalPriceSum + $markupFee, 2) :  round($travelersTotalPriceSum, 2),
            'fromCurrency' => Currency::SAR->value,
            'toCurrency' => $currency
        ];
        $loyaltyPoints = $this->getLoyaltyPoints($payload);
        $redeemDetails = [];
        if ($customerId != '') {
            $redeemDetails = $this->getCustomerMaxRedeemPoints($customerId, $bookingType, $currency);
        }

        // $travelersTotalPriceSum = (float) $travelersTotalPriceSum - (float) $chargeableSeatPrice;



        $travellerServiceFee = 0;
        if ($bookingType == ServiceType::Flight->value) {

            $travellerMarkupFee = ($travellerCount > 0)
                ? (($travellerCount == 1) ? $markupFee : ((float) $markupFee / $travellerCount))
                : 0.0;
            if (isset($bookingDetails['searchDetails']['isSeparateFlight'])) {
                if ($bookingDetails['searchDetails']['isSeparateFlight'] === "true") {
                    $totalInclusiveTax -= $travellerMarkupFee;
                    $totalInclusiveTax += $travellerMarkupFee * ($travellerCount + 1);
                }
            }

            if ($service_calculation_method == 'per_passenger' && $service_calculation_method != '') {
                // $travellerServiceFee = ($travellerCount > 0)
                // ? (($travellerCount == 1) ? $serviceFee : ((float) $serviceFee / $travellerCount))
                // : 0.0;
                $travellerServiceFee = $serviceFee;
                $totalInclusiveTax -= $serviceFee;
                $serviceFee = $serviceFee * $travellerCount;
                $totalInclusiveTax += $serviceFee;
            }
            if ($service_calculation_method == 'from_total' && $service_calculation_method != '') {
                $travellerServiceFee = 0;
            }

            foreach ($bookingDetails['flightDetails']['flightInfo'] as &$flightDetail) {
                foreach ($flightDetail['travelerPricings'] as &$travelerPricing) {
                    $existingTax = $travelerPricing['price']['totalTaxAmount'] ?? 0.0;

                    $newTax = $existingTax + $travellerMarkupFee + $totalSeatMarkup + $totalBaggageMarkup;

                    $total = $travelerPricing['price']['total'];
                    if (array_key_exists('initialTotal', $travelerPricing['price'])) {
                        $total = $travelerPricing['price']['initialTotal'];
                    }

                    $travelerPricing['price']['totalTaxAmount'] = $newTax;
                    $travelerPricing['price']['markupValue'] = $travellerMarkupFee;
                    $travelerPricing['price']['serviceFee'] = $travellerServiceFee;
                    $travelerPricing['price']['serviceFeeType'] = $service_calculation_method ?? '';
                    $travelerPricing['price']['total'] = $total + $travellerMarkupFee + $totalSeatMarkup + $totalBaggageMarkup;

                    $totalTaxAmount += $travellerMarkupFee + $totalSeatMarkup + $totalBaggageMarkup;
                    $totalInclusiveTax += $totalSeatMarkup + $totalBaggageMarkup;
                    // if ($travellerCount > 1) {
                    //     $travelerPricing['price']['total'] = $travelerPricing['price']['base'];
                    // }
                }
                unset($travellerPricing);
            }
            unset($flightDetail);
        }


        // Cancellation Policy - Service Fee Apply
        if ($bookingType == ServiceType::Hotel->value) {
            $cancellationPolicyDetails = $bookingDetails['hotelDetails']['cancellationPolicy'] ?? [];

            if (!empty($cancellationPolicyDetails)) {
                $cancellationPolicies = $cancellationPolicyDetails['CancellationPolicies'] ?? [];

                if (!empty($cancellationPolicies) && is_array($cancellationPolicies)) {
                    foreach ($cancellationPolicies as &$policy) {
                        if (isset($policy['Refundable']) && !$policy['Refundable']) {
                            $policy['Amount'] = $totalInclusiveTax ?? 0;
                        }
                    }
                    unset($policy);
                }
                $cancellationPolicyDetails['CancellationPolicies'] = $cancellationPolicies;
                $bookingDetails['hotelDetails']['cancellationPolicy'] = $cancellationPolicyDetails;
            }
        }



        $processedPrice = [
            'basePriceRemovedVAT' => $travelersBasePriceSum,
            'finalPrice' => $finalPrice,
            'currency' => $currency,
            'markupFee' => $markupFee,
            'serviceFeeType' => $service_calculation_method ?? '',
            'serviceFee' => $serviceFee,
            'markupServiceFee' => $serviceFee,
            'vatPercentage' => $bookingType == ServiceType::Hotel->value ? 0 : $generalVATPercentage,
            // 'vat' => round($vat, '2', '.', ''),
            'vat' => $bookingType == ServiceType::Hotel->value ? 0 : $vat,
            'chargeableSeatPrice' => $chargeableSeatPrice,
            'totalPrice' => $travelersTotalPriceSum,
            'grandTotal' => $totalInclusiveTax,
            'defaultMarkup' => $defaultMarkup,
            'beforePrice' => $beforePrice,
            'markupType' => $markupType,
            'markupValue' => $markupValue ?? 0,
            'totalTaxAmount' => $bookingType == ServiceType::Hotel->value ? 0 : $totalTaxAmount,
            'loyaltyPoints' => $loyaltyPoints,
            'redeemDetails' => $redeemDetails,
            'walletBalanceUsed' => 0,
            'isBagUsed' => $bagAmount == 0  ? false : true,
            'bagAmount' => $bagAmount,
            'bagMarkup' => $totalBaggageMarkup ?? 0,
            'bagCount' => $selectedbagsCount ?? 0,
            'isSeatUsed' => $isSeatUsed,
            'seatMarkup' => $totalSeatMarkup ?? 0,
            'selectedSeatsCount' => $selectedSeatsCount ?? 0,
            'selectedSeatsTotalPrice' => $selectedSeatsTotalPrice,
            'isAddOnUsed' => $isAddOnUsed,
            'addOnAmount' => $addOnAmount,
            'isCouponUsed' => $isCouponUsed,
            'couponAmount' => $couponAmount,
            'isRedeemUsed' => $isRedeemUsed,
            'redeemAmount' => $redeemAmount,
            'isWalletUsed' => $isWalletUsed,
            'walletAmount' => $walletAmount,
            'walletUsage' => $walletUsage,
            'loyaltyPointsUsage' => $redeemUsageDetails,
            'isPriceChanged' => $bookingDetails['processedPrice']['isPriceChanged'] ?? false,
            'existingPrice' => $bookingDetails['processedPrice']['existingPrice'] ?? $totalInclusiveTax,
            'newPrice' => $bookingDetails['processedPrice']['newPrice'] ?? $totalInclusiveTax,
            'booking_details' => $bookingDetails,
        ];
        return $processedPrice;
    }

    public function getFlightProcessedPrice($currency, $customerId, $bookingDetails)
    {
        $isAddOnUsed = $isCouponUsed = $isRedeemUsed = $isWalletUsed = false;
        $addOnAmount = $couponAmount = $redeemAmount = $walletAmount = 0;

        $supplierTypeId = $bookingDetails['bookingDetails']['service_type_id'];
        $defaultMarkup = DefaultMarkup::where('service_type_id', $supplierTypeId)->first();
        $markupType = $defaultMarkup['b2c_markup_type'];
        $markupValue = $defaultMarkup['b2c_markup'];
        $generalVATPercentage = Setting::where('config_key', 'general|site|defaultVatPercentage')->get('value')[0]['value'];
        $generalServiceFee = Setting::where('config_key', 'general|site|flightServiceFee')->get('value')[0]['value'];
        $loyaltyPointsService = 'Airline Ticket';

        $travelersBasePriceSum = $finalPrice = $bookingDetails['flightDetails']['flightInfo'][0]['processedPrice']['travelersPrice'];
        $totalTaxAmount = $bookingDetails['flightDetails']['flightInfo'][0]['processedPrice']['totalTaxAmount'];
        $chargeableSeatPrice = $bookingDetails['flightDetails']['flightInfo'][0]['processedPrice']['chargeableSeatPrice'];
        $travelersTotalPriceSum = $travelersBasePriceSum + $totalTaxAmount;
        //$travelersTotalPriceSum = $travelersTotalPriceSum - $chargeableSeatPrice;

        if (array_key_exists('addon_details', $bookingDetails['paymentDetails']) && $bookingDetails['paymentDetails']['addon_details']['status']) {
            $isAddOnUsed = true;
            $addOnAmount = $bookingDetails['paymentDetails']['addon_details']['amount'];
            $travelersTotalPriceSum = $travelersTotalPriceSum + $bookingDetails['paymentDetails']['addon_details']['amount'];
        }

        //remove vat from traveler's base price
        $baseRemovedVAT = $travelersBasePriceSum / (1 + ($generalVATPercentage / 100));
        $travelerVAT = $travelersBasePriceSum * $generalVATPercentage / 100;
        $travelersBasePriceSum = $baseRemovedVAT;

        $markupFee = ($markupType == 'percentage') ? $travelersTotalPriceSum * $markupValue / 100 : $markupValue;
        $markupServiceFee = $markupFee + $generalServiceFee;
        $vat = $markupServiceFee * $generalVATPercentage / 100;
        $totalInclusiveTax = $travelersTotalPriceSum  + $chargeableSeatPrice + $markupServiceFee + $vat;

        if (array_key_exists('coupon_details', $bookingDetails['paymentDetails']) && $bookingDetails['paymentDetails']['coupon_details']['status']) {
            $isCouponUsed = true;
            $couponAmount = $bookingDetails['paymentDetails']['coupon_details']['amount'];
            $totalInclusiveTax = $totalInclusiveTax - $bookingDetails['paymentDetails']['coupon_details']['amount'];
        }
        if (array_key_exists('redeem_details', $bookingDetails['paymentDetails']) && $bookingDetails['paymentDetails']['redeem_details']['status']) {
            $isRedeemUsed = true;
            $redeemAmount = $bookingDetails['paymentDetails']['redeem_details']['amount'];
            $totalInclusiveTax = $totalInclusiveTax - $bookingDetails['paymentDetails']['redeem_details']['amount'];
        }
        if (array_key_exists('wallet_details', $bookingDetails['paymentDetails']) && $bookingDetails['paymentDetails']['wallet_details']['status']) {
            $isWalletUsed = true;
            $walletAmount = $bookingDetails['paymentDetails']['wallet_details']['amount'];
            $totalInclusiveTax = $totalInclusiveTax - $bookingDetails['paymentDetails']['wallet_details']['amount'];
        }

        $payload = [
            'service' => $loyaltyPointsService,
            'amount' => $totalInclusiveTax,
            'fromCurrency' => Currency::SAR->value,
            'toCurrency' => $currency
        ];
        $loyaltyPoints = $this->getLoyaltyPoints($payload);
        $redeemDetails = [];
        if ($customerId != '') {
            $redeemDetails = $this->getCustomerMaxRedeemPoints($customerId, $bookingDetails['bookingDetails'], $currency);
        }

        $processedPrice = [
            'basePriceRemovedVAT' => $travelersBasePriceSum,
            'finalPrice' => $finalPrice,
            'currency' => $currency,
            'markupFee' => round($markupFee, 2),
            'serviceFee' => round($generalServiceFee,  2),
            'markupServiceFee' => round($markupServiceFee,  2),
            'vatPercentage' => $generalVATPercentage,
            'vat' => round($vat, '2', '.', ''),
            'chargeableSeatPrice' => round($chargeableSeatPrice,  2),
            'totalPrice' => round($travelersTotalPriceSum,  2),
            'grandTotal' => round($totalInclusiveTax,  2),
            'markupType' => $markupType,
            'markupValue' => $markupValue,
            'totalTaxAmount' => $totalTaxAmount,
            'loyaltyPoints' => $loyaltyPoints,
            'redeemDetails' => $redeemDetails,
            'walletBalanceUsed' => 0,

            'isAddOnUsed' => $isAddOnUsed,
            'addOnAmount' => $addOnAmount,
            'isCouponUsed' => $isCouponUsed,
            'couponAmount' => $couponAmount,
            'isRedeemUsed' => $isRedeemUsed,
            'redeemAmount' => $redeemAmount,
            'isWalletUsed' => $isWalletUsed,
            'walletAmount' => $walletAmount
        ];
        return $processedPrice;
    }

    public function getCheckoutId(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'ref_id' => 'required',
        ]);
        if ($validator->fails()) {
            return $this->sendError('Invalid Request', ['error' => $validator->errors()], 422);
        }
        $requestData = $request->all();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);
        $requestData['booking_via'] = $request->input('booking_via', '');
        $requestData['type'] = $request->input('type', '');
        $requestData['customer_id'] = $request->input('customer_id', '');
        $refId = $request->input('ref_id', '');
        $savedBookingCheckout = BookingCheckout::getBookingCheckout($refId);
        if (!$savedBookingCheckout['status']) {
            return $this->sendError('Something went wrong. Please try again', [], 422);
        }
        $bookingCheckoutId = $savedBookingCheckout['data']['id'];
        $customerId = $savedBookingCheckout['data']['customer_id'];
        $bookingDetails = json_decode($savedBookingCheckout['data']['booking_details'], true);
        if (!$customerId && $requestData['customer_id']) {
            $customerData = Customer::select('first_name', 'last_name', 'email')->where('id', $requestData['customer_id'])->first();
            if ($customerData) {
                $bookingDetails['userDetails']['first_name'] = $bookingDetails['userDetails']['first_name'] ? $bookingDetails['userDetails']['first_name'] : $customerData?->first_name;
                $bookingDetails['userDetails']['last_name'] = $bookingDetails['userDetails']['last_name'] ? $bookingDetails['userDetails']['last_name'] : $customerData?->last_name;
                $bookingDetails['userDetails']['email'] = $bookingDetails['userDetails']['email'] ? $bookingDetails['userDetails']['email'] : $customerData?->email;
            }
            $bookingDetails['userDetails']['userId'] = $requestData['customer_id'];
            $bookingPayload = [
                'customer_id' => $requestData['customer_id'],
                'booking_details' =>  json_encode($bookingDetails)
            ];
            $savedBookingCheckout = BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);
            $customerId = $savedBookingCheckout['data']['customer_id'];
        }
        $checkoutUrl = $savedBookingCheckout['data']['checkout_url'];

        $isAlreadyBooked = false;
        $isBooked = Bookings::where('ref_id', $refId)->first();
        if ($isBooked) {
            if ($isBooked->booking_status == BookingStatus::Confirmed->value) {
                $isAlreadyBooked = true;
                return $this->sendResponse([
                    'ref_id' => $refId,
                    'isAlreadyBooked' => $isAlreadyBooked,
                ], 'Booking already confirmed');
            } else {
                $isAlreadyBooked = false;
            }
        }

        if ($savedBookingCheckout['data']['booking_type'] == ServiceType::Flight->value) {
            if (
                $bookingDetails['flightDetails']['additionalDetails']['addBags']['status'] == true &&
                $bookingDetails['flightDetails']['additionalDetails']['addBags']['addedToFlightOffer'] == false
            ) {
                $isBagAdded = false;
                $addedBags = $bookingDetails['flightDetails']['additionalDetails']['addBags']['selectedBagDetails'];
                $addedBagsBySegment = [];
                foreach ($addedBags as $addedBag) {
                    $addedBagsBySegment[$addedBag['segmentId']] = $addedBag;
                }
                foreach ($bookingDetails['flightDetails']['flightInfo'] as &$flightOffer) {
                    $isBagAvailable = false;
                    $bagAmount = 0;
                    foreach ($flightOffer['travelerPricings'] as &$travelerPricing) {
                        foreach ($travelerPricing['fareDetailsBySegment'] as &$fareDetailsBySegment) {
                            if (
                                in_array($travelerPricing['travelerId'], $addedBagsBySegment[$fareDetailsBySegment['segmentId']]['bag']['travelerIds']) &&
                                in_array($fareDetailsBySegment['segmentId'], array_keys($addedBagsBySegment))
                            ) {
                                $fareDetailsBySegment['additionalServices'] = [
                                    'chargeableCheckedBags' => [
                                        'quantity' => $addedBagsBySegment[$fareDetailsBySegment['segmentId']]['bag']['quantity']
                                    ]
                                ];
                                $bagAmount = $addedBagsBySegment[$fareDetailsBySegment['segmentId']]['bag']['price']['amount'];
                                $isBagAvailable = true;
                            }
                        }
                    }
                    if ($isBagAvailable) {
                        if (array_key_exists('additionalServices', $flightOffer['price']) == false) {
                            $flightOffer['price']['additionalServices'] = [];
                        }
                        array_push($flightOffer['price']['additionalServices'], [
                            'amount' => $bagAmount,
                            'type' => 'CHECKED_BAGS'
                        ]);
                        $flightOffer['price']['grandTotal'] = $flightOffer['price']['grandTotal'] + $bagAmount;
                        $isBagAdded = true;
                    }
                }
                if ($isBagAdded) {
                    $bookingDetails['flightDetails']['additionalDetails']['addBags']['addedToFlightOffer'] = true;
                    $bookingPayload = ['booking_details' => json_encode($bookingDetails)];
                    BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);
                }
            }
        }

        $walletBalance = 0;
        if ($savedBookingCheckout['data']['customer_id'] != '') {
            $walletBalance = $this->getCustomerWalletBalance($savedBookingCheckout['data']['customer_id']);
            $convertedBalance = convertCurrencyExchangeRate($walletBalance, Currency::SAR->value, $requestData['currency']);
            if ($convertedBalance['status'] == 'true') {
                $walletBalance = $convertedBalance['data']['convertedRate'];
            } else {
                $walletBalance = $walletBalance;
            }
        }

        $paymentBrand = env('HYPER_PAY_DEFAULT_PAYMENT_METHOD');
        $bookingAmount = $savedBookingCheckout['data']['amount'];
        $maxAttempts = 3;
        $checkoutId = '';
        $paymentPayload = $this->getHotelCheckoutPayload($paymentBrand, $bookingAmount, $requestData['currency'], $bookingDetails, $refId, $requestData['booking_via'], $checkoutUrl);
        for ($currentAttempt = 1; $currentAttempt <= $maxAttempts; $currentAttempt++) {
            $checkoutDetails = $this->getPaymentCheckoutId($paymentPayload, $requestData['booking_via']);
            if ($checkoutDetails['status'] == 'true') {
                $checkoutId = $checkoutDetails['data']['id'];
                break;
            }
        }
        if ($checkoutId != '') {
            $paymentCheckoutIdLog = PaymentCheckoutIdLog::createLog([
                'ref_id' => $refId,
                'checkout_id' => $checkoutId,
                'payment_brand' => $paymentBrand,
                'amount' => $bookingAmount,
                'request' => json_encode($checkoutDetails['request']),
                'response' => json_encode($checkoutDetails['response'])
            ]);
            $checkoutIds = $this->getCheckoutIdList();
            $checkoutIds[$paymentBrand] = $checkoutId;
            $processedPrice = $this->getProcessedPrice($requestData['currency'], $customerId, $bookingDetails, $initialLoad = false, $requestData);
            if ($requestData['type'] == 'updatePrice') {
                $processedPrice['isPriceChanged'] = false;
                $processedPrice['existingPrice'] = $processedPrice['grandTotal'];
                $processedPrice['newPrice'] = $processedPrice['grandTotal'];
            }
            $bookingDetails = $processedPrice['booking_details'];
            unset($processedPrice['booking_details']);
            $bookingDetails = json_encode(array_merge($bookingDetails, ['processedPrice' => $processedPrice]));
            $bookingPayload = [
                'checkout_ids' => json_encode($checkoutIds),
                'amount' => $bookingAmount,
                'booking_details' => $bookingDetails
            ];
            $savedCheckoutDetails = BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);

            // Translate

            if ($savedBookingCheckout['data']['booking_type'] == ServiceType::Flight->value) {
                $savedBookingData = json_decode($savedCheckoutDetails['data']['booking_details'], true);
                $bookingDetails = [
                    'booking_details' => [
                        'booking_details' => $savedBookingData,
                    ]
                ];
                $userSelectedLang = $savedBookingData['searchDetails']['languageCode'] ?? Locale::English->value;
                if ($userSelectedLang != $requestData['locale']) {
                    $bookingDetails = $this->translatedBookingDetails($bookingDetails, $requestData['locale'], $savedBookingCheckout['data']['booking_type']);
                    if (!empty($bookingDetails)) {
                        $bookingRefUrl = env('WEBSITE_URL') . $requestData['locale'];
                        $bookingPayload = [
                            'checkout_url' => $bookingRefUrl,
                            'booking_details' =>  json_encode($bookingDetails)
                        ];
                        $savedBookingCheckout =  BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);
                        $savedCheckoutDetails['data'] = $savedBookingCheckout['data'];
                    }
                }
            } else {
                $savedBookingData = json_decode($savedCheckoutDetails['data']['booking_details'], true);
                $userSelectedLang = $savedBookingData['bookingDetails']['customer_language_code'] ?? Locale::English->value;
                if ($userSelectedLang != $requestData['locale']) {
                    $bookingDetails = $this->translatedBookingDetails($savedBookingData, $requestData['locale'], $savedBookingCheckout['data']['booking_type']);

                    if (!empty($bookingDetails)) {
                        $bookingRefUrl = env('WEBSITE_URL') . $requestData['locale'];
                        $bookingPayload = [
                            'checkout_url' => $bookingRefUrl,
                            'booking_details' =>  json_encode($bookingDetails)
                        ];
                        $savedBookingCheckout =  BookingCheckout::updateBookingCheckout($savedBookingCheckout['data']['id'], $bookingPayload);
                        $savedCheckoutDetails['data'] = $savedBookingCheckout['data'];
                    }
                }
            }

            if ($savedCheckoutDetails['status'] == 'true') {
                $result['success'] = true;
                $successUrl = env('BOOKING_PROCESS_URL') . '?ref_id=' . $refId . '&payment_brand=';
                $redeemAvailable = false;
                if (count($processedPrice['redeemDetails']) > 0 && $processedPrice['redeemDetails']['maxRedeemPrice'] > 0) {
                    $redeemAvailable = true;
                }
                $result['data'] = [
                    'ref_id' => $refId,
                    'reference_details' => $savedCheckoutDetails['data'],
                    'checkout_ids' => $checkoutIds,
                    'success_url' => $successUrl,
                    'processed_price' => $processedPrice,
                    'wallet_balance' => $walletBalance,
                    'redeem_available' => $redeemAvailable,
                    'isAlreadyBooked' => false,
                    'paymentCheckoutIdLog' => $paymentCheckoutIdLog
                ];
                return $this->sendResponse($result['data'], 'Checkout Id genereated successfully');
            } else {
                return $this->sendResponse([], 'Unable to process payment. Please try again.', false);
            }
        }
        return $this->sendResponse([], 'Unable to process payment. Please try again.', false);
    }

    public function getCheckoutIds(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'ref_id' => 'required',
        ]);
        if ($validator->fails()) {
            return $this->sendError('Invalid Request', ['error' => $validator->errors()], 422);
        }
        $requestData = $request->all();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);
        $requestData['booking_via'] = $request->input('booking_via', '');

        $refId = $requestData['ref_id'];

        $savedBookingCheckout = BookingCheckout::getBookingCheckout($refId);
        if (!$savedBookingCheckout['status']) {
            return $this->sendError("Something went wrong. please try again", [], 422);
        }
        $bookingCheckoutId = $savedBookingCheckout['data']['id'];
        $bookingDetails = json_decode($savedBookingCheckout['data']['booking_details'], true);
        $checkoutIds = json_decode($savedBookingCheckout['data']['checkout_ids'], true);

        $requestData['customer_id'] = $request->input('customer_id', '');
        $customerId = $savedBookingCheckout['data']['customer_id'];

        if (!$customerId && $requestData['customer_id']) {
            $customerData = Customer::select('first_name', 'last_name', 'email')->where('id', $requestData['customer_id'])->first();
            if ($customerData) {
                $bookingDetails['userDetails']['first_name'] = $bookingDetails['userDetails']['first_name'] ? $bookingDetails['userDetails']['first_name'] : $customerData?->first_name;
                $bookingDetails['userDetails']['last_name'] = $bookingDetails['userDetails']['last_name'] ? $bookingDetails['userDetails']['last_name'] : $customerData?->last_name;
                $bookingDetails['userDetails']['email'] = $bookingDetails['userDetails']['email'] ? $bookingDetails['userDetails']['email'] : $customerData?->email;
            }
            $bookingDetails['userDetails']['userId'] = $requestData['customer_id'];
            $bookingPayload = [
                'customer_id' => $requestData['customer_id'],
                'booking_details' => json_encode($bookingDetails)
            ];
            $savedBookingCheckout =  BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);

            $customerId = $savedBookingCheckout['data']['customer_id'];
        }
        $paymentBrand = $requestData['payment_brand'];
        $bookingAmount = $savedBookingCheckout['data']['amount'];
        $maxAttempts = 3;
        $checkoutId = '';
        $paymentPayload = $this->getHotelCheckoutPayload($paymentBrand, $bookingAmount, $requestData['currency'], $bookingDetails, $refId, $requestData['booking_via']);
        for ($currentAttempt = 1; $currentAttempt <= $maxAttempts; $currentAttempt++) {
            $checkoutDetails = $this->getPaymentCheckoutId($paymentPayload);
            if ($checkoutDetails['status'] == 'true') {
                $checkoutId = $checkoutDetails['data']['id'];
                break;
            }
        }
        if ($checkoutId != '') {
            $paymentCheckoutIdLog = PaymentCheckoutIdLog::createLog([
                'ref_id' => $refId,
                'checkout_id' => $checkoutId,
                'payment_brand' => $paymentBrand,
                'amount' => $bookingAmount,
                'request' => json_encode($checkoutDetails['request']),
                'response' => json_encode($checkoutDetails['response'])
            ]);

            $checkoutIds[$paymentBrand] = $checkoutId;

            $bookingPayload = [
                'checkout_ids' => json_encode($checkoutIds),
                'amount' => $bookingAmount,
            ];
            $savedCheckoutDetails = BookingCheckout::updateBookingCheckout($bookingCheckoutId, $bookingPayload);

            if ($savedCheckoutDetails['status'] == 'true') {
                $result['success'] = true;
                $result['data'] = [
                    'checkout_ids' => $checkoutIds
                ];
                return $this->sendResponse($result['data'], '');
            } else {
                return $this->sendResponse([], 'Unable to process payment. Please try again.', false);
            }
        }
        return $this->sendResponse([], 'Something went wrong. Please try again.', false);
    } 
}
