<?php

namespace App\Http\Controllers\API\V1;

use Exception;
use App\Enums\Locale;
use App\Models\Wallet;
use App\Enums\Currency;
use App\Models\Markups;
use App\Models\Setting;
use App\Models\Bookings;
use App\Models\Customer;
use App\Traits\Gimmonix;
use App\Models\BrandName;
use App\Models\HotelName;
use App\Traits\HotelBeds;
use App\Enums\ServiceType;
use App\Models\Favourites;
use App\Models\RehlteLink;
use App\Traits\WithinEarth;
use App\Models\GoogleMapLog;
use App\Models\HotelAddress;
use App\Models\PopularGuide;
use App\Services\RedisCache;
use Illuminate\Http\Request;
use App\Models\DefaultMarkup;
use App\Models\FeaturedGuide;
use App\Models\HotelFacility;
use App\Models\MarkupSetting;
use App\Models\PopularAirline;
use App\Models\BookingCheckout;
use App\Traits\CheckHotelPrice;
use App\Traits\HotelListMarkup;
use App\Models\SeoTrendingHotel;
use App\Services\ImageUrlMasker;
use App\Models\PopularAttraction;
use App\Models\SeoTrendingFlight;
use App\Models\HotelSpecialRequest;
use App\Traits\LoyaltyPointsService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Crypt;
use App\Traits\HotelRecommendationFilter;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\API\V1\BaseController as BaseController;

class HotelController extends BaseController
{

    use WithinEarth, Gimmonix, HotelBeds, LoyaltyPointsService, HotelListMarkup, CheckHotelPrice, HotelRecommendationFilter;

    /*
     * default controller that use to set default values for this class
     */
    public function __construct()
    {
        $this->perPage = count(Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')) > 0 ? Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')[0]['value'] : "20";
    }

    /**
     * @OA\Post(
     ** path="/v1/hotel/hotel-auto-search",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Hotel"},
     *   summary="This will response hotel list of the requested keywords ",
     *   description="Get hotel list on base of requested keywords(City name or Hotel name),<br>
                      The language code (e.g., Locale::English->value for English, 'ar' for Arabic)<br><br>",
     *   operationId="hotel-auto-search",
     *   @OA\RequestBody(
     *     required=true,
     *     description="get hotel list on base of requested keywords",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"search"},
     *             @OA\Property(property="search", type="string", description=" Represents the keyword used to search for either a hotel name or city name."),
     *             @OA\Property(property="languageCode", default="en", type="string", description="Denotes the language code(en/ar) used for localization or specifying the language of the search results")
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function hotelSearch(Request $request)
    {
        $requestData = $request->json()->all();
        $customerId = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }
        // $validator = Validator::make($requestData, [
        //     'search' => 'required'
        // ]);

        // if ($validator->fails()) {
        //     return response()->json(['status' => false, 'message' => 'Invalid request', 'data' => [$validator->errors()]], 200);
        // }
        try {
            $locale = $request->input('locale', Locale::English->value);

            $result = $this->getHotelSearch($locale, $requestData, $customerId);

            return $this->sendResponse($result['data'], $result['message'], $result['status']);
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    public function hotelSearchByCity(Request $request)
    {
        $requestData = $request->json()->all();
        $customerId = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }
        try {
            $locale = $request->input('locale', Locale::English->value);
            $currency = $request->input('currency', Currency::SAR->value);

            $result = $this->getHotelSearchByCity($locale, $currency, $requestData, $customerId);

            return $this->sendResponse($result['data'], $result['message'], $result['status']);
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    public function hotelDetailsByHotelCode(Request $request)
    {
        $requestData = $request->json()->all();
        $logId = $customerId = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];
        }
        try {
            $locale = $request->input('locale', Locale::English->value);
            $currency = $request->input('currency', Currency::SAR->value);

            $result = $this->getHotelDetailsByHotelCode($locale, $currency, $requestData, $customerId, $logId);

            return $this->sendResponse($result['data'], $result['message'], $result['status']);
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/hotel/hotel-availability",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Hotel"},
     *   summary="Send request for get Availability for hotels",
     *   description="This API provides information about hotel availability :<br>
                                The language code (e.g., Locale::English->value for English, 'ar' for Arabic)<br>
                                destinationCode : Represents destination code(City ISO Code) to get from hotel search.<br>
                                checkIn : Represents the date of hotel check-in in YYYY-MM-DD format.<br>
                                checkOut : Represents the date of hotel check-out in YYYY-MM-DD format.<br>
                                rooms : Number of requested rooms of this occupancy.<br>
                                adults : Number of adult passengers for this room.<br>
                                children : Number of child passengers for this room.",
     *   operationId="hotel-availability",
     *   @OA\RequestBody(
     *     required=true,
     *     description="Get Availability by hotel code",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *              required={"destinationCode","checkIn","checkOut","occupancies","rooms","adults","children"},
     *              @OA\Property(property="languageCode", default="en", type="string", description="Denotes the language code(en/ar) used for localization or specifying the language of the search results"),
     *              @OA\Property(property="currency",example="SAR",type="string",description="Represents convert to currency"),
     *              @OA\Property(property="agencyId", type="string", default="0", example="0", description="The agency name throught which gt hotel"),
     *              @OA\Property(property="destinationCode",example="PMI",type="string",description="Represents destination code(City ISO Code) to get from hotel search"),
     *              @OA\Property(property="checkIn", default="2024-05-15", type="string", description="The date when the reservation check-in will occur. Format: YYYY-MM-DD"),
     *              @OA\Property(property="checkOut", default="2024-05-30", type="string", description="The date when the reservation check-out will occur. Format: YYYY-MM-DD"),
     *              @OA\Property(property="occupancies", title="occupancies",
     *                  minItems=1, maxItems=18,
     *                  type="array",
     *                  description="An array represents occupancies",
     *                  @OA\Items(
     *                          @OA\Property(
     *                                      property="rooms",
     *                                      default="1",
     *                                      example="1",
     *                                      type="integer",
     *                                      description="Represents Room Number."
     *                          ),
     *                          @OA\Property(
     *                                      property="adults",
     *                                      default="2",
     *                                      example="2",
     *                                      type="integer",
     *                                      description="Represents total number of Adults."
     *                          ),
     *                          @OA\Property(
     *                                      property="children",
     *                                      example="0",
     *                                      type="integer",
     *                                      description="Represents total number of child."
     *                          )
     *                 )
     *             ),
     *          )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function hotelAvailability(Request $request)
    {
        $data = [];
        $requestData = $request->only([
            'currency',
            'supplier',
            'destinationCode',
            'checkIn',
            'checkOut',
            'occupancies',
            'languageCode'
        ]);

        $validator = Validator::make($requestData, [
            'destinationCode' => 'required',
            'checkIn' => 'required',
            'checkOut' => 'required',
            'occupancies' => 'required|array',
            'occupancies.*.rooms' => 'required|integer',
            'occupancies.*.adults' => 'required|integer',
            'occupancies.*.children' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 200);
        }

        try {
            $hotalAvailabilityData = $this->getHotelAvailability($requestData);

            $defaultCurrency = "EUR";
            $currency = $requestData['currency'] ?? '';
            $checkAllowedCurrency = convertCurrencyExchangeRate('1', $defaultCurrency, $currency, []);

            if ($checkAllowedCurrency['status'] == false) {
                $success = [];
                return $this->sendError($success, 'Currency not allowed.');
            } else {
                return $this->sendResponse($hotalAvailabilityData, 'Get hotel availability successfuly.');
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    /**
     * @OA\Post(
     **  path="/v1/hotel/hotel-details",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Hotel"},
     *   summary="This will response hotel details of the requested keywords ",
     *   description="get hotel details on base of requested keywords(hotel codes) <br><br>
                                The language code (e.g., Locale::English->value for English, 'ar' for Arabic)<br>
                                checkIn : Represents the date of hotel check-in in YYYY-MM-DD format.<br>
                                checkOut : Represents the date of hotel check-out in YYYY-MM-DD format.<br>
                                rooms : Number of requested rooms of this occupancy.<br>
                                adults : Number of adult passengers for this room.<br>
                                children : Number of child passengers for this room.",
     *   operationId="hotel-details",
     *   @OA\RequestBody(
     *     required=true,
     *     description="get hotel details on base of requested keywords",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"checkIn","checkOut","occupancies","rooms","adults","children","hotelCode"},
     *              @OA\Property(property="languageCode", default="en", type="string", description="Denotes the language code(en/ar) used for localization or specifying the language of the search results"),
     *              @OA\Property(property="currency",example="SAR",type="string",description="Represents convert to currency"),
     *              @OA\Property(property="supplier", type="string", default="HOTEL_BEDS", description="enter supplier's name"),
     *              @OA\Property(property="agencyId", type="string", default="0", example="0", description="The agency name throught which gt hotel"),
     *              @OA\Property(property="checkIn", default="2024-05-15", type="string", description="The date when the reservation check-in will occur. Format: YYYY-MM-DD"),
     *              @OA\Property(property="checkOut", default="2024-05-30", type="string", description="The date when the reservation check-out will occur. Format: YYYY-MM-DD"),
     *              @OA\Property(property="occupancies", title="occupancies",
     *                  minItems=1, maxItems=18,
     *                  type="array",
     *                  description="An array represents occupancies",
     *                  @OA\Items(
     *                          @OA\Property(
     *                                      property="rooms",
     *                                      default="1",
     *                                      example="1",
     *                                      type="integer",
     *                                      description="Represents Room Number."
     *                          ),
     *                          @OA\Property(
     *                                      property="adults",
     *                                      default="2",
     *                                      example="2",
     *                                      type="integer",
     *                                      description="Represents total number of Adults."
     *                          ),
     *                          @OA\Property(
     *                                      property="children",
     *                                      example="0",
     *                                      type="integer",
     *                                      description="Represents total number of child."
     *                          )
     *                 )
     *             ),
     *             @OA\Property(property="hotelCode", type="string", default="314", example="314", description="Represents hotel code for hotel details list"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function hotelDetail(Request $request)
    {
        $data = [];
        // $requestData = $request->only([
        //     'currency',
        //     'supplier',
        //     'checkIn',
        //     'checkOut',
        //     'occupancies',
        //     'languageCode',
        //     'hotelCode'
        // ]);

        // $validator = Validator::make($requestData, [
        //     'checkIn' => 'required',
        //     'checkOut' => 'required',
        //     'hotelCode' => 'required',
        //     'occupancies' => 'required|array',
        //     'occupancies.*.rooms' => 'required|integer',
        //     'occupancies.*.adults' => 'required|integer',
        //     'occupancies.*.children' => 'required|integer'
        // ]);

        // if ($validator->fails()) {
        //     return response()->json(['status' => false, 'message' => 'Invalid request', 'data' => [$validator->errors()]], 200);
        // }
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $logId = $customerId = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];
        }

        try {
            $locale = $request->input('locale', Locale::English->value);

            $result = $this->getHotelDetail($locale, $requestData, $logId, $customerId);

            return $this->sendResponse($result['data'], $result['message'], $result['status']);
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    public function getHotelsByGeo(Request $request)
    {
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);

        $customerId = Auth::check() ? Auth::id() : '';

        try {
            // Fetch hotel list once
            // $result = $this->getGimmonixHotels('list', $requestData, $customerId);

            $type = 'list';

            // // Cache the record for 5 minutes
            $key = 'hotels_by_geo_' . md5(
                $type . json_encode($requestData) . $customerId
            );
            $result = RedisCache::cache(
                $key,
                300,
                function () use ($type, $requestData, $customerId) {
                    $data =  $this->getGimmonixHotels($type, $requestData, $customerId);

                    // Only return data for caching if it's not empty
                    if (isset($data['data']['Hotels']) && !empty($data['data'])) {
                        return $data;
                    }

                    // Returning null prevents caching
                    return null;
                }
            );
            if ($result === null) {
                $result = $this->getGimmonixHotels('list', $requestData, $customerId);
            }
            $result = $this->getHotelListMarkup($result, $requestData);
            $result = $this->hotelRecommendationFilter($result, $requestData);

            // $result = ImageUrlMasker::mask($result);

            unset($result['data']['PageReponse']);
            unset($result['data']['UnMatchedHotelIds']);


            return $this->sendResponse($result['data'], $result['message'] ?? 'Success', $result['status'] ?? true);
        } catch (Exception $e) {
            RedisCache::forgetCache($key);
            return $this->sendError([], 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    public function getHotelsByIds(Request $request)
    {
        $requestData = $request->all();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);

        try {
            $result = $this->getGimmonixHotelsByIds('list-by-ids', $requestData);
            return $this->sendResponse($result['data'], $result['message'], $result['status']);
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    public function getHotelsByFilters(Request $request)
    {
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);

        $logId = $customerId = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];
        }
        try {
            $result = $this->getGimmonixHotels('filter-list', $requestData, $customerId, $logId);

            $brand_names = BrandName::pluck('chain_name', 'chain_code');

            if (!empty($result['data']['HotelsResultFilter']['HotelChain'])) {
                foreach ($result['data']['HotelsResultFilter']['HotelChain'] as &$chain) {
                    $code = $chain['Value'];
                    $chain['name'] = $brand_names[$code] ?? null;
                }
            }
            return $this->sendResponse($result['data'], $result['message'], $result['status']);
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage() . $e->getLine()], 500);
        }
    }

    public function getHotelDetailsById(Request $request)
    {
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);
        $customer_id = $request->input('customer_id', '');
        $logId = $customerId = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];
        }

        try {
            $result = $this->getGimmonixHotelDetails($requestData, $customerId, $logId);
            $result = $this->getHotelListMarkupById($result, $requestData);
            $result = $this->hotelRecommendationFilter($result, $requestData);
            // Sort Price by Ascending Order
            $rooms = $result['data']['Rooms'] ?? [];
            $roomPrices = [];
            foreach ($rooms as $roomKey => $roomData) {
                $minPrice = INF;

                foreach ($roomData as $boardType => $ratePlans) {
                    foreach ($ratePlans as $ratePlan => $ratePlanData) {
                        foreach ($ratePlanData as $refundType => $details) {
                            if (isset($details['SimplePrice'])) {
                                $price = floatval($details['SimplePrice']);
                                if ($price < $minPrice) {
                                    $minPrice = $price;
                                }
                            }
                        }
                    }
                }

                if ($minPrice !== INF) {
                    $roomPrices[$roomKey] = $minPrice;
                }
            }
            asort($roomPrices);
            $sortedRooms = [];
            foreach (array_keys($roomPrices) as $roomKey) {
                $sortedRooms[$roomKey] = $rooms[$roomKey];
            }

            $result['data']['Rooms'] = $sortedRooms;

            $result['data']['showSupplierName'] = false;
            if ($customer_id) {
                $customer = Customer::where('id', $customer_id)->first();
                if ($customer) {
                    $result['data']['showSupplierName'] = $customer->show_supplier_name ? true : false;
                }
            }

            return $this->sendResponse($result['data'], $result['message'], $result['status']);
        } catch (Exception $e) {
            \Log::error('Error in getHotelDetailsById: ' . $e->getMessage());
            return $this->sendError([], 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }


    public function getRoomCancellationPolicy(Request $request)
    {
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);
        $requestData['price'] = $request->input('price', '');
        $logId = $customerId = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];

            try {
                $result = $this->getGimmonixRoomCancellationPolicy($requestData, $customerId, $logId);
                return $this->sendResponse($result['data'], $result['message'], $result['status']);
            } catch (Exception $e) {
                $success = [];
                return $this->sendError($success, 'Something went wrong', ['error' => $e->getMessage()], 500);
            }
        } else {
            return $this->sendError('Something went wrong: Session ID not found', []);
        }
    }

    public function add_favourites(Request $request)
    {
        $requestData = $request->json()->all();
        $locale = $request->input('locale', Locale::English->value);
        try {
            // $customerId = Auth::id();
            $customerId = $request->input('customer_id');
            $existingFavourite = Favourites::where('hotel_id', $requestData['hotel_id'])
                ->where('customer_id', $customerId)
                ->first();

            if ($existingFavourite) {
                if ($requestData['status'] == 0) {
                    $existingFavourite->delete();

                    return $this->sendResponse(null, 'Favourite removed successfully', 200);
                } else {
                    // Update the status
                    $existingFavourite->status = $requestData['status'];
                    $existingFavourite->save();

                    return $this->sendResponse($existingFavourite, 'Status updated successfully', 200);
                }
            } else {
                // Create a new favourite record
                $result = Favourites::create([
                    'hotel_id'    => $requestData['hotel_id'],
                    'customer_id' => $requestData['customer_id'],
                    'image'       => $requestData['image'],
                    'title'       => $requestData['title'],
                    'address'     => $requestData['address'],
                    'price'       => $requestData['price'],
                    'rating'      => $requestData['rating'],
                    'status'      => $requestData['status'],
                    'url_params'  => $requestData['url_params'],
                    'type'        => $requestData['type'],
                    'language_code'        => $locale,
                ]);

                return $this->sendResponse($result, 'Favourite added successfully', 200);
            }
        } catch (Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }





    public function get_favourites(Request $request)
    {
        $request->validate([
            'customer_id' => 'required',
            'locale' => 'sometimes|string|in:en,ar'
        ]);
        $customerId = $request->input('customer_id');
        $locale = $request->input('locale');
        // $customerId = Auth::id();

        try {
            $favourites = Favourites::where('customer_id', $customerId)->get();
            if ($favourites->isEmpty()) {
                return $this->sendResponse([], 'No favourites found for this customer', 404);
            }
            foreach ($favourites as &$favourite) {
                if ($favourite->language_code == $locale) {
                    continue;
                }

                if ($favourite->language_code == Locale::English->value) {
                    $translatedHotelName = HotelName::select('display_name')->where('hotel_id', $favourite->hotel_id)->first();
                    $favourite->title = $translatedHotelName ? $translatedHotelName->display_name : $favourite->title;
                } else {
                    $translatedHotelName = HotelAddress::select('displayname')->where('hotelid', $favourite->hotel_id)->first();
                    $favourite->title = $translatedHotelName ? $translatedHotelName->displayname :  $favourite->title;
                }
            }
            unset($favourite);

            return $this->sendResponse($favourites, 'Success', 200);
        } catch (Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }


    public function delete_favourites(Request $request)
    {
        try {
            // $favId = $request->input('fav_id');
            $customerId = $request->input('customer_id');
            // $customerId = Auth::id();

            // $favourite = Favourites::find($favId);
            $favourite = Favourites::where('customer_id', $customerId);

            if (!$favourite) {
                // return $this->sendError('Favourite not found', [], 404);
                return $this->sendError('Favourite not found for the given customer_id', [], 404);
            }

            $favourite->delete();
            return $this->sendResponse([], 'Favourite deleted successfully', 200);
        } catch (Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }




    // public function get_LastBooking(Request $request)
    // {
    //     $requestData = $request->json()->all();

    //     try {

    //         $customerId = $requestData['customer_id'];

    //         $lastBooking = Bookings::where('customer_id', $customerId)
    //             ->where('booking_type', 'Hotel')
    //             ->orderBy('created_at', 'desc')
    //             ->first(['id', 'email', 'booking_ref', 'booking_type']);

    //         if ($lastBooking) {
    //             return $this->sendResponse($lastBooking, 'Success', 200);
    //         } else {
    //             return $this->sendError('No bookings found for this customer', [], 404);
    //         }

    //     } catch (Exception $e) {
    //         return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
    //     }
    // }

    /**
     * Get Last Booking Details for a customer -  my Trip
     */
    public function get_LastBooking(Request $request)
    {
        $requestData = $request->json()->all();
        $validated = Validator::make($requestData, [
            'email' => 'required|email',
        ]);
        if ($validated->fails()) {
            return $this->sendError('Invalid request', [$validated->errors()], 422);
        }

        try {
            $validatedData = $validated->validated();

            // $customer = Customer::where('email', $email)->first();

            // if (!$customer) {
            //     return $this->sendError('Customer not found', [], 404);
            // }

            // $lastBooking = Bookings::where('customer_id', $customer->id)
            //     ->where('booking_type', 'Hotel')
            //     ->orderBy('created_at', 'desc')
            //     ->first(['id', 'booking_ref', 'booking_type', 'created_at']);
            $lastBooking = Bookings::where('email', $validatedData['email'])
                ->where('booking_type', ServiceType::Hotel->value)
                ->orderBy('created_at', 'desc')
                ->first(['id', 'booking_ref', 'booking_type', 'created_at']);

            if ($lastBooking) {
                $lastBooking->email = $validatedData['email'];

                return $this->sendResponse($lastBooking, 'Success', 200);
            } else {
                return $this->sendError('No bookings found for this customer', [], 404);
            }
        } catch (Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }



    public function getWalletDetails(Request $request)
    {
        $requestData = $request->json()->all();
        $currency = $request->input('currency', Currency::SAR->value);

        try {
            $customerId = Auth::check() ? Auth::id() : $requestData['customer_id'] ?? null;
            if (!$customerId) {
                return $this->sendError('Customer ID is required', [], 400);
            }

            $walletType = $requestData['wallet_type'] ?? '';
            $date = date('Y-m-d');

            if (!in_array($walletType, ['credit', 'debit', 'expired'])) {
                return $this->sendError('Invalid wallet type', [], 400);
            }

            $walletQuery = Wallet::where('customer_id', $customerId)->where('status', 'active');

            switch ($walletType) {
                case 'credit':
                    $walletQuery->where('type', 'credit')->where('expiry_date', '>=', $date);
                    break;
                case 'debit':
                    $walletQuery->where('type', 'debit');
                    break;
                case 'expired':
                    $walletQuery->where('expiry_date', '<', $date);
                    break;
            }

            $walletDetails = $walletQuery->get();

            if ($walletDetails->isEmpty()) {
                return $this->sendError('Wallet not found', [
                    'totalWalletAmount' => 0,
                    'currency' => $currency,
                    'WalletDetails' => [],
                    'BookingDetails' => [],
                ], 200);
            }

            $bookingIds = $walletDetails->pluck('booking_id')->toArray();

            $bookingDetails = Bookings::whereIn('id', $bookingIds)->get([
                'id as booking_id',
                'booking_ref',
                'rehtle_ref',
                'ref_id',
                'booking_type',
                'booking_date',
                'wallet_amount',
                'wallet_used',
                'supplier_currency'
            ]);

            $walletDetails = $walletDetails->map(function ($item) use ($bookingDetails) {
                $booking = $bookingDetails->where('booking_id', $item['booking_id'])->first();
                $item['currency'] = $booking ? $booking->supplier_currency : null;
                return $item;
            });

            $walletTotalAmount = Wallet::getTotalWalletAmount($customerId);
            $walletTotalAmount = convertCurrencyExchangeRate($walletTotalAmount, Currency::SAR->value, $currency);

            if (!isset($walletTotalAmount['status']) || $walletTotalAmount['status'] === false) {
                return $this->sendError('Currency not allowed. please try again', 400);
            }

            return $this->sendResponse([
                'totalWalletAmount' => $walletTotalAmount['data']['convertedRate'] ?? 0,
                'currency' => $currency,
                'WalletDetails' => $walletDetails,
                'BookingDetails' => $bookingDetails,
            ], 'Success', 200);
        } catch (Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }



    public function getTrendingSearchDetails(Request $request)
    {
        $requestData = $request->json()->all();
        if (empty($requestData['type'])) {
            return $this->sendError('The "type" field is required.', [], 400);
        }

        try {
            $type = $requestData['type'];

            $modelMap = [
                'seo_trending_flight' => SeoTrendingFlight::class,
                'seo_trending_hotel' => SeoTrendingHotel::class,
                'featured_guide' => FeaturedGuide::class,
                'popular_attractions' => PopularAttraction::class,
                'popular_airlines' => PopularAirline::class,
                'popular_guide' => PopularGuide::class,
                'rehlte_link' => RehlteLink::class,
            ];

            if (!array_key_exists($type, $modelMap)) {
                return $this->sendError('Invalid "type" provided. Please use one of: ' . implode(', ', array_keys($modelMap)), [], 400);
            }

            $data = $modelMap[$type]::all();

            if ($data->isEmpty()) {
                return $this->sendError(ucwords(str_replace('_', ' ', $type)) . ' details not found.', [], 404);
            }

            // Update banner column with full URL
            $data = $data->map(function ($item) {
                if (!empty($item->banner)) {
                    $item->banner = asset('storage/' . $item->banner);
                }
                return $item;
            });

            return $this->sendResponse(['data' => $data], 'Success', 200);
        } catch (\Throwable $e) {
            return $this->sendError('Something went wrong.', ['error' => $e->getMessage()], 500);
        }
    }


    /**
     * Get Map and Street Map in Hotel List page based on type
     */

    public function getHotelListMap(Request $request)
    {
        try {
            $validate = Validator::make($request->all(), [
                'latitude' => 'required|numeric',
                'longitude' => 'required|numeric',
                'type' => 'required|in:map,streetview'
            ]);

            if ($validate->fails()) {
                return $this->sendError('Invalid request', [$validate->errors()], 422);
            }

            $validated = $validate->validated();
            $latitude = $validated['latitude'];
            $longitude = $validated['longitude'];
            $type = $validated['type'];
            $customMarkerUrl = asset('storage/images/general/map-marker2.png');
            $hotel = [];
            if ($type == 'map') {
                $userId = Auth::id();
                $logName = "Google Map - Hotel List";
                GoogleMapLog::createGoogleMapLog($userId, $logName);
                $hotel['Map'] = 'https://maps.googleapis.com/maps/api/staticmap?center='
                    . $latitude . ',' . $longitude
                    . '&zoom=15&size=640x480'
                    . '&markers=icon:' . $customMarkerUrl . '|' . $latitude . ',' . $longitude
                    . '&key=' . env('GOOGLE_MAP_KEY');
            }
            if ($type == 'streetview') {
                $userId = Auth::id();
                $logName = "Google Map Street View - Hotel List";
                GoogleMapLog::createGoogleMapLog($userId, $logName);
                $hotel['StreetView'] = 'http://maps.googleapis.com/maps/api/streetview?size=640x480&location='
                    . $latitude . ',' . $longitude
                    . '&fov=120&heading=235&pitch=10&sensor=false&key=' . env('GOOGLE_MAP_KEY');
            }
            //  need to change in frontend as well so make it pending to change
            return response()->json(['status' => 'Success', 'data' => $hotel], 200);
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong.', ['errors' => $e->getMessage()]);
        }
    }

    /**
     * Hotel List Page - Separate API call to get Facilities and Description response to initial increase loading speed 
     */
    public function getGimmonixHotelFacilities(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'hotelIDs' => 'required|array',
            'currency' => 'required',
            'locale' => 'required',
            'logId' => 'sometimes|string',
            'type' => 'string'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', $validator->errors(), 422);
        }


        $requestData = [
            'currency' => $request->input('currency', Currency::SAR->value),
            'locale' => $request->input('locale', Locale::English->value),
            'hotelIDs' => $request->input('hotelIDs'),
            'log_id' => $request->input('log_id', ''),
            'type' => $request->input('type', ''),

        ];

        try {

            $result = $this->getGimmonixHotelsDescription($requestData);

            // $key = 'hotel_facilities_' . md5(json_encode($requestData));
            // $result = Cache::remember($key, 300, function () use ($requestData) {
            //     return $this->getGimmonixHotelsDescription($requestData);
            // });

            if ($result['status']) {
                return $this->sendResponse($result['data'], $result['message'] ?? 'Success');
            } else {
                return $this->sendError($result['message'] ?? 'Failed to fetch hotel facilities', [], 400);
            }
        } catch (Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get Updated Hotel Price if user stay in page long time
     */
    public function getUpdatedHotelPrice(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'ref_id' => 'required|string',
            'searchFrom' => 'required | string',
        ]);
        if ($validator->fails()) {
            return $this->sendError('Something went wrong', [$validator->getMessageBag()], 422);
        }
        $validatedData =  $validator->validated();
        $savedBookingCheckout = BookingCheckout::getBookingCheckout($validatedData['ref_id']);
        if ($savedBookingCheckout['status'] === false) {
            return $this->sendError('Booking not found', [], 404);
        }

        $response = $this->getUpdatedHotelPrices($savedBookingCheckout, $validatedData);
        if ($response['status']) {
            return $this->sendResponse($response, 'Hotel Updated Price get successfully');
        }
        return $this->sendResponse($response, 'Hotel Updated Price get successfully');
    }

    /**
     * Hotel Details Page - Separate API call to get Review and around this hotel response to reduce Google Map Cost
     */
    public function getAroundThisHotel(Request $request)
    {

        $requestData = $request->all();
        $requestHeaders = $request->header();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);

        $logId = $customerId = '';
        if (Auth::check()) {
            $customerId = Auth::id();
        }
        if (isset($requestHeaders['log-id'])) {
            $logId = $requestHeaders['log-id'][0];
        }

        try {
            $result = $this->getAroundthisHotelDetails($requestData, $customerId, $logId);

            return $this->sendResponse($result['data'], $result['message'], $result['status']);
        } catch (Exception $e) {
            \Log::error('Error in getHotelDetailsById: ' . $e->getMessage());
            return $this->sendError([], 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }



    public function getHotelRecommendations(Request $request)
    {
        $requestData = $request->all();
        $requestHeaders = $request->header();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);

        try {
            $result = $this->hotelRecommendationFilter($requestData);
            return $this->sendResponse($result['data'], $result['message'], $result['status']);
        } catch (Exception $e) {
            \Log::error('Error in getHotelRecommendations: ' . $e->getMessage());
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }




    /**
     *  For Testing Purpose - Hotel List Page - Encrypt Json Purpose Data
     * */

    public function getEncryptHotelsByGeo(Request $request)
    {
        $requestData = $request->all();
        $requestData['currency'] = $request->input('currency', Currency::SAR->value);
        $requestData['locale'] = $request->input('locale', Locale::English->value);
        $customerId = Auth::check() ? Auth::id() : '';

        try {
            $result = $this->getGimmonixHotels('list', $requestData, $customerId);
            $result = $this->getHotelListMarkup($result, $requestData);
            $result = $this->hotelRecommendationFilter($result, $requestData);


            $plainText = json_encode($result['data']);

            $key = '8aLzZm9sJf0n3c1o7g0hBTBReXn6aAo=';
            $iv = openssl_random_pseudo_bytes(16);

            $encrypted = openssl_encrypt($plainText, 'AES-256-CBC', $key, OPENSSL_RAW_DATA, $iv);

            $data = [
                'data' => base64_encode($encrypted),
                'iv' => base64_encode($iv),
                'status' => true,
                'message' => 'Success',
            ];
            return $this->sendResponse($data, 'Hotels List Successfully', $result['status']);
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()]);
        }
    }


    public function getHotelSpecialRequest()
    {
        try {
            $hotelSpecialRequest = HotelSpecialRequest::select('id', 'special_request_name',  'special_request_name_ar')->where('status', 1)->orderBy('special_request_name')->get();
            if ($hotelSpecialRequest->isEmpty()) {
                return $this->sendResponse([], 'No Hotels Special Request found');
            }
            return $this->sendResponse($hotelSpecialRequest, 'Hotels Special Request Listed Successfully');
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()]);
        }
    }
}
