<?php

/**
 * @package     GoogleMap
 * @subpackage  GoogleMap
 * @Author      Amar Technolabs Pvt. ltd(info@amarinfotech.com)
 * @Copyright(C) 2023 [NAME OF THE ORGANISATION THAT ON BEHALF OF THE CODE WE ARE WORKING].
 * @Version 1.0.0
 * module of the GoogleMap.
 */

namespace App\Http\Controllers\API\V1;

use App\Http\Controllers\API\V1\BaseController as BaseController;
use App\Http\Controllers\Controller;
use App\Models\Country;
use App\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use GuzzleHttp;
use App\Models\Setting;
use App\Traits\CommonService;

class GoogleMapController extends BaseController
{
    use CommonService;
    /**
     * @OA\Post(
     ** path="/v1/google-place-search",
     *   tags={"GoogleMapApi"},
     *   summary="Search for google Place",
     *   description="Calling an API for Search google Places
                ",
     *   operationId="google-place-search",
     *   @OA\Parameter(
            name="body",
            in="query",
            required=false,
            explode=true,
            @OA\Schema(
                 collectionFormat="multi",
                 required={"search"},
                 @OA\Property(property="search", type="string",  ),
            ),
        ),     
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/
    public function placeSearch(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'search' => 'required|min:3',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }

            // check for google map api key exist or not
            $isGoogleMapEnabled = Setting::select('value')
                ->where('config_key', '=', 'general|site|googleApiKey')
                ->first();

            if (!@$isGoogleMapEnabled->value) {
                return $this->sendError('Google map is not enabled', [], 200);
            }

            // prepare API URL
            $search = $request->search;
            $googleKey = @$isGoogleMapEnabled->value;

            $url = "https://maps.googleapis.com/maps/api/place/queryautocomplete/json";
            $params = [
                'key'    => $googleKey,
                'input'  => $search,
                'region' => 'in'
            ];

            // Use libCurl for GET request
            $result = $this->libCurl(
                $url,
                'GET',
                ['Content-Type: application/json'],
                $params
            );

            if ($result['status'] === 'true') {
                return $this->sendResponse([$result['data']], 'Successfully fetched Google Places');
            } else {
                return $this->sendError('Error fetching Google Places', [$result['message'] ?? 'Unknown error']);
            }
        } catch (\Exception $ex) {
            return $this->sendError('Error Google Place API Search', []);
        }
    }



    /**
     * @OA\Post(
     ** path="/v1/google-detail-place-search",
     *   tags={"GoogleMapApi"},
     *   summary="Search for google Place Detail",
     *   description="Calling an API for Search google Place Detail
                ",
     *   operationId="google-detail-place-search",
     *   @OA\Parameter(
            name="body",
            in="query",
            required=false,
            explode=true,
            @OA\Schema(
                 collectionFormat="multi",
                 required={"place_id"},
                 @OA\Property(property="place_id", type="string",  ),
            ),
        ),     
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/
    public function placeDetailSearch(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'place_id' => 'required',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }

            // check for google map api key
            $isGoogleMapEnabled = Setting::select('value')
                ->where('config_key', '=', 'general|site|googleApiKey')
                ->first();

            if (!@$isGoogleMapEnabled->value) {
                return $this->sendError('Google map is not enabled', [], 200);
            }

            // prepare API URL and params
            $googleKey = @$isGoogleMapEnabled->value;
            $url = "https://maps.googleapis.com/maps/api/place/details/json";
            $params = [
                'place_id' => $request->place_id,
                'fields'   => 'address_components,formatted_address,geometry,adr_address',
                'key'      => $googleKey
            ];

            // Call using libCurl
            $result = $this->libCurl(
                $url,
                'GET',
                ['Content-Type: application/json'],
                $params
            );

            if ($result['status'] === 'true') {
                return $this->sendResponse([$result['data']], 'Successfully Fetch Google Places Detail');
            } else {
                return $this->sendError('Error Google Place Detail API Search', [$result['message'] ?? 'Unknown error']);
            }
        } catch (Exception $ex) {
            return $this->sendError('Error Google Place Detail API Search', []);
        }
    }


    /**
     * @OA\Post(
     ** path="/v1/geo-location-by-ip",
     *   tags={"GoogleMapApi"},
     *   summary="Get Geo Details By IP",
     *   description="Calling an API for Get Geo Detail",
     *   operationId="geo-location-by-ip",
      
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/
    public function geoLocationByIp(Request $request)
    {
        try {
            $ip = $_SERVER['REMOTE_ADDR'];

            $url = "http://www.geoplugin.net/json.gp";
            $params = [
                'ip' => $ip
            ];

            // Use libCurl for GET request
            $result = $this->libCurl(
                $url,
                'GET',
                ['Content-Type: application/json'],
                $params
            );

            if ($result['status'] !== 'true' || empty($result['data'])) {
                return $this->sendError('Error Geo Location Detail Fetch', []);
            }

            $apiData = $result['data']; 

            $isoCode = $apiData['geoplugin_currencyCode'] ?? null;
            $countryCode = $apiData['geoplugin_countryCode'] ?? null;

            $iso_code = Currency::where('code', $isoCode)->value('code');
            $country_data = Country::select('isd_code', 'max_mobile_number_length')
                ->where('iso_code', $countryCode)
                ->first();

            $isd = $iso_code ? true : false;

            $res = [
                'is_available_in_system' => $isd,
                'isd_code' => $country_data['isd_code'] ?? null,
                'max_mobile_number_length' => $country_data['max_mobile_number_length'] ?? null
            ];

            // Merge arrays
            $mergedData = array_merge($apiData, $res);

            // Convert to JSON and back to object (same as before)
            $finalJsonString = json_encode($mergedData, JSON_PRETTY_PRINT);
            $data = json_decode($finalJsonString);

            return $this->sendResponse([$data], 'Successfully Fetch Geo Location Detail');
        } catch (Exception $ex) {
            return $this->sendError('Error Geo Location Detail Fetch', []);
        }
    }
}
