<?php

/**
 * @package     Geography
 * @subpackage  Geography
 * @Author     Rehlte Travel and Tourism LLC
 * @Copyright(C) 2025 [NAME OF THE ORGANISATION THAT ON BEHALF OF THE CODE WE ARE WORKING].
 * @Version 1.0.0
 * module of the Geography.
 */

namespace App\Http\Controllers\API\V1;

use App\Models\City;
use App\Enums\Locale;
use App\Models\State;
use App\Models\Airport;
use App\Models\Country;
use App\Models\Setting;
use App\Models\CityI18n;
use App\Models\HotelName;
use App\Models\HotelAddress;
use Illuminate\Http\Request;
use App\Models\TopDestination;
use App\Models\TopDestinationI18s;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\URL;
use App\Http\Controllers\Controller;
use App\Models\CountryI18ns;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
use App\Services\HotelElasticSearchService;

class GeographyController extends BaseController
{
    /**
     * @OA\Get(
     ** path="/v1/geography/get-countries",
     *   tags={"Geography"},
     *   summary="get countries list",
     *   description="get countries list<br><br>Pass Country name or iso code or isd code in Search<br>Pass Per page",
     *   operationId="get-countries",
     *   @OA\Parameter(
     *       name="body",
     *       in="query",
     *       required=false,
     *       explode=true,
     *       @OA\Schema(
     *            collectionFormat="multi",
     *            required={"search"},
     *            @OA\Property(property="search", type="string",  ),
     *            @OA\Property(property="per_page", type="string",  ),
     *       ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/
    public function getCountries(Request $request)
    {
        $data = [];
        $requestData = $request->only(['search']);
        //set validation for search keyword
        $validator = Validator::make($requestData, []);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()]);
        }
        try {
            if (isset($request->per_page) && $request->per_page != "") {
                $this->perPage = $request->per_page;
            } else {
                $this->perPage = Setting::where('config_key', 'general|setting|pagePerAPIRecords')->value('value');
            }

            if (isset($requestData['search']) && $requestData['search'] != '') {
                $getCountryList = Country::select('id', 'iso_code', 'isd_code', 'max_mobile_number_length', 'status')
                    ->with(['countryCode' => function ($country) {
                        $country->select(['country_id', 'country_name', 'language_code']);
                    }])
                    ->whereHas('countryCode', function ($q) use ($requestData) {
                        $q->orHaving('country_name', 'like', '%' . $requestData['search'] . '%');
                    })
                    ->orWhere('isd_code', $requestData['search'])
                    ->orWhere('iso_code', $requestData['search'])
                    ->orderByRaw("iso_code = '{$requestData['search']}' DESC, iso_code ASC")
                    ->paginate($this->perPage);
            } else {
                $getCountryList = Country::select('id', 'iso_code', 'isd_code', 'max_mobile_number_length', 'status')
                    ->with(['countryCode' => function ($country) {
                        $country->select(['country_id', 'country_name', 'language_code']);
                    }])
                    ->orderByRaw("iso_code ASC")
                    ->paginate($this->perPage);
            }

            $CountryData = collect($getCountryList->items())->map(function ($country) {
                $country_en = '';
                $country_ar = '';

                foreach ($country['countryCode'] as $countryName) {
                    switch ($countryName['language_code']) {
                        case Locale::English->value:
                            $country_en = $countryName['country_name'];
                            break;
                        case Locale::Arabic->value:
                            $country_ar = $countryName['country_name'];
                            break;
                    }
                }

                return [
                    'id' => $country['id'],
                    'iso_code' => $country['iso_code'],
                    'isd_code' => $country['isd_code'],
                    'max_mobile_number_length' => $country['max_mobile_number_length'],
                    'status' => $country['status'],
                    'country_en' => $country_en,
                    'country_ar' => $country_ar,
                    'flag' => URL::to('/') . '/storage/country-flags/' . strtolower($country['iso_code']) . '.png',
                ];
            });
            if (isset($requestData['search']) && $requestData['search'] != '') {
                $queryString = http_build_query([
                    'search' => urlencode($requestData['search']),
                    'per_page' => urlencode($this->perPage),
                ]);
            } else {
                $queryString = http_build_query([
                    'per_page' => urlencode($this->perPage),
                ]);
            }
            $countries = $CountryData->values()->toArray();

            if (count($countries) > 0) {
                $arabicCountryNameMap = $rankCountries = $rankCountriesIso = $searchCountriesInEnglish = $searchCountries = $searchRankCountries = [];
                foreach ($countries as $key => $country) {
                    $countryName = $country['country_en'];
                    $arabicCountryNameMap[$country['country_ar']] = $countryName;
                }
                $configRankCountries = Setting::where('config_key', 'general|site|rankCountries')->value('value');
                $configRankCountries = explode(',', $configRankCountries);
                if ($request->input('search_countries') != '') {
                    $locale = $request->input('locale');
                    $searchCountries = $searchCountriesTemp = array_unique(explode(',', $request->input('search_countries')));
                    foreach ($searchCountries as $key => $searchCountry) {
                        $searchCountry = ($locale == Locale::Arabic->value) ? $arabicCountryNameMap[$searchCountry] : $searchCountry;
                        array_push($searchCountriesInEnglish, $searchCountry);
                        if (($searchKey = array_search($searchCountry, $configRankCountries)) !== false) {
                            unset($configRankCountries[$searchKey]);
                        }
                    }
                    if (count($searchCountries) > 0) {
                        $matchingField = ($locale == Locale::Arabic->value) ? 'country_ar' : 'country_en';
                        foreach ($countries as $key => $country) {
                            foreach ($searchCountries as $key => $searchCountry) {
                                if (
                                    $country[$matchingField] == $searchCountry &&
                                    array_key_exists($country[$matchingField], $searchRankCountries) == false
                                ) {
                                    $searchRankCountries[$country[$matchingField]] = $country;
                                    unset($searchCountries[$key]);
                                }
                            }
                        }
                        $searchRankCountriesTemp = [];
                        foreach ($searchCountriesTemp as $key => $searchCountry) {
                            if (array_key_exists($searchCountry, $searchRankCountries)) {
                                array_push($searchRankCountriesTemp, $searchRankCountries[$searchCountry]);
                            }
                        }
                        $searchRankCountries = $searchRankCountriesTemp;
                    }
                }
                foreach ($countries as $key => $country) {
                    $countryName = $country['country_en'];
                    $arabicCountryNameMap[$country['country_ar']] = $countryName;
                    if (in_array($countryName, $configRankCountries) && in_array($countryName, $searchCountriesInEnglish) == false) {
                        $countryKeyPos = array_search($countryName, $configRankCountries);
                        $rankCountries[$countryKeyPos] = $country;
                        unset($countries[$key]);
                    }
                }
                ksort($rankCountries);
                $searchKeyPos = array_search('{{search}}', $configRankCountries);
                if (count($searchRankCountries) > 0 && $searchKeyPos > -1) {
                    $searchRankCountries = array_values($searchRankCountries);
                    array_splice($rankCountries, $searchKeyPos, 0, $searchRankCountriesTemp);
                }
                if (count($rankCountries) > 0) {
                    foreach ($rankCountries as $key => $rankCountry) {
                        array_push($rankCountriesIso, $rankCountry['iso_code']);
                    }
                    foreach ($countries as $key => $country) {
                        if (in_array($country['iso_code'], $rankCountriesIso)) {
                            unset($countries[$key]);
                        }
                    }
                }
                $countries = array_merge($rankCountries, $countries);
            }


            $output = [
                'current_page' => $getCountryList->currentPage(),
                'data' => $countries,
                'first_page_url' => $getCountryList->url(1) . '&' . $queryString,
                'from' => $getCountryList->firstItem(),
                'last_page' => $getCountryList->lastPage(),
                'last_page_url' => $getCountryList->url($getCountryList->lastPage()) . '&' . $queryString,
                'links' => [
                    [
                        'url' => $getCountryList->previousPageUrl() . '&' . $queryString,
                        'label' => '&laquo; Previous',
                        'active' => $getCountryList->onFirstPage(),
                    ],
                    [
                        'url' => $getCountryList->url(1) . '&' . $queryString,
                        'label' => '1',
                        'active' => $getCountryList->currentPage() === 1,
                    ],
                    [
                        'url' => $getCountryList->nextPageUrl() . '&' . $queryString,
                        'label' => 'Next &raquo;',
                        'active' => $getCountryList->hasMorePages(),
                    ],
                ],
                'next_page_url' => $getCountryList->nextPageUrl() . '&' . $queryString,
                'path' => $getCountryList->path() . '?' . $queryString,
                'per_page' => $getCountryList->perPage(),
                'prev_page_url' => $getCountryList->previousPageUrl() . '&' . $queryString,
                'to' => $getCountryList->lastItem(),
                'total' => $getCountryList->total(),
            ];

            if ($output) {
                $success = 1;
                return $this->sendResponse($output, 'Country Listed Successfully!', $success);
            } else {
                $success = [];
                return $this->sendError('Country List Not Found', $success, 200);
            }
            //in success response need to send active country list with only fields [country_code, country_name, city_name, latitude, longitude]

        } catch (Exception $ex) {
            return $this->sendError($data, 'Something went wrong', ['error' => $ex->getMessage()], 500);
        }
    }
    /**
     * @OA\Get(
     ** path="/v1/geography/get-country",
     *   tags={"Geography"},
     *   summary="get country list with name arabic and english",
     *   description="get country list <br><br>",
     *   operationId="countryList",
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/
    public function getCountryDetails(Request $request)
    {
        $getCountryDetails = Country::select('countries.id', 'countries.iso_code', 'countries.isd_code')
            ->with(['countryCode' => function ($query) {
                $query->orderBy('country_name', 'asc');
            }])
            ->orderBy('iso_code', 'asc')
            ->get()
            ->toArray();

        // Initialize an empty array to store the transformed data
        $transformedData = [];

        // Iterate through each record in the original data
        foreach ($getCountryDetails as $country) {
            // Extract relevant information from the original record
            $id = $country['id'];
            $isoCode = $country['iso_code'];
            $isdCode = $country['isd_code'];

            // Iterate through the "country_code" array to find language-specific country names
            $countryNameEn = null;
            $countryNameAr = null;

            foreach ($country['country_code'] as $countryCode) {
                if ($countryCode['language_code'] == Locale::English->value) {
                    $countryNameEn = $countryCode['country_name'];
                } elseif ($countryCode['language_code'] == Locale::Arabic->value) {
                    $countryNameAr = $countryCode['country_name'];
                }
            }

            // Build a new array with the desired structure
            $transformedData[] = [
                'id' => $id,
                'iso_code' => $isoCode,
                'isd_code' => $isdCode,
                'country_name_en' => $countryNameEn,
                'country_name_ar' => $countryNameAr,
                'flag' => URL::to('/') . '/storage/country-flags/' . strtolower($isoCode) . '.png',
            ];
        }
        $success = 1;
        return $this->sendResponse($transformedData, 'Country List Fetched Successfully.', $success);
    }
    /**
     * @OA\Get(
     ** path="/v1/geography/get-states",
     *   tags={"Geography"},
     *   summary="get states list with name arabic and english",
     *   description="get states list <br>pass country code to get country related state list<br>",
     *   operationId="stateList",
     *   @OA\Parameter(
     *       name="body",
     *       in="query",
     *       required=false,
     *       explode=true,
     *       @OA\Schema(
     *            collectionFormat="multi",
     *            required={"country_code"},
     *            @OA\Property(property="country_code", type="string",  )
     *       ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/
    public function getStateDetails(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'country_code' => 'required|exists:states,country_code',
        ]);
        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 422);
        }
        $getStateDetails = State::select('states.id', 'states.iso_code', 'states.country_code', 'states.latitude', 'states.longitude')
            ->with(['stateName' => function ($q) {
                $q->orderBy('state_name', 'asc');
            }])
            ->where('country_code', $request->country_code)
            ->orderBy('iso_code', 'asc')
            ->get()
            ->toArray();

        // Initialize an empty array to store the transformed data
        $transformedData = [];

        // Iterate through each record in the original data
        foreach ($getStateDetails as $state) {
            // Extract relevant information from the original record
            $id = $state['id'];
            $isoCode = $state['iso_code'];
            $latitute = $state['latitude'];
            $longitude = $state['longitude'];

            // Iterate through the "country_code" array to find language-specific country names
            $stateNameEn = null;
            $stateNameAr = null;

            foreach ($state['state_name'] as $countryCode) {
                if ($countryCode['language_code'] == Locale::English->value) {
                    $stateNameEn = $countryCode['state_name'];
                } elseif ($countryCode['language_code'] == Locale::Arabic->value) {
                    $stateNameAr = $countryCode['state_name'];
                }
            }

            // Build a new array with the desired structure
            $transformedData[] = [
                'id' => $id,
                'iso_code' => $isoCode,
                'latitude' => $latitute,
                'longitude' => $longitude,
                'state_name_en' => $stateNameEn,
                'state_name_ar' => $stateNameAr,
            ];
        }
        $success = 1;
        return $this->sendResponse($transformedData, 'State List Fetched Successfully', $success);
    }
    /**
     * @OA\Get(
     ** path="/v1/geography/get-city",
     *   tags={"Geography"},
     *   summary="get city list with name arabic and english",
     *   description="get city list <br>pass country code to get country related city list<br>",
     *   operationId="cityList",
     *   @OA\Parameter(
     *       name="body",
     *       in="query",
     *       required=false,
     *       explode=true,
     *       @OA\Schema(
     *            collectionFormat="multi",
     *            required={"country_code"},
     *            @OA\Property(property="country_code", type="string",  )
     *       ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     *)
     **/
    public function getCityDetails(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'country_code' => 'required|exists:cities,country_code',
        ]);
        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 422);
        }
        $getCityDetails = City::select('cities.id', 'cities.iso_code', 'cities.country_code', 'cities.latitude', 'cities.longitude')
            ->with(['cityCode' => function ($q) {
                $q->orderBy('city_name', 'asc');
            }])
            ->where('country_code', $request->country_code)
            ->orderBy('iso_code', 'asc')
            ->get()
            ->toArray();

        // Initialize an empty array to store the transformed data
        $transformedData = [];

        // Iterate through each record in the original data
        foreach ($getCityDetails as $city) {
            // Extract relevant information from the original record
            $id = $city['id'];
            $isoCode = $city['iso_code'];
            $latitute = $city['latitude'];
            $longitude = $city['longitude'];

            // Iterate through the "country_code" array to find language-specific country names
            $cityNameEn = null;
            $cityNameAr = null;

            foreach ($city['city_code'] as $countryCode) {
                if ($countryCode['language_code'] == Locale::English->value) {
                    $cityNameEn = $countryCode['city_name'];
                } elseif ($countryCode['language_code'] == Locale::Arabic->value) {
                    $cityNameAr = $countryCode['city_name'];
                }
            }

            // Build a new array with the desired structure
            $transformedData[] = [
                'id' => $id,
                'iso_code' => $isoCode,
                'latitude' => $latitute,
                'longitude' => $longitude,
                'city_name_en' => $cityNameEn,
                'city_name_ar' => $cityNameAr,
            ];
        }
        $success = 1;
        return $this->sendResponse($transformedData, 'City List Fetched Successfully', $success);
    }

    public function getCityListOld(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'search' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', $validator->messages(), 422);
        }

        $locale = $request->locale;
        $search = $request->search;
        if (strlen($search) < 3) {
            return $this->sendResponse([], 'Search must be at least 3 characters long');
        }


        $query = City::select(
            'country_i18ns.country_name',
            'city_i18ns.city_name',
            'cities.latitude',
            'cities.longitude',
            'cities.sorting',
            'city_en.city_name as city_name_en',
            'city_ar.city_name as city_name_ar',
            'country_en.country_name as country_name_en',
            'country_ar.country_name as country_name_ar'
        )
            ->join("city_i18ns", "city_i18ns.city_id", "=", "cities.id")
            ->join("countries", "countries.iso_code", "=", "cities.country_code")
            ->leftJoin('country_i18ns', function ($join) use ($locale) {
                $join->on('country_i18ns.country_id', '=', 'countries.id');
                if ($locale) {
                    $join->where('country_i18ns.language_code', '=', $locale);
                }
            })
            ->leftJoin('city_i18ns as city_en', function ($join) {
                $join->on('city_en.city_id', '=', 'cities.id')
                    ->where('city_en.language_code', '=', Locale::English->value);
            })
            ->leftJoin('city_i18ns as city_ar', function ($join) {
                $join->on('city_ar.city_id', '=', 'cities.id')
                    ->where('city_ar.language_code', '=', Locale::Arabic->value);
            })
            ->leftJoin('country_i18ns as country_en', function ($join) {
                $join->on('country_en.country_id', '=', 'countries.id')
                    ->where('country_en.language_code', '=', Locale::English->value);
            })
            ->leftJoin('country_i18ns as country_ar', function ($join) {
                $join->on('country_ar.country_id', '=', 'countries.id')
                    ->where('country_ar.language_code', '=', Locale::Arabic->value);
            })
            ->where('cities.status', '=', 'active')
            ->where('city_i18ns.city_name', 'LIKE', '%' . $search . '%');





        $localeQuery = clone $query;
        if ($locale) {
            $localeQuery->where('city_i18ns.language_code', '=', $locale);
        }

        // Execute the locale-prioritized query first
        // $getCityList = $localeQuery->orderBy('city_name', 'asc')->get()->toArray();

        $getCityList = $localeQuery->orderBy('cities.sorting', 'asc')
            ->orderBy('city_i18ns.city_name', 'asc')
            ->get()->toArray();


        if (empty($getCityList)) {
            // $getCityList = $query->orderBy('city_name', 'asc')->get()->toArray();

            $getCityList = $query->orderBy('cities.sorting', 'asc')
                ->orderBy('city_i18ns.city_name', 'asc')
                ->get()->toArray();
        }
        if (!empty($getCityList)) {
            foreach ($getCityList as &$city) {
                $city['type'] = 'city';
                $city['hotel_id'] = '';
                $city['hotel_name'] = '';
            }
        }

        if ($locale == Locale::English->value) {
            $searchQuery = [
                'where' => [
                    ['name_en', 'LIKE', '%' . $search . '%']
                ]
            ];
        } else {
            $searchQuery = [
                'where' => [
                    ['name_ar', 'LIKE', '%' . $search . '%']
                ]
            ];
        }

        $tempHotels =  HotelName::getHotelsName($searchQuery);

        // $searchQuery = [
        //     'where' => [
        //         ['displayname', 'LIKE', '%' . $search . '%']
        //     ]
        // ];
        // $tempHotels = HotelAddress::getHotels($searchQuery);

        /*$searchArray = explode(' ', $search);
		$tempHotels = HotelAddress::query();
		// Base query
		foreach($searchArray as $searchWord) {
			if (!empty(trim($searchWord)) && $searchWord != 'in' && $searchWord != 'hotel') {
				$tempHotels->orWhere('displayname', 'LIKE', '%' . trim($searchWord) . '%');
			}
		}
		$tempHotels = $tempHotels->get();
		echo "<pre>"; print_r($tempHotels); die;*/

        if ($tempHotels) {
            $hotels = [];
            foreach ($tempHotels as $hotel) {
                array_push($hotels, [
                    'type' => 'hotel',
                    'hotel_id' => $hotel['hotel_id'],
                    // 'hotel_name' => $hotel['displayname'],
                    'hotel_name' => $hotel['name_en'],
                    'hotel_name_ar' => $hotel['name_ar'],
                    'city_name' => $hotel['cityname'],
                    'country_name' => $hotel['countryname'],
                    'latitude' => $hotel['lat'],
                    'longitude' => $hotel['lng'],
                ]);
            }
            $getCityList = array_merge($getCityList, $hotels);
        }
        //echo "<pre>"; print_r($getHotelList); die;

        if (!empty($getCityList)) {
            return $this->sendResponse($getCityList, 'Cities have been listed successfully', true);
        } else {
            return $this->sendResponse([], 'No city matches with this search string', false);
        }
    }
    public function getCityList(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'search' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()]);
        }

        $locale = $request->locale;
        $search = $request->search;
        if (strlen($search) < 3) {
            return $this->sendResponse([], 'Please enter at least 3 characters to search', false);
        }


        $query = City::select(
            'country_i18ns.country_name',
            'city_i18ns.city_name',
            'cities.latitude',
            'cities.longitude',
            'cities.sorting',
            'city_en.city_name as city_name_en',
            'city_ar.city_name as city_name_ar',
            'country_en.country_name as country_name_en',
            'country_ar.country_name as country_name_ar'
        )
            ->distinct()
            ->join("city_i18ns", "city_i18ns.city_id", "=", "cities.id")
            ->join("countries", "countries.iso_code", "=", "cities.country_code")
            ->leftJoin('country_i18ns', function ($join) use ($locale) {
                $join->on('country_i18ns.country_id', '=', 'countries.id');
                if ($locale) {
                    $join->where('country_i18ns.language_code', '=', $locale);
                }
            })
            ->leftJoin('city_i18ns as city_en', function ($join) {
                $join->on('city_en.city_id', '=', 'cities.id')
                    ->where('city_en.language_code', '=', Locale::English->value);
            })
            ->leftJoin('city_i18ns as city_ar', function ($join) {
                $join->on('city_ar.city_id', '=', 'cities.id')
                    ->where('city_ar.language_code', '=', Locale::Arabic->value);
            })
            ->leftJoin('country_i18ns as country_en', function ($join) {
                $join->on('country_en.country_id', '=', 'countries.id')
                    ->where('country_en.language_code', '=', Locale::English->value);
            })
            ->leftJoin('country_i18ns as country_ar', function ($join) {
                $join->on('country_ar.country_id', '=', 'countries.id')
                    ->where('country_ar.language_code', '=', Locale::Arabic->value);
            })
            ->where('cities.status', '=', 'active')
            ->where('city_i18ns.city_name', 'LIKE', '%' . $search . '%')
            ->groupBy('cities.latitude', 'cities.longitude');




        $localeQuery = clone $query;
        if ($locale) {
            $localeQuery->where('city_i18ns.language_code', '=', $locale);
        }

        // Execute the locale-prioritized query first
        // $getCityList = $localeQuery->orderBy('city_name', 'asc')->get()->toArray();

        $getCityList = $localeQuery->orderBy('cities.sorting', 'asc')
            ->orderBy('city_i18ns.city_name', 'asc')
            ->get()->toArray();


        if (empty($getCityList)) {
            // $getCityList = $query->orderBy('city_name', 'asc')->get()->toArray();

            $getCityList = $query->orderBy('cities.sorting', 'asc')
                ->orderBy('city_i18ns.city_name', 'asc')
                ->get()->toArray();
        }
        if (!empty($getCityList)) {
            foreach ($getCityList as &$city) {
                $city['type'] = 'city';
                $city['hotel_id'] = '';
                $city['hotel_name'] = '';
            }
        }

        // if ($locale == Locale::English->value) {
        //     $searchQuery = [
        //         'where' => [
        //             ['name_en', 'LIKE', '%' . $search . '%']
        //         ]
        //     ];
        // } else {
        //     $searchQuery = [
        //         'where' => [
        //             ['name_ar', 'LIKE', '%' . $search . '%']
        //         ]
        //     ];
        // }

        // $tempHotels =  HotelName::getHotelsName($searchQuery);

        $searchQuery = [
            'where' => [
                ['displayname', 'LIKE', '%' . $search . '%']
            ]
        ];
        $tempHotels = HotelAddress::getHotels($searchQuery);

        /*$searchArray = explode(' ', $search);
		$tempHotels = HotelAddress::query();
		// Base query
		foreach($searchArray as $searchWord) {
			if (!empty(trim($searchWord)) && $searchWord != 'in' && $searchWord != 'hotel') {
				$tempHotels->orWhere('displayname', 'LIKE', '%' . trim($searchWord) . '%');
			}
		}
		$tempHotels = $tempHotels->get();
		echo "<pre>"; print_r($tempHotels); die;*/

        if ($tempHotels) {
            $hotels = [];
            foreach ($tempHotels as $hotel) {
                array_push($hotels, [
                    'type' => 'hotel',
                    'hotel_id' => $hotel['hotelid'],
                    'hotel_name' => $hotel['displayname'],
                    // 'hotel_name' => $hotel['name_en'],
                    // 'hotel_name_ar' => $hotel['name_ar'],
                    'city_name' => $hotel['cityname'],
                    'country_name' => $hotel['countryname'],
                    'latitude' => $hotel['lat'],
                    'longitude' => $hotel['lng'],
                ]);
            }
            $getCityList = array_merge($getCityList, $hotels);
        }
        //echo "<pre>"; print_r($getHotelList); die;

        if (!empty($getCityList)) {
            return $this->sendResponse($getCityList, 'Cities have been listed successfully', true);
        } else {
            return $this->sendResponse([], 'No city matches with this search string', false);
        }
    }

    public function getCitiesDetails(Request $request)
    {
        $locale = $request->input('locale', Locale::English->value);
        $result = [
            'success' => true,
            'data' => [],
            'message' => ''
        ];
        $requestData = $request->all();
        $cities = [];
        foreach ($requestData['search_params'] as $key => $value) {
            if (in_array($value['originLocationCode'], $cities) == false) {
                array_push($cities, $value['originLocationCode']);
            }
            if (in_array($value['destinationLocationCode'], $cities) == false) {
                array_push($cities, $value['destinationLocationCode']);
            }
        }
        $citiesDetails = [];
        if (count($cities) > 0) {
            foreach ($cities as $cityCode) {
                $cityNameEn = $cityNameAr = $countryCode = $countryNameEn = $countryNameAr = '';
                $cityDetails = DB::table('cities')->join('city_i18ns', 'city_i18ns.city_id', 'cities.id')->select('cities.country_code', 'city_i18ns.city_name')->where('cities.iso_code', $cityCode)->get();
                if ($cityDetails) {
                    $cityNameEn = $cityDetails[0]->city_name;
                    $cityNameAr = $cityDetails[1]->city_name;
                    $countryCode = $cityDetails[0]->country_code;
                } else {
                    $airportCityCode = Airport::where('iata_code', $cityCode)->value('city_code');
                    $cityDetails = DB::table('cities')->join('city_i18ns', 'city_i18ns.city_id', 'cities.id')->select('cities.country_code', 'city_i18ns.city_name')->where('cities.iso_code', $airportCityCode)->get();
                    if ($cityDetails) {
                        $cityNameEn = $cityDetails[0]->city_name;
                        $cityNameAr = $cityDetails[1]->city_name;
                    }
                    $countryCode = Airport::where('iata_code', $cityCode)->value('country_code');
                }
                if ($countryCode) {
                    $countryDetails = DB::table('countries')->join('country_i18ns', 'country_i18ns.country_id', 'countries.id')->select('country_i18ns.country_name')->where('countries.iso_code', $countryCode)->get();
                    $countryNameEn = $countryDetails[0]->country_name;
                    $countryNameAr = $countryDetails[1]->country_name;
                }

                $citiesDetails[$cityCode] = [
                    'city_name' => $cityNameEn,
                    'city_name_ar' => $cityNameAr,
                    'country_name' => $countryNameEn,
                    'country_name_ar' => $countryNameAr,
                ];
            }
        }
        $result['data'] = $citiesDetails;
        return $result;
    }


    public function getTopDestinationCountries()
    {
        $countries = TopDestination::with('country')->get();

        if (empty($countries)) {
            return $this->sendResponse([], 'No top destination countries found', false);
        }

        $transformedData = [];
        foreach ($countries as $country) {
            $transformedData[] = [
                'id' => $country->id,
                'iso_code' => $country->country_code,
                "isd_code" => $country->country->isd_code ?? '',
                "max_mobile_number" => $country->country->max_mobile_number_length ?? '',
                "status" => $country->status == 1 ? 'active' : 'inactive',
                "country_en" => $country->country_name ?? '',
                "country_ar" => $country->country_name_arabic ?? '',
                "flag" => url('storage/country-flags/' . strtolower($country->country_code) . '.png'),
            ];
        }

        return $this->sendResponse($transformedData, 'Top Destination Countries Fetched Successfully', true);
    }

    public function getTopDestinationCities(Request $request)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'country_code' => 'required|exists:top_destinations,country_code',
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 200);
        }


        $topDestination = TopDestination::where('country_code', $request->country_code)->get();

        if ($topDestination->isEmpty()) {
            return $this->sendResponse([], 'No top destination found for this country', true);
        }


        $topDestinationIds = $topDestination->pluck('id');


        $cities = TopDestinationI18s::with('city')->whereIn('top_destination_id', $topDestinationIds)->get();

        if ($cities->isNotEmpty()) {


            $transformedData = [];

            foreach ($cities as $city) {
                $transformedData[] = [
                    'id' => $city->id ?? '',
                    'iso_code' => $city->city_code ?? '',
                    'latitude' => $city->city->latitude ?? '',
                    'longitude' => $city->city->longitude ?? '',
                    'city_name_en' => $city->city_name ?? '',
                    'city_name_ar' => $city->city_name_arabic ?? '',
                    'thumbnail' => asset('storage/' . $city->thumbnail),
                ];
            }

            return $this->sendResponse($transformedData, 'Top Destination Cities Fetched Successfully', true);
        }

        return $this->sendResponse([], 'No top destination cities found for this country', true);
    }

    // For Mobile browser and app
    public function getAllTopDestinationCities(Request $request)
    {

        $cities = TopDestinationI18s::with([
            'city:id,iso_code,latitude,longitude',
            'topDestination:id,country_code,country_name,country_name_arabic'
        ])->get();


        if ($cities->isNotEmpty()) {

            $transformedData = [];

            foreach ($cities as $city) {
                $transformedData[] = [
                    'id' => $city->id ?? '',
                    'iso_code' => $city->city_code ?? '',
                    'latitude' => $city->city->latitude ?? '',
                    'longitude' => $city->city->longitude ?? '',
                    'city_name_en' => $city->city_name ?? '',
                    'city_name_ar' => $city->city_name_arabic ?? '',
                    'country_name_en' => $city->topDestination->country_name ?? '',
                    'country_name_ar' => $city->topDestination->country_name_arabic ?? '',
                    'country_code' => $city->topDestination->country_code ?? '',
                    'thumbnail' => asset('storage/' . $city->thumbnail),
                ];
            }

            return $this->sendResponse($transformedData, 'Popular Cities Fetched Successfully', true);
        }

        return $this->sendResponse([], 'No popular cities found for this country', true);
    }


    // Update City Latitude

    public function updateCityLatitude(Request $request)
    {

        //  $city = 'Aberdeen';
        // $country = 'United Kingdom';
        // $address = $city . ', ' . $country;
        // $apiKey = "AIzaSyA238MUPVd0fUR9SFieFdvdsHBIB3aMLqU";
        // $url = "https://maps.google.com/maps/api/geocode/json?address=" . urlencode($address) . "&key=" . $apiKey;

        // $ch = curl_init();
        // curl_setopt($ch, CURLOPT_URL, $url);
        // curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        // $responseJson = curl_exec($ch);
        // curl_close($ch);

        // $response = json_decode($responseJson);
        // $lat = [
        //   'lat' => $response->results[0]->geometry->location->lat,
        //  'lon' => $response->results[0]->geometry->location->lng,  
        // ];

        // return $lat;

        $duplicateCityNames = City::select('city_i18ns.city_name')
            ->join('city_i18ns', 'cities.id', '=', 'city_i18ns.city_id')
            ->where('city_i18ns.language_code', 'en')
            ->groupBy('city_i18ns.city_name')
            ->havingRaw('COUNT(DISTINCT cities.country_code) > 1')
            ->pluck('city_i18ns.city_name')
            ->toArray();

        if (empty($duplicateCityNames)) {
            return $this->sendResponse([], 'No duplicate cities found.');
        }

        $cities = City::select('cities.id', 'city_i18ns.city_name', 'cities.country_code', 'cities.latitude', 'cities.longitude')
            ->join('city_i18ns', 'cities.id', '=', 'city_i18ns.city_id')
            ->where('city_i18ns.language_code', 'en')
            ->whereIn('city_i18ns.city_name', $duplicateCityNames)
            ->get();

        $countryCodes = $cities->pluck('country_code')->unique()->toArray();

        // Fetch country names
        $countryNames = Country::select('iso_code', 'country_i18ns.country_name')
            ->join('country_i18ns', 'countries.id', '=', 'country_i18ns.country_id')
            ->whereIn('iso_code', $countryCodes)
            ->where('country_i18ns.language_code', 'en')
            ->get()
            ->keyBy('iso_code');

        $debugList = [];


        $groupedCities = $cities->groupBy('city_name');

        foreach ($groupedCities as $cityName => $cityGroup) {
            $correctCity = $cityGroup->first(function ($city) {
                return !is_null($city->latitude) && !is_null($city->longitude);
            });

            if (!$correctCity) {
                continue;
            }

            $citiesToUpdate = $cityGroup->where('id', '!=', $correctCity->id);

            foreach ($citiesToUpdate as $city) {
                $countryName = $countryNames->has($city->country_code) ? $countryNames[$city->country_code]->country_name : 'Country not found';

                $address = $city->city_name . ', ' . $countryName;

                $debugList[] = [
                    'city_id' => $city->id,
                    'city_name' => $city->city_name,
                    'country_code' => $city->country_code,
                    'country_name' => $countryName,
                    'address_for_api' => $address,
                    'current_latitude' => $city->latitude,
                    'current_longitude' => $city->longitude,
                    'source_latitude' => $correctCity->latitude,
                    'source_longitude' => $correctCity->longitude,
                ];
            }
        }
        return $debugList;
        $updatedCount = 0;
        $errors = [];
        $apiKey = '';
        foreach ($debugList as $city) {

            if (empty($city['city_name']) || empty($city['country_name'])) {
                $errors[] = [
                    'city_id' => $city['city_id'],
                    'city_name' => $city['city_name'] ?? 'Unknown',
                    'error' => 'Incomplete address data',
                ];
                continue;
            }

            $address = $city['city_name'] . ', ' . $city['country_name'];
            $url = "https://maps.googleapis.com/maps/api/geocode/json?address=" . urlencode($address) . "&key=" . $apiKey;

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            $responseJson = curl_exec($ch);

            if (curl_errno($ch)) {
                $errors[] = [
                    'city_id' => $city['city_id'],
                    'city_name' => $city['city_name'],
                    'error' => 'cURL error: ' . curl_error($ch),
                ];
                curl_close($ch);
                continue;
            }
            curl_close($ch);

            $response = json_decode($responseJson);

            if ($response && $response->status === 'OK' && isset($response->results[0])) {
                $latitude = $response->results[0]->geometry->location->lat;
                $longitude = $response->results[0]->geometry->location->lng;

                $updated = City::where('id', $city['city_id'])->update([
                    'latitude' => $latitude,
                    'longitude' => $longitude,
                ]);

                if ($updated) {
                    $updatedCount++;
                } else {
                    $errors[] = [
                        'city_id' => $city['city_id'],
                        'city_name' => $city['city_name'],
                        'error' => 'Failed to update city coordinates in the database',
                    ];
                }
            } else {
                $errors[] = [
                    'city_id' => $city['city_id'],
                    'city_name' => $city['city_name'],
                    'error' => $response->status ?? 'Invalid API response',
                ];
            }

            usleep(200000);
        }

        $data = [
            'updated_rows' => $updatedCount,
            'total_checked' => count($debugList),
            'errors' => $errors,
        ];

        return $this->sendResponse($data, 'City latitude and longitude update process completed.');
    }






    public function getCityListElasticSearch(Request $request, HotelElasticSearchService $es)
    {
        $validator = Validator::make($request->all(), [
            'search' => 'required|min:3',
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 422);
        }

        $search = trim($request->search);

        // --- Search cities (higher boost for exact matches) ---
        $cities = $es->search('cities', $search);
        $cityLookup = [];
        $formattedCities = array_map(function ($city) use (&$cityLookup) {
            $key = strtolower($city['city_name'] ?? '');
            if ($key) {
                $cityLookup[$key] = $city;
            }

            return [
                'type'            => 'city',
                'hotel_id'        => '',
                'hotel_name'      => '',
                'city_name'       => $city['city_name'] ?? '',
                'country_name'    => $city['country_name_en'] ?? '',
                'latitude'        => $city['latitude'] ?? null,
                'longitude'       => $city['longitude'] ?? null,
                'sorting'         => $city['sorting'] ?? 0,
                'city_name_en'    => $city['city_name_en'] ?? '',
                'city_name_ar'    => $city['city_name_ar'] ?? '',
                'country_name_en' => $city['country_name_en'] ?? '',
                'country_name_ar' => $city['country_name_ar'] ?? '',
                '_score'          => $city['_score'] ?? 0,
                'exact_match'     => false,
            ];
        }, $cities);

        // --- Search hotels (lower boost for exact matches) ---
        $hotels = $es->search('hotels', $search);

        $lazyCityCache = [];

        $formattedHotels = array_map(function ($hotel) use ($search, $es, &$cityLookup, &$lazyCityCache) {
            $hotelName = $hotel['hotel_name_en'] ?? $hotel['hotel_name'] ?? '';
            $exactMatch = (strcasecmp($hotelName, $search) === 0);
            $cityKey = strtolower($hotel['city_name'] ?? '');
            $cityInfo = $cityLookup[$cityKey] ?? null;

            if (!$cityInfo && $cityKey) {
                if (!isset($lazyCityCache[$cityKey])) {
                    $cityResults = $es->search('cities', $hotel['city_name']);
                    $lazyCityCache[$cityKey] = $cityResults[0] ?? [];
                }
                $cityInfo = $lazyCityCache[$cityKey];
            }

            return [
                'type'            => 'hotel',
                'hotel_id'        => $hotel['hotel_id'] ?? '',
                'hotel_name'      => $hotelName,
                'hotel_name_en'   => $hotel['hotel_name_en'] ?? '',
                'hotel_name_ar'   => $hotel['hotel_name_ar'] ?? '',
                'city_name'       => $hotel['city_name'] ?? '',
                'country_name'    => $hotel['country_name'] ?? '',
                'latitude'        => $hotel['latitude'] ?? null,
                'longitude'       => $hotel['longitude'] ?? null,
                'sorting'         => $hotel['sorting'] ?? 0,
                'city_name_en'    => $cityInfo['city_name_en'] ?? '',
                'city_name_ar'    => $cityInfo['city_name_ar'] ?? '',
                'country_name_en' => $cityInfo['country_name_en'] ?? '',
                'country_name_ar' => $cityInfo['country_name_ar'] ?? '',
                '_score'          => $hotel['_score'] ?? 0,
                'exact_match'     => $exactMatch,
            ];
        }, $hotels);


        $results = array_merge($formattedCities, $formattedHotels);

        usort($results, function ($a, $b) {
            return $b['_score'] <=> $a['_score'];
        });

        return !empty($results)
            ? $this->sendResponse($results, 'Cities & Hotels listed successfully', true)
            : $this->sendResponse([], 'No results found', false);
    }




    public function exportAllCountriesJson()
    {
        try {
            // Get all active countries with relationships
            $countries = Country::exportAllCountriesJson();
            if ($countries) {

                return $this->sendResponse($countries, 'Country list exported successfully');
            }
            return $this->sendError('Unable to export Countries', []);
        } catch (\Exception $ex) {
            return $this->sendError('Failed to export countries', $ex->getMessage());
        }
    }

    public function getAllCountriesJsonOld()
    {

        $path = storage_path('app/public/country_search/countries.json');

        if (!file_exists($path)) {
            return $this->sendError('Country list not found', [], 200);
        }

        $jsonContent = file_get_contents($path);


        $countries = json_decode($jsonContent, true);

        if (!is_array($countries)) {
            return $this->sendError('Invalid country list data', [], 500);
        }

        return $this->sendResponse($countries, 'Country list retrieved successfully', true);
    }


    public function getAllCountriesJson()
    {
        // return Cache::forget('countries_list');
        $countries = Cache::rememberForever('countries_list', function () {
            $path = storage_path('app/public/country_search/countries.json');

            if (!file_exists($path)) {
                return null;
            }

            $jsonContent = file_get_contents($path);
            $decoded = json_decode($jsonContent, true);

            if (!is_array($decoded)) {
                return null;
            }

            return $decoded;
        });

        if ($countries === null) {
            return $this->sendError('Country list not found', [], 200);
        }

        return $this->sendResponse($countries, 'Country list retrieved successfully', true);
    }
}
