<?php

namespace App\Http\Controllers\API\V1;

use DateTime;
use Carbon\Carbon;
use App\Models\City;
use App\Models\State;
use App\Models\Agency;
use App\Models\Coupon;
use App\Models\Wallet;
use App\Enums\Currency;
use App\Models\Country;
use App\Models\Setting;
use App\Models\Bookings;
use App\Models\Customer;
use App\Traits\ActiveLog;
use App\Traits\SmsService;
use Illuminate\Support\Str;
use App\Enums\BookingStatus;
use App\Models\AdminUserOtp;
use App\Traits\EmailService;
use Illuminate\Http\Request;
use App\Enums\CustomerStatus;
use App\Traits\WhatsappService;
use App\Models\CustomerDeviceId;
use App\Models\CustomerLoginOtp;
use App\Models\UserLoginHistory;
use App\Models\CustomerAddresses;
use App\Models\CustomerTraveller;
use App\Models\CustomerLoginToken;
use Illuminate\Support\Facades\DB;
use Laravel\Passport\HasApiTokens;
use App\Http\Controllers\Controller;
use App\Traits\LoyaltyPointsService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\CustomerLoyaltyPoints;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\API\V1\BaseController as BaseController;

class CustomerController extends BaseController
{
    use EmailService, SmsService, ActiveLog, LoyaltyPointsService, WhatsappService;

    /**
     * @OA\Post(
     *   path="/v1/customer/customer-login",
     *   tags={"Customer"},
     *   summary="Customer login",
     *   description="Pass language code either en or ar<br>Pass login_with key either mobile or email<br>
        Pass isd code example:+91<br>
        if login with email pass email address Ex:example@gmail.com </br>
            or<br>
        if login with mobile pass mobile number Ex: 9999999999<br>
        Note: isd code field is required when login with mobile",
     *   operationId="customerLogin",
     *   @OA\RequestBody(
     *     required=true,
     *     description="Login Details",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"login_with","mobile_or_email","password"},
     *             @OA\Property(property="language_code",default="en", type="string", description="enter value for language code either 'en' or 'ar'"),
     *             @OA\Property(property="login_with", type="string", description="enter value for login_with either mobile or email"),
     *             @OA\Property(property="isd_code", type="string", description="enter isd code"),
     *             @OA\Property(property="mobile_or_email", type="string", description="enter existing email address or mobile"),
     *             @OA\Property(property="password", type="string" , title="password", description="enter password"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Login api
     *
     * @return \Illuminate\Http\Response
     */
    public function login(Request $request)
    {

        try {

            //check validation for either mobile or email using extends method
            // Validator::extend('check_type', function ($attribute, $value, $parameters, $validator) {
            //     $condition = $parameters[0]; // Get the condition field name
            //     $conditionValue = $validator->getData()[$condition]; // Get the value of the condition field

            //     if ($conditionValue === 'mobile') {
            //         // Check if the value is a valid mobile number
            //         return preg_match('/^[0-9]+$/', $value);
            //     } elseif ($conditionValue === 'email') {
            //         // Check if the value is a valid email address
            //         return filter_var($value, FILTER_VALIDATE_EMAIL);
            //     }

            //     return false; // Invalid condition value
            // }, 'The customer :conditionValue is in-valid');
            // Validator::replacer('check_type', function ($message, $attribute, $rule, $parameters) use ($request) {
            //     $signupWith = $request->input('login_with'); // Access signup_with from the request
            //     return str_replace(':conditionValue', $signupWith, $message);
            // });
            //isd code validation
            // Validator::extend('valid_isd_code', function ($attribute, $value, $parameters, $validator) {
            //     // Implement the validation logic for ISD code
            //     return preg_match('/^\+\d+([- ]\d+)?$/', $value);
            // }, "The :attribute must start with + followed by numbers, and may be followed by - or space and more numbers.");
            $validator = Validator::make($request->all(), [
                'language_code' => 'nullable|in:en,ar',
                'login_with' => 'required|in:email,mobile',
                'isd_code' => 'nullable',
                'mobile_or_email' => 'required',
                'password' => 'required',

            ]);
            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }
            $validator->sometimes('isd_code', 'valid_isd_code', function ($input) {
                return $input->type == 'mobile';
            });
            $inputType = self::checkInputType($request->mobile_or_email);
            if ($inputType == "mobile") {
                // $customer = Customer::where('mobile', '=', $request->isd_code . ' ' . $request->mobile_or_email)->where('status', '!=', 'deleted')->get()->first();
                $customer = Customer::whereRaw("TRIM(SUBSTRING_INDEX(mobile, ' ', -1)) = ?", [$request->mobile_or_email])
                    ->latest()->first();
            } else {

                $customer = Customer::where('email', '=', $request->mobile_or_email)->latest()->first();
            }

            if (!empty($customer)) {

                //check that current host if avaible for login user
                $curentDateTime = date("Y-m-d H:i:s");
                $loginAttemptsDetail = DB::table('user_login_attempts')->Where('host', $_SERVER['REMOTE_ADDR'])->where('next_login_available_at', '>', $curentDateTime)->latest('next_login_available_at')->get()->toArray();
                $lockoutTimePerUserOrHostType = Setting::where('config_key', 'loginAttempts|lockOutTimePeriodType')->get('value')[0]['value'];
                if (!empty($loginAttemptsDetail)) {
                    $datetime1 = new DateTime($curentDateTime);
                    $datetime2 = new DateTime($loginAttemptsDetail[0]->next_login_available_at);
                    $interval = $datetime1->diff($datetime2);
                    $days = $interval->d;
                    $hours = $interval->h;
                    $minutes = $interval->i;
                    $secounds = $interval->s;
                    $success = [];
                    if ($lockoutTimePerUserOrHostType == 'minute') {
                        if ($minutes == '0') {

                            return $this->sendError('Your account has been locked, you can login after ' . $secounds . ' seconds', $success, 200);
                        } else {
                            return $this->sendError('Your account has been locked, you can login after ' . $minutes . ' minutes', $success, 200);
                        }
                    }
                    if ($lockoutTimePerUserOrHostType == 'hour') {
                        if ($hours != '0') {
                            return $this->sendError('Your account has been locked, you can login after ' . $hours . ' hour', $success, 200);
                        } else if ($minutes == '0') {

                            return $this->sendError('Your account has been locked, you can login after ' . $secounds . ' seconds', $success, 200);
                        } else {
                            return $this->sendError('Your account has been locked, you can login after ' . $minutes . ' minutes', $success, 200);
                        }
                    }
                    if ($lockoutTimePerUserOrHostType == 'day') {
                        if ($days != '0') {
                            return $this->sendError('Your account has been locked, you can login after ' . $days . ' day', $success, 200);
                        } else if ($hours != '0') {
                            return $this->sendError('Your account has been locked, you can login after ' . $hours . ' hour', $success, 200);
                        } else if ($minutes == '0') {

                            return $this->sendError('Your account has been locked, you can login after ' . $secounds . ' seconds', $success, 200);
                        } else {
                            return $this->sendError('Your account has been locked, you can login after ' . $minutes . ' minutes', $success, 200);
                        }
                    }
                }

                if (self::checkValidCustomer($customer)) {
                    $success = [];
                    return $this->sendError('Your Account has been ' . $customer->status . ' Please contact customersupport@rehlte.com.', $success, 200);
                }
                if (!Hash::check($request->password, $customer['password'])) {

                    $isEnableLoginAttempt = Setting::where('config_key', 'loginAttempts|enable')->get('value')[0]['value'];
                    if ($customer != "" && $isEnableLoginAttempt == '1') {



                        //function to check login attempt when credential are wrong
                        $checkLoginAttempt = self::checkLoginAttempt($customer->id, $request->language_code);

                        $success = [];

                        return $this->sendError($checkLoginAttempt, $success, 200);
                    } else {
                        $success = [];
                        return $this->sendError('Wrong password. Try again or click Forgot password to reset it.', $success, 200);
                    }
                } else {

                    $checkAgencyStatus = Agency::where('id', $customer['agency_id'])->value('status');
                    if ($checkAgencyStatus == CustomerStatus::Active->value) {
                        //set route redirection for b2c
                        $success[] = 1;
                        return $this->sendResponse([$success], 'Welcome! You’ve been successfully logged in');
                    } else {

                        //send mail to customer if agency status is inavtive or terminated
                        $agencyLogo = Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'] ?? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'];
                        $code = 'AGENCY_BLOCK';
                        $siteName = count(Setting::where('config_key', 'general|basic|siteName')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'] : "Amar Infotech";
                    }
                }

                if (Auth::guard('appuser')->loginUsingId($customer->id)) {
                    $customerData = Auth::guard('appuser')->user();

                    if ($customer->status == CustomerStatus::InActive->value) {
                        $success = [];
                        return $this->sendError('User is In-Active, please contact to administrator', $success, 200);
                    } else if ($customer->status == CustomerStatus::Deleted->value || $customer->status == CustomerStatus::Terminated->value) {
                        $success = [];
                        return $this->sendError('You no longer access your account. Please create a new one to continue', $success, 200);
                    } else if ($customer->is_mobile_verified ==  0) {
                        $success = [];
                        return $this->sendError('Please verify your email address or mobile number to proceed.', $success, 200);
                    } else {
                        $success = $customer;
                        $success['token'] = $customerData->createToken('AuthToken')->accessToken;
                        $success['isPasswordSet'] = $customer->password ? true : false;
                        $success['profile_photo'] = $customer->profile_photo ?  url($customer->profile_photo) : null;
                        return $this->sendResponse($success, 'Welcome! You’ve been successfully logged in');
                    }
                } else {
                    $success = [];
                    return $this->sendError('Error During Login', $success, 200);
                }
            } else {

                $success = [];
                return $this->sendError('Not Found!. Please verify the provided information and try again.', $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }
    //check login attempts criteria based o login attempts preference
    function checkLoginAttempt($username, $language_code)
    {
        $loginAttemptPerHost = Setting::where('config_key', 'loginAttempts|perHost')->get('value')[0]['value'];
        $loginAttemptPerUser = Setting::where('config_key', 'loginAttempts|perUser')->get('value')[0]['value'];
        $isEnableNotification = Setting::where('config_key', 'loginAttempts|emailNotification')->get('value')[0]['value'];
        $lockoutTimePerUserOrHost = Setting::where('config_key', 'loginAttempts|lockOutTimePeriod')->get('value')[0]['value'];
        $lockoutTimePerUserOrHostType = Setting::where('config_key', 'loginAttempts|lockOutTimePeriodType')->get('value')[0]['value'];
        $lockoutTimePeriod = Setting::where('config_key', 'loginAttempts|lockOutTimePeriod')->get('value')[0]['value'];
        $lockoutTimePeriodType = Setting::where('config_key', 'loginAttempts|lockOutTimePeriodType')->get('value')[0]['value'];
        $loginTimePeriod = Setting::where('config_key', 'loginAttempts|loginTimePeriod')->get('value')[0]['value'];
        $loginTimePeriodType = Setting::where('config_key', 'loginAttempts|loginTimePeriodType')->get('value')[0]['value'];
        $getTimeZone = count(Setting::where('config_key', 'general|site|timeZone')->get('value')) > 0 ? Setting::where('config_key', 'general|site|timeZone')->get('value')[0]['value'] : "Asia/Kolkata";


        //modify date of lockouttimeperiod
        $nextAttemptDay = Carbon::now()->modify("+" . $lockoutTimePeriod . " " . $lockoutTimePeriodType);
        $modifyNextAttemptDate = $nextAttemptDay->format('Y-m-d H:i:s');

        //modify date of logintimeperiod
        $loginAttemptDate = Carbon::now()->modify("+" . $loginTimePeriod . " " . $loginTimePeriodType);
        $modifyLoginAttemptDate = $loginAttemptDate->format('Y-m-d H:i:s');

        //get current datetime
        $currentDate = date("Y-m-d H:i:s");

        //check if next attempt datetime available for the same user and it's greater than now
        $isNextDateAvailable = DB::table('user_login_attempts')->where('username', $username)->where('next_login_available_at', '<', $currentDate)->get()->toArray();

        //check wrong attempt count within givin login time period type
        $checkAttemptCount = Db::table('user_login_attempts')->where('username', $username)->where('attempt_at', '<', $modifyLoginAttemptDate)->count();
        if (!empty($isNextDateAvailable)) {
            DB::table('user_login_attempts')->where('username', $username)->delete();
        }


        $loginDetails = [
            'host' => $_SERVER['REMOTE_ADDR'],
            'username' => $username,
            'attempt_at' => date('Y-m-d H:i:s'),
            'created_at' => now(),
            'updated_at' => now(),
        ];
        $data = DB::table('user_login_attempts')->insert($loginDetails);
        $attemptPerHost = Db::table('user_login_attempts')->where('host', $loginDetails['host'])->count();
        $attemptPerUser = Db::table('user_login_attempts')->where('username', $loginDetails['username'])->count();

        if ($attemptPerUser == $loginAttemptPerHost) {

            $lastRow = DB::table('user_login_attempts')->latest()->first();
            DB::table('user_login_attempts')->where('id', $lastRow->id)->update([
                'next_login_available_at' => $modifyNextAttemptDate
            ]);
            if ($isEnableNotification == '1') {

                $userData = Customer::where('id', $username)->get()->toArray();

                if ($userData[0]['agency_id'] == '0') {
                    $agencyLogo = count(Setting::where('config_key', 'general|basic|colorLogo')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'] : Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'];
                    $agencyName = count(Setting::where('config_key', 'general|basic|siteName')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'] : "Amar Infotech";
                } else {

                    $agencyName = Agency::where('id', $userData[0]['agency_id'])->value('full_name');
                    $agencyLogo = Agency::where('id', $userData[0]['agency_id'])->value('logo');
                }
                $code = 'LOGIN_ATTEMPTS_EXCEED';
                $customerName = ucwords($userData[0]['first_name']);
                $data = array(
                    'customer_name' => $customerName,
                    'agency_name' => $agencyName,
                    'agency_logo' => $agencyLogo,
                    'hours' => $lockoutTimePerUserOrHost,
                    'attempt_count' => $attemptPerUser,
                    'ip_address' => $_SERVER['REMOTE_ADDR'],
                    'time' => date('Y-m-d H:i:s'),
                    'duration' => ucwords($lockoutTimePerUserOrHostType)
                );

                $getTemplateData = $this->mailTemplateBlockAccount($code, $data, $language_code);
                if ($getTemplateData['status'] == 'false') {

                    return back()->with('error', $getTemplateData['error']);
                } else {
                    $subject = $getTemplateData['data']['subject'];
                    $mailData = $getTemplateData['data']['mailData'];
                    $toEmail = $userData[0]['email'];
                    $files = [];

                    // set data in sendEmail function
                    $data = $this->sendEmail($toEmail, $subject, $mailData, $files, $agencyName, $code);

                    $data = array(
                        'customer_name' => $customerName,
                        'customer_email' => $userData[0]['email'],
                        'agency_name' => $agencyName,
                        'agency_logo' => $agencyLogo,
                        'hours' => $lockoutTimePerUserOrHost,
                        'attempt_count' => $attemptPerUser,
                        'ip_address' => $_SERVER['REMOTE_ADDR'],
                        'time' => date('Y-m-d H:i:s'),
                        'duration' => ucwords($lockoutTimePerUserOrHostType)
                    );
                    $code = 'LOGIN_ATTEMPTS_NOTIFY_ADMIN';
                    $getLoginAttemptNotifyTemplate = EmailService::loginAttemptNotifyTemplete($code, $data,  $language_code);
                    // return $getLoginAttemptNotifyTemplate;
                    $subject = $getLoginAttemptNotifyTemplate['data']['subject'];
                    $mailData = $getLoginAttemptNotifyTemplate['data']['mailData'];
                    $toEmail = env('MAIL_FROM_ADDRESS');
                    $files = [];

                    // set data in sendEmail function
                    $data = $this->sendEmail($toEmail, $subject, $mailData, $files, $agencyName, $code);
                    return 'Your account has been locked, you can login after ' . $lockoutTimePeriod . ' ' . $lockoutTimePeriodType;
                }
            } else {
                return 'Your account has been locked, you can login after ' . $lockoutTimePeriod . ' ' . $lockoutTimePeriodType;
            }
        }

        $userAttemptTimeperiodCount = Db::table('user_login_attempts')->where('username', $loginDetails['username'])->where('attempt_at', '<', $modifyLoginAttemptDate)->count();
        if ($loginAttemptPerUser == $userAttemptTimeperiodCount) {

            $lastRow = DB::table('user_login_attempts')->latest()->first();
            DB::table('user_login_attempts')->where('id', $lastRow->id)->update([
                'next_login_available_at' => $modifyNextAttemptDate
            ]);
            if ($isEnableNotification == '1') {

                $userData = Customer::where('id', $username)->get()->toArray();
                if ($userData[0]['agency_id'] == '0') {
                    $agencyLogo = count(Setting::where('config_key', 'general|basic|colorLogo')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'] : Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'];
                    $agencyName = count(Setting::where('config_key', 'general|basic|siteName')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'] : "Amar Infotech";
                } else {

                    $agencyName = Agency::where('id', $userData[0]['agency_id'])->value('full_name');
                    $agencyLogo = Agency::where('id', $userData[0]['agency_id'])->value('logo');
                }
                $code = 'LOGIN_ATTEMPTS_EXCEED';

                $customerName = ucwords($userData[0]['first_name']);

                $data = array(
                    'customer_name' => $customerName,
                    'agency_name' => $agencyName,
                    'agency_logo' => $agencyLogo,
                    'hours' => $lockoutTimePerUserOrHost,
                    'duration' => ucwords($lockoutTimePerUserOrHostType)


                );

                $getTemplateData = $this->mailTemplateBlockAccount($code, $data, $language_code);
                if ($getTemplateData['status'] == 'false') {
                    return back()->with('error', $getTemplateData['error']);
                } else {
                    $subject = $getTemplateData['data']['subject'];
                    $mailData = $getTemplateData['data']['mailData'];
                    $toEmail = $userData[0]['email'];
                    $files = [];

                    // set data in sendEmail function
                    $data = $this->sendEmail($toEmail, $subject, $mailData, $files, $agencyName, $code);
                    return 'Your account has been locked, you can login after ' . $lockoutTimePeriod . ' ' . $lockoutTimePeriodType;
                }
            } else {
                return 'Your account has been locked, you can login after ' . $lockoutTimePeriod . ' ' . $lockoutTimePeriodType;
            }
        } else {
            return 'Wrong password. Try again or click Forgot password to reset it.';
        }
    }
    /**
     * @OA\Post(
     *   path="/v1/customer/customer-signup",
     *   tags={"Customer"},
     *   summary="Customer signup",
     *   description="Send request for Customer Sign Up using mobile or email",
     *   operationId="customerSignup",
     *   @OA\RequestBody(
     *     required=true,
     *     description="Customer Sign Up",
     *     @OA\MediaType(
     *       mediaType="multipart/form-data",
     *       @OA\Schema(
     *             required={"language_code","signup_with","customer_mobile_or_email","password" },
     *             @OA\Property(property="language_code",default="en", type="string",description="enter language code either 'en' or 'ar'"),
     *             @OA\Property(property="signup_with", type="string",description="enter value either 'mobile' or 'email'"),
     *             @OA\Property(property="isd_code", type="string",description="isd code is required when signup with mobile number"),
     *             @OA\Property(property="customer_mobile_or_email", type="string",description="enter mobile number if signup_with value is 'mobile' else email address"),
     *             @OA\Property(property="password", type="string",description="enter a password"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */


    //  Calculate Otp Attempt while sign up

    function checkSignUpOtpAttempt($username, $language_code)
    {
        // Load settings from DB
        $loginAttemptPerUser = Setting::where('config_key', 'signUpOtpAttempts|perCust')->value('value');
        $lockoutTimePeriod = Setting::where('config_key', 'signUpOtpAttempts|lockOutTimePeriod')->value('value');
        $lockoutTimePeriodType = Setting::where('config_key', 'signUpOtpAttempts|lockOutTimePeriodType')->value('value');
        $loginTimePeriod = Setting::where('config_key', 'signUpOtpAttempts|SignUpTimePeriod')->value('value');
        $loginTimePeriodType = Setting::where('config_key', 'signUpOtpAttempts|SignUpTimePeriodType')->value('value');

        $now = now();
        $currentDate = $now->format("Y-m-d H:i:s");

        $unlockTime = $now->copy()->add($lockoutTimePeriodType, $lockoutTimePeriod)->format('Y-m-d H:i:s');
        $attemptWindowStart = $now->copy()->sub($loginTimePeriodType, $loginTimePeriod)->format('Y-m-d H:i:s');

        // Check if the number is currently locked
        $existingLock = DB::table('customer_otp_attempts')
            ->where('username', $username)
            ->where('next_login_available_at', '>', $currentDate)
            ->first();

        if ($existingLock) {
            return [
                'status' => true,
                'message' => "You have reached the maximum limit. Please try again after {$lockoutTimePeriod} {$lockoutTimePeriodType}."
            ];
        }

        // Clear expired lock
        DB::table('customer_otp_attempts')
            ->where('username', $username)
            ->where('next_login_available_at', '<', $currentDate)
            ->delete();

        // Count recent attempts for this number
        $attemptPerUser = DB::table('customer_otp_attempts')
            ->where('username', $username)
            ->where('attempt_at', '>', $attemptWindowStart)
            ->count();

        // Check if limit exceeded
        if ($attemptPerUser >= $loginAttemptPerUser) {
            DB::table('customer_otp_attempts')->insert([
                'host' => request()->ip(),
                'username' => $username,
                'attempt_at' => $currentDate,
                'next_login_available_at' => $unlockTime,
                'created_at' => $currentDate,
                'updated_at' => $currentDate,
            ]);

            return [
                'status' => true,
                'message' => "You have reached the maximum limit. Please try again after {$lockoutTimePeriod} {$lockoutTimePeriodType}."
            ];
        }

        DB::table('customer_otp_attempts')->insert([
            'host' => request()->ip(),
            'username' => $username,
            'attempt_at' => $currentDate,
            'created_at' => $currentDate,
            'updated_at' => $currentDate,
        ]);

        return [
            'status' => false,
            'message' => ''
        ];
    }



    public function signUpCustomer(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'type' => 'required',
                'language_code' => 'required',
                'searchFrom' => 'required|in:b2c_desktop,b2c_mobile_app,b2c_mobile_browser',
                'deviceID' => 'nullable'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation failed', $validator->errors(), 200);
            }

            $type = $request->type;

            if ($type === 'initial_signup' || $type === 'verify_otp' || $type === 'signup' || $type === 'resend_otp' || $type === 'send_otp') {
                $validator = Validator::make($request->all(), [
                    'isd_code' => 'required',
                    'customer_mobile' => 'required'
                ]);

                if ($validator->fails()) {
                    return $this->sendError('Validation failed', $validator->errors(), 200);
                }
                $isdCode = '+' . str_replace('+', '', $request->isd_code);
                $phoneNumber = $request->customer_mobile;
                $country = Country::where('isd_code', $isdCode)->first();

                if (!$country || strlen($phoneNumber) !== $country->max_mobile_number_length) {
                    return $this->sendResponse(null, 'Invalid Mobile Number. Please enter a valid mobile number', false);
                }
            }
            // get OTP verification setting key value 'general|otp|phoneVerification'
            $otp_verification = false;
            $otp_setting_data = \App\Models\Setting::where('config_key', 'general|otp|phoneVerification')->get()->toArray();

            if ($type === 'initial_signup' || $type === 'resend_otp' || $type === 'send_otp') {
                $customer = Customer::where('mobile', $isdCode . ' ' . $phoneNumber)->where('status', '!=', CustomerStatus::Deleted->value)->first();

                if ($customer?->status == CustomerStatus::Terminated->value || $customer?->status == CustomerStatus::InActive->value) {
                    return $this->sendResponse(null, 'You no longer access your account. Please create new one to continue', false);
                }

                if ($type != 'send_otp'  && $type != 'resend_otp' &&  $customer && $customer?->status != CustomerStatus::Deleted->value) {
                    return $this->sendResponse(null, 'Customer already exists with the provided information.', false);
                }
                $checkOtpAttempt = self::checkSignUpOtpAttempt($phoneNumber, $request->language_code);

                if ($checkOtpAttempt['status']) {
                    return $this->sendResponse(null, $checkOtpAttempt['message'], false);
                }

                CustomerLoginOtp::where('mobile', $phoneNumber)->delete();

                $otp = rand(1000, 9999);

                $created = CustomerLoginOtp::create([
                    'phone_code' => $isdCode,
                    'mobile' => $phoneNumber,
                    'otp' => $otp,
                    'expired_at' => Carbon::now()->addMinute(1),
                ]);

                if ($created) {
                    $otp_expire = 1;
                    $temp = array(
                        "otp" => $otp,
                        "mobile" => $isdCode . ' ' . $phoneNumber,
                        "expired" => $otp_expire,

                    );

                    $message = "$otp is your One-Time Password (OTP) to complete your registration on Rehlte. Please use this OTP to create your account. It will remain valid for $otp_expire minutes.";
                    if (!empty($otp_setting_data)) {
                        if ($otp_setting_data[0]['value'] == "on") {
                            $otp_verification = true;
                            //send OTP SMS code
                            $mobile_number = $isdCode . $phoneNumber;
                            // $mobile_number = '+966533949225';
                            $isWhatsappSent = $this->sendWhatsappmessageOtp($mobile_number, $otp);
                            $this->sendSms($mobile_number, $message);
                            $data = [
                                'otp' => $otp,
                            ];
                            if (!$isWhatsappSent) {
                                return $this->sendResponse(null, 'Something went wrong. Unable to send otp', false);
                            }
                            return $this->sendResponse([], 'OTP sent to your registered Mobile Number', true);
                        }
                    }
                    return $this->sendResponse(null, 'Something went wrong.Please Try Again', false);
                }
            } elseif ($type === 'verify_otp') {
                $validator = Validator::make($request->all(), [
                    'isd_code' => 'required',
                    'customer_mobile' => 'required',
                    'otp' => 'required|numeric',
                    'deviceId' => 'nullable',
                ]);

                if ($validator->fails()) {
                    return $this->sendError('Validation failed', $validator->errors(), 200);
                }

                $otp = $request->otp;

                $isOtpValid = CustomerLoginOtp::where('mobile', $phoneNumber)
                    ->where('otp', $otp)
                    ->first();

                if ($isOtpValid) {
                    if (Carbon::parse($isOtpValid->expired_at)->isFuture()) {
                        $token = Str::random(16);
                        $isOtpValid->is_verified = 1;
                        CustomerLoginOtp::where('mobile', $phoneNumber)
                            ->where('otp', $otp)->update([
                                'is_verified' => 1,
                                'token' => $token,
                            ]);
                        // Store Device Id
                        $deviceIdExists = CustomerDeviceId::where('mobile_no', $request->customer_mobile)->where('device_id', $request->deviceId)->exists();
                        if (!$deviceIdExists) {
                            CustomerDeviceId::create([
                                'customer_id' => Auth::id() ?? 'Guest',
                                'device_id' => $request->deviceId,
                                'mobile_code' => $isdCode,
                                'mobile_no' => $phoneNumber,
                                'is_verified' => 1,
                                'search_from'  => $request->searchFrom
                            ]);
                        }
                        $data = [
                            "token" => $token
                        ];
                        return $this->sendResponse($data, 'OTP Verified Successfully', true);
                    } else {
                        return $this->sendResponse(null, 'OTP has expired. Please request a new one.', false);
                    }
                }
                return $this->sendResponse(null, 'Invalid  / Expired OTP. Please Try Again', false);
            } elseif ($type === 'signup') {
                $validator = Validator::make($request->all(), [
                    'isd_code' => 'required',
                    'customer_mobile' => 'required',
                    'first_name' => 'required|string',
                    'last_name' => 'required|string',
                    'email' => 'required',
                    'password' => 'required',
                    'searchFrom' => 'required|in:b2c_desktop,b2c_mobile_app,b2c_mobile_browser',
                    'token' => 'required',

                ]);



                if ($validator->fails()) {
                    return $this->sendError('Validation failed', $validator->errors(), 200);
                }
                $customer = Customer::where('mobile', $isdCode . ' ' . $phoneNumber)->orWhere('email', $request->email)->where('status', '!=', CustomerStatus::Deleted->value)->first();
                if ($customer) {
                    return $this->sendResponse(null, 'This Email is already registered. Please try with a different email.', false);
                }

                $customerOTP = CustomerLoginOtp::where('mobile', $phoneNumber)->where('token', $request->token)->first();
                if ($customerOTP) {

                    if ($customerOTP->is_verified == 0) {
                        return $this->sendResponse(null, 'Please Verify your Mobile Number before signup', false);
                    }
                    if ($customerOTP->token != $request->token) {
                        return $this->sendResponse(null, 'Invalid Token', false);
                    }
                } else {
                    return $this->sendResponse(null, 'Invalid Token / Mobile Number not verified', false);
                }

                $customer = Customer::create([
                    'first_name' => $request->first_name,
                    'last_name' => $request->last_name,
                    'mobile_code' => $isdCode,
                    'mobile' => $isdCode . ' ' . $phoneNumber,
                    'email' => $request->email,
                    'password' => Hash::make($request->password),
                    'is_mobile_verified' => 1,
                ]);

                if ($customer) {
                    $isOtpValid = CustomerLoginOtp::where('mobile', $phoneNumber)
                        ->delete();
                    $token = $customer->createToken('AuthToken')->accessToken;
                    $isPasswordSet = $customer->password ? true : false;

                    $responseData = Customer::where('mobile', $isdCode . ' ' . $phoneNumber)->first();
                    $responseData['token'] = $token;
                    $responseData['isPasswordSet'] = $isPasswordSet;

                    $isMail = Setting::where('config_key', 'mail|smtp|server')->value('value');

                    if ($isMail == '0') {
                        return $this->sendResponse($responseData, 'Customer SignUp Saved Successfully', true);
                    }

                    if (!empty($customer->email)) {
                        $languageCode = Setting::where('config_key', 'general|site|defaultLanguageCode')->value('value');
                        $welcomeAgencyCode = 'WELCOME_AGENCY';
                        $agencyLogo = Setting::where('config_key', 'general|basic|colorLogo')->value('value') ??
                            Setting::where('config_key', 'general|basic|siteName')->value('value');
                        $siteName = Setting::where('config_key', 'general|basic|siteName')->value('value') ?? 'Amar Infotech';

                        $userData = [
                            'agency_logo' => $agencyLogo,
                            'customer_name' => 'customer',
                            'agency_name' => $siteName,
                            'password' => $request->password,
                        ];

                        $template = EmailService::customerWelcomeMailTemplete($welcomeAgencyCode, $userData, $languageCode);

                        if ($template['status'] != 'false') {
                            EmailService::sendEmail(
                                $customer->email,
                                $template['data']['subject'],
                                $template['data']['mailData'],
                                [],
                                $siteName
                            );
                        }
                    }
                    return $this->sendResponse($responseData, 'Your account has been created successfully. Please hold on while we set things up for you.', true);
                }
            }
            return $this->sendResponse(null, 'Invalid request type', false);
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    public function signUpCustomerOld(Request $request)
    {
        try {
            //check validation for either mobile or email using extends method
            Validator::extend('check_type', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition]; // Get the value of the condition field

                if ($conditionValue === 'mobile') {
                    // Check if the value is a valid mobile number
                    return preg_match('/^[0-9]+$/', $value);
                } elseif ($conditionValue === 'email') {
                    // Check if the value is a valid email address
                    return filter_var($value, FILTER_VALIDATE_EMAIL);
                }

                return false; // Invalid condition value
            }, 'The customer :conditionValue is in-valid');
            Validator::replacer('check_type', function ($message, $attribute, $rule, $parameters) use ($request) {
                $signupWith = $request->input('signup_with'); // Access signup_with from the request
                return str_replace(':conditionValue', $signupWith, $message);
            });
            //check validation for either mobile or email already exist or not
            Validator::extend('check_if_exist', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition];
                $conditionForISD = $parameters[1]; // Get the condition field name
                $conditionISDValue = $validator->getData()[$conditionForISD];

                if ($conditionValue == 'mobile') {
                    // Check if the value is a valid mobile number
                    $query = Customer::where('mobile', $conditionISDValue . ' ' . $value)
                        ->whereNotIn('status', ['terminated', 'deleted'])->get();
                    return !$query->count();
                } elseif ($conditionValue == 'email') {
                    // Check if the value is a valid email address
                    $query = Customer::where('email', $value)
                        ->whereNotIn('status', ['terminated', 'deleted'])->get();
                    return !$query->count();
                }

                return false; // Invalid condition value
            }, 'The customer :conditionValue already exist');

            //replacer validation message for either mobile or email
            Validator::replacer('check_if_exist', function ($message, $attribute, $rule, $parameters) use ($request) {
                $signupWith = $request->input('signup_with'); // Access signup_with from the request
                return str_replace(':conditionValue', $signupWith, $message);
            });

            //check mobile length must be as per mentioned in admin panel country module
            Validator::extend('check_mobile_length', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition];
                $conditionForISD = $parameters[1]; // Get the condition field name
                $conditionISDValue = $validator->getData()[$conditionForISD];

                if ($conditionValue == 'mobile') {
                    // Check if the value is a valid mobile number
                    $query = Country::where('isd_code', $conditionISDValue)->first();
                    if (!empty($query)) {

                        $mobileLength = $query['max_mobile_number_length'];
                        return (strlen($value) == $mobileLength);
                    } else {
                        return true;
                    }
                } else {
                    return true;
                }

                return false; // Invalid condition value
            }, 'The customer mobile number must be contain :mobileLength digits');

            //replacer validation message for mobile number length
            Validator::replacer('check_mobile_length', function ($message, $attribute, $rule, $parameters) use ($request) {
                $isdCode = $request->input('isd_code'); // Access signup_with from the request
                $query = Country::where('isd_code', $isdCode)->first();
                return str_replace(':mobileLength', $query['max_mobile_number_length'], $message);
            });


            //isd code validation
            Validator::extend('valid_isd_code', function ($attribute, $value, $parameters, $validator) {
                // Implement the validation logic for ISD code
                return preg_match('/^\+\d+([- ]\d+)?$/', $value);
            }, "The :attribute must start with + followed by numbers, and may be followed by - or space and more numbers.");


            $validator = Validator::make($request->all(), [
                'language_code' => 'required|in:ar,en',
                'signup_with' => 'required|in:email,mobile',
                'isd_code' => 'nullable|required_if:signup_with,mobile|valid_isd_code',
                'customer_mobile_or_email' => 'required|check_type:signup_with|check_if_exist:signup_with,isd_code|check_mobile_length:signup_with,isd_code',
                'password' => 'required',

            ]);
            // echo "hello";die;
            if ($validator->fails()) {
                $firstError = $validator->errors()->first();
                return $this->sendError('Validation failed', $validator->errors(), 200);
            }
            // if ($validator->fails()) {
            //     return response()->json(['status' => false, 'message' => $validator->errors(), 'data' => [$validator->errors()]], 200);
            // }
            $data['key'] = 'password';
            $data['password'] = $request->password;
            $responseData = resetPassword($data);

            if ($responseData['valid'] != '1') {
                return $this->sendError('Something went wrong', [$responseData], 422);
            }

            $requestData = $request->all();
            $customerData = [
                'email' => ($request['signup_with'] == 'email') ? $request['customer_mobile_or_email'] : null,
                'mobile' => ($request['signup_with'] == 'mobile') ? $requestData['isd_code'] . ' ' . $request['customer_mobile_or_email'] : "",
                'password' => Hash::make($requestData['password'])
            ];
            //insert data into customer table
            $customer = Customer::create($customerData);

            if ($customer) {

                $isMail = Setting::select('value')->where('config_key', '=', 'mail|smtp|server')->first();
                if (isset($isMail) && $isMail->value == '0') {

                    return $this->sendResponse([$customer], 'Customer SignUp Saved Successfully');
                } else {
                    if ($customer['email'] != "") {
                        $language_code = Setting::where('config_key', 'general|site|defaultLanguageCode')->get('value')[0]['value'];
                        //send welcome agency mail to primary user
                        $welcomeAgencyCode = 'WELCOME_AGENCY';
                        $agencyLogo = Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'] ?? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'];

                        $siteName = count(Setting::where('config_key', 'general|basic|siteName')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'] : "Amar Infotech";

                        $user = [
                            'agency_logo' => $agencyLogo,
                            'customer_name' => 'customer',
                            'agency_name' => $siteName
                        ];

                        $getWelcomeAgencyTemplateData = EmailService::customerWelcomeMailTemplete($welcomeAgencyCode, $user, $request->language_code);

                        if ($getWelcomeAgencyTemplateData['status'] == 'false') {
                            return back()->with('error', $getWelcomeAgencyTemplateData['error']);
                        } else {
                            $welcomeMailsubject = $getWelcomeAgencyTemplateData['data']['subject'];
                            $welcomeMailData = $getWelcomeAgencyTemplateData['data']['mailData'];
                            $welcomeAgencyToEmail = $customer['email'];
                            $files = [];

                            // set data in sendEmail function
                            $data = EmailService::sendEmail($welcomeAgencyToEmail, $welcomeMailsubject, $welcomeMailData, $files, $siteName);
                        }


                        return $this->sendResponse([$customer], 'Customer signUp saved successfully, verification link send to registered e-mail address');
                    }

                    return $this->sendResponse([$customer], 'Customer signUp saved successfully');
                }
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Get(
     *   path="/v1/customer/get-customer-profile",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Get Customer Profile",
     *   description="get Customer Profile",
     *   operationId="get-customer-profile",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",

     *           @OA\Property(property="language_code",default="en", type="string")
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * get driver status
     *
     * @return \Illuminate\Http\Response
     */
    public function customerProfile(Request $request)
    {
        try {

            $filter = array(
                'id' => Auth::id()
            );
            $response = Customer::getCustomers($filter);

            $customer_address = CustomerAddresses::where('customer_id', $request->id)->get();
            $userDetail = $response['data'];

            if ($customer_address->isNotEmpty()) {

                if (is_object($userDetail)) {
                    $userDetail = $userDetail->toArray();
                }


                $userDetail = array_merge(
                    $userDetail,
                    $customer_address->toArray()
                );
            }







            $activityLog['request'] =  $request->all();
            $activityLog['request_url'] =  $request->url();
            $activityLog['response'] =  $response;
            ActiveLog::createActiveLog($activityLog);
            if ($response['status'] == 1 && !empty($response['data'])) {
                return $this->sendResponse($userDetail, 'Customer Fetched Successfully');
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }
    /**
     * @OA\Post(
     *   path="/v1/customer/update-customer",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Update Customer",
     *   description="Send request for Update Customer",
     *   operationId="updateCustomer",
     *   @OA\RequestBody(
     *     required=true,
     *     description="Update Customer",
     *     @OA\MediaType(
     *       mediaType="multipart/form-data",
     *       @OA\Schema(
     *           required={"title","first_name","last_name","isd_code","mobile","email","date_of_birth","gender","marital_status","address1","country_id","state_id","city_code","pincode"},
     *           @OA\Property(property="language_code", type="string",description="pass language_code either 'en' or 'ar'", default="en" ),
     *           @OA\Property(property="title", type="string",description="pass title either 'mr','mrs' or 'miss'", default="" ),
     *           @OA\Property(property="first_name", type="string",description="need to pass a first name", default="" ),
     *           @OA\Property(property="last_name", type="string",description="need to pass last name", default="" ),
     *           @OA\Property(property="isd_code", type="string",description="need to pass isd code with format Ex: +91 or +355 1 or +355-1", default="" ),
     *           @OA\Property(property="mobile", type="string",description="need to pass mobile number", default="" ),
     *           @OA\Property(property="email", type="string",description="need to pass email address", default="" ),
     *           @OA\Property(property="date_of_birth", type="string",description="need to pass date of birth in Y-m-d format", default="" ),
     *           @OA\Property(property="gender", type="string",description="need to pass gender either 'male' or 'female'", default="" ),
     *           @OA\Property(property="marital_status", type="string",description="need to pass marital status one of from (married, single, other)", default="" ),
     *           @OA\Property(property="marriage_aniversary_date", type="string",description="need to pass marriage aniversary date with format Y-m-d", default="" ),
     *           @OA\Property(property="address1", type="string",description="need to pass address1", default="" ),
     *           @OA\Property(property="address2", type="string",description="need to pass address2", default="" ),
     *           @OA\Property(property="country_id", type="string",description="need to pass country id that fetch by call API 'get-country'", default="" ),
     *           @OA\Property(property="state_id", type="string",description="need to pass state id that fetch by call API 'get-states'", default="" ),
     *           @OA\Property(property="city_code", type="string",description="need to pass iso_code that fetch by call API 'get-city'", default="" ),
     *           @OA\Property(property="pincode", type="string",description="need to pass pincode", default="" ),
     *           @OA\Property(property="profile_photo", type="string", format="binary",description="select profile photo image *ensure that you are uploading an image is 2MB or less and one of the following types: JPG,JPEG, or PNG"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function updateCustomer(Request $request)
    {

        $return = array(
            'status' => 0,
            'data' => [],
            'message' => 'Something went wrong'
        );

        $customer_id = Auth::user()->id;
        Validator::extend('check_if_exist', function ($attribute, $value, $parameters, $validator) use ($customer_id) {
            $condition = $parameters[0]; // Get the condition field name
            $conditionValue = $validator->getData()[$condition];

            // Check if the value is a valid mobile number
            $query = Customer::where('mobile', $conditionValue . ' ' . $value)->where('id', '!=', $customer_id)
                ->where('status', '!=', CustomerStatus::Terminated->value)->get();
            return !$query->count();


            return false; // Invalid condition value
        }, 'The customer mobile already exist');

        try {
            // $validator = Validator::make($request->all(), [
            //     'language_code' => 'nullable|in:en,ar',
            //     'title' => 'required|in:mr,mrs,miss',
            //     'first_name' => 'required',
            //     'last_name' => 'required',
            //     'isd_code' => 'required',
            //     'mobile' => 'required|check_if_exist:isd_code',
            //     'email' => 'required|unique:customers,email,' . Auth::user()->id,
            //     'date_of_birth' => 'required|date_format:Y-m-d|before:today',
            //     'gender' => 'required|in:male,female',
            //     'marital_status' => 'required|in:single,married,other',
            //     'marriage_aniversary_date' => [
            //         'required_if:marital_status,married',
            //         function ($attribute, $value, $fail) use ($request) {
            //             if ($request->marital_status == 'married') {
            //                 if (!strtotime($value) || date('Y-m-d', strtotime($value)) !== $value) {
            //                     $fail("The $attribute does not match the format Y-m-d for married individuals.");
            //                 } elseif (strtotime($value) >= strtotime('today')) {
            //                     $fail("The $attribute must be a date before today for married individuals.");
            //                 }
            //             }
            //         },
            //     ],
            //     'address1' => 'required',
            //     'address2' => 'nullable',
            //     'country_id' => 'required',
            //     'state_id' => 'required',
            //     'city_code' => 'nullable',
            //     'pincode' => 'nullable',
            //     'profile_photo' => 'nullable|mimes:jpeg,jpg,png|max:2048',

            // ]);
            $validator = Validator::make($request->all(), [
                'language_code' => 'nullable|in:en,ar',
                'title' => 'nullable|in:mr,mrs,miss',
                'first_name' => 'nullable',
                'last_name' => 'nullable',
                'isd_code' => 'nullable',
                'mobile' => 'nullable|check_if_exist:isd_code',
                'email' => 'nullable|unique:customers,email,' . Auth::user()->id,
                'date_of_birth' => 'nullable|date_format:Y-m-d|before:today',
                'gender' => 'nullable|in:male,female',
                'marital_status' => 'nullable|in:single,married,other',
                'marriage_aniversary_date' => 'nullable',
                'address1' => 'nullable',
                'address2' => 'nullable',
                'country_id' => 'nullable',
                'state_id' => 'nullable',
                'city_code' => 'nullable',
                'pincode' => 'nullable',
                'profile_photo' => 'nullable|mimes:jpeg,jpg,png|max:2048',
            ]);
            if ($validator->fails()) {
                return $this->sendError('Validation failed', $validator->errors(), 200);
            }
            if (request()->input('mobile') != '') {
                $numlength = strlen((string)request()->input('mobile'));
                $validMobileLength = Country::where('isd_code', request()->input('isd_code'))->where('status', 1)->value('max_mobile_number_length');
                if ($numlength != $validMobileLength) {
                    $success = [];
                    return $this->sendError('Mobile number must be at least ' . $validMobileLength . ' digits', $success, 400);
                }
            }
            if (request()->input('country_id') != '') {
                //validation foe country_id
                $isoCodecountry = Country::where('id', request()->input('country_id'))->where('status', CustomerStatus::Active->value)->value('id');
                if (request()->input('country_id') != $isoCodecountry) {
                    $success = [];
                    return $this->sendError('Please Enter Valid country(id)', $success, 400);
                }
            }
            if (request()->input('state_id') != '') {
                //validation foe state_id
                $isoCode = State::where('id', request()->input('state_id'))->where('status', CustomerStatus::Active->value)->value('id');
                if (request()->input('state_id') != $isoCode) {
                    $success = [];
                    return $this->sendError('Please Enter Valid state(id)', $success, 400);
                }
            }
            if (request()->input('city_code') != '') {
                //validation foe city_code
                $isoCode = City::where('iso_code', request()->input('city_code'))->where('status', CustomerStatus::Active->value)->value('iso_code');
                if (request()->input('city_code') != $isoCode) {
                    $success = [];
                    return $this->sendError('Please Enter Valid City(iso_code)', $success, 400);
                }
            }
            $customerData = $request->all();
            $requestData = array(
                'customer_id' => Auth::id(),
                'title' => ucwords($customerData['title']),
                'first_name' => ucwords($customerData['first_name']),
                'last_name' => ucwords($customerData['last_name']),
                'mobile' => $customerData['mobile'],
                'isd_code' => $customerData['isd_code'],
                'email' => $customerData['email'],
                'status' => CustomerStatus::Active->value,
                'date_of_birth' => date('Y-m-d', strtotime($customerData['date_of_birth'])),
                'gender' => $customerData['gender'],
                'marital_status' => $customerData['marital_status'],
                'marriage_aniversary_date' => $customerData['marriage_aniversary_date'],
                'profile_photo' => $customerData['profile_photo'],
                'address1' => $customerData['address1'],
                'address2' => $customerData['address2'],
                'country' => $customerData['country_id'],
                'state_code' => $customerData['state_id'],
                'city_code' => $customerData['city_code'],
                'pincode' => $customerData['pincode'],
                'alternative_email' => $customerData['alternative_email'],
                'user_currency' => $customerData['user_currency'],
                'user_language' => $customerData['user_language'],
            );

            $response = Customer::updateCustomer($requestData);
            $activityLog['request'] =  $request->all();
            $activityLog['request_url'] =  $request->url();
            $activityLog['response'] =  $response;
            ActiveLog::createActiveLog($activityLog);
            if ($response) {
                $success = true;
                return $this->sendResponse($response, 'Customer Updated Successfully!', $success);
            } else {
                $success = [];
                return $this->sendError('Customer Not Saved', $success, 200);
            }
        } catch (\Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    public function updateCustomerImage(Request $request)
    {
        try {
            $customerData = $request->all();
            $requestData = [
                'customer_id' => Auth::id(),
                'profile_photo' => $customerData['profile_photo'] ?? null
            ];
            // return $requestData;
            $response = Customer::updateCustomerImage($requestData);

            // Log
            $activityLog['request'] = $request->all();
            $activityLog['request_url'] = $request->url();
            $activityLog['response'] = $response;
            ActiveLog::createActiveLog($activityLog);

            if (!empty($response['status'])) {
                return $this->sendResponse(
                    $response['data'],
                    $response['message'],
                    true
                );
            } else {
                return $this->sendError(
                    $response['message'] ?? 'Customer image not updated',
                    [],
                    200
                );
            }
        } catch (\Exception $e) {
            return $this->sendError(
                'Something went wrong',
                ['error' => $e->getMessage()],
                500
            );
        }
    }


    /**
     * @OA\Post(
     *   path="/v1/customer/verification-otp/send",
     *   tags={"Customer"},
     *   summary="OTP Send On Mobile or Email",
     *   description="Pass type value either mobile or email<br>Pass ISD Code Ex: +358, it's required when type is mobile<br>Pass mobile_or_email value Ex: 9999999999 or example@gmail.com</br>Pass Language code Ex: en or ar",
     *   operationId="otpVerificationSend",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",
                 required={"type","isd_code","mobile_or_email","language_code"},

     *           @OA\Property(property="type", type="string"),
     *           @OA\Property(property="isd_code", type="string"),
     *           @OA\Property(property="mobile_or_email", type="string"),
     *           @OA\Property(property="language_code",default="en", type="string"),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Login Send
     *
     * @return \Illuminate\Http\Response
     */
    public function otpVerificationSend(Request $request)
    {
        try {
            //check validation for either mobile or email using extends method
            Validator::extend('check_type', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition]; // Get the value of the condition field

                if ($conditionValue === 'mobile') {
                    // Check if the value is a valid mobile number
                    return preg_match('/^[0-9]+$/', $value);
                } elseif ($conditionValue === 'email') {
                    // Check if the value is a valid email address
                    return filter_var($value, FILTER_VALIDATE_EMAIL);
                }

                return false; // Invalid condition value
            }, 'The customer :conditionValue is in-valid');
            Validator::replacer('check_type', function ($message, $attribute, $rule, $parameters) use ($request) {
                $signupWith = $request->input('type'); // Access signup_with from the request
                return str_replace(':conditionValue', $signupWith, $message);
            });


            //check mobile length must be as per mentioned in admin panel country module
            Validator::extend('check_mobile_length', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition];
                $conditionForISD = $parameters[1]; // Get the condition field name
                $conditionISDValue = $validator->getData()[$conditionForISD];

                if ($conditionValue == 'mobile') {
                    // Check if the value is a valid mobile number
                    $query = Country::where('isd_code', $conditionISDValue)->first();
                    if (!empty($query)) {

                        $mobileLength = $query['max_mobile_number_length'];
                        return (strlen($value) == $mobileLength);
                    } else {
                        return true;
                    }
                } else {
                    return true;
                }

                return false; // Invalid condition value
            }, 'The customer mobile number must be contain :mobileLength digits');

            //replacer validation message for mobile number length
            Validator::replacer('check_mobile_length', function ($message, $attribute, $rule, $parameters) use ($request) {
                $isdCode = $request->input('isd_code'); // Access signup_with from the request
                $query = Country::where('isd_code', $isdCode)->first();
                return str_replace(':mobileLength', $query['max_mobile_number_length'], $message);
            });


            //isd code validation
            Validator::extend('valid_isd_code', function ($attribute, $value, $parameters, $validator) {
                // Implement the validation logic for ISD code
                return preg_match('/^\+\d+([- ]\d+)?$/', $value);
            }, "The :attribute must start with + followed by numbers, and may be followed by - or space and more numbers.");

            $validator = Validator::make($request->all(), [
                'type' => 'required|in:email,mobile',
                'isd_code' => 'nullable|required_if:type,mobile|' . ($request->input('type') == 'mobile' ? 'valid_isd_code' : ''),
                'mobile_or_email' => 'required|check_type:type|check_mobile_length:type,isd_code',
                'language_code' => 'required|in:ar,en',

            ], [
                'type.in' => 'entered type must be either email or mobile',
                'language_code.in' => 'entered language code is wrong, it must be either ar or en'
            ]);
            $validator->sometimes('isd_code', 'valid_isd_code', function ($input) {
                return $input->type == 'mobile';
            });

            if ($validator->fails()) {
                return $this->sendError('Validation failed', $validator->errors(), 422);
            }
            if ($request['type'] != "email") {

                $getCustomerDetails = Customer::where('mobile', $request->isd_code . ' ' . $request->mobile_or_email)->first();
            } else {

                $getCustomerDetails = Customer::where('email', $request->mobile_or_email)->first();
            }


            if ($getCustomerDetails) {

                //get OTP verification setting key value 'general|otp|phoneVerification'
                $otp_verification = false;
                $otp_setting_data = \App\Models\Setting::where('config_key', 'general|otp|phoneVerification')->get()->toArray();

                if ($getCustomerDetails->status == CustomerStatus::Active->value) {
                    if ($request['type'] != 'email') {
                        $otp = rand(100000, 999999);

                        $currentDate = date('Y-m-d H:i:s');
                        $datee = strtotime($currentDate);
                        $otp_expire_minute = 5;
                        $date1 = strtotime("+" . $otp_expire_minute . "minute", $datee);
                        $otp_expire = date('Y-m-d H:i:s', $date1);

                        $temp = array(
                            "otp" => $otp,
                            "mobile" => $request->isd_code . ' ' . $request->mobile_or_email,
                            "expired" => $otp_expire,
                        );

                        $mobile_number = ['mobile' => $request->isd_code . ' ' . $request->mobile_or_email];
                        $message = "$otp is your one time password to proceed on Web Site. It is valid for $otp_expire_minute minutes";

                        if (!empty($otp_setting_data)) {
                            if ($otp_setting_data[0]['value'] == "on") {
                                $otp_verification = true;
                                //send OTP SMS code
                                $isd_mobile = $request->isd_code . $request->mobile_or_email;
                                $this->sendSms($isd_mobile, $message);
                            }
                        }

                        AdminUserOtp::updateOrCreate($mobile_number, $temp);

                        $success['otp'] = $otp;
                        $success['otp_verify'] = $otp_verification;
                        $response = array($success);

                        return $this->sendResponse($response, "OTP send successfully to your registered Mobile");
                    } else {
                        if ($getCustomerDetails['first_name'] != "" && $getCustomerDetails['last_name'] != "") {
                            $customerName = $getCustomerDetails['first_name'] . ' ' . $getCustomerDetails['last_name'];
                        } else {
                            $customerName = 'Customer';
                        }
                        $email = $getCustomerDetails->email;
                        $otp = rand(100000, 999999);

                        $currentDate = date('Y-m-d H:i:s');
                        $datee = strtotime($currentDate);
                        $otp_expire_minute = 5;
                        $date1 = strtotime("+" . $otp_expire_minute . "minute", $datee);
                        $otp_expire = date('Y-m-d H:i:s', $date1);

                        $temp = array(
                            "otp" => $otp,
                            "mobile" => $request->mobile_or_email,
                            "expired" => $otp_expire,
                        );

                        $emailAddress = ['mobile' => $request->mobile_or_email];
                        $code = 'SEND_OTP';
                        $siteName = count(Setting::where('config_key', 'general|basic|siteName')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'] : "Amar Infotech";
                        $otp = $otp;
                        $otp_expire_minute = $otp_expire_minute;
                        $agencyLogo = Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'] ?? Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'];
                        $agencyName = count(Setting::where('config_key', 'general|basic|siteName')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'] : "Rehlte";
                        $customerMailData = [
                            'customer_name' => $customerName,
                            'otp' => $otp,
                            'otp_expire_minute' => $otp_expire_minute,
                            'site_name' => $siteName,
                            'agencyLogo' => $agencyLogo,
                            'agencyName' => $agencyName
                        ];

                        $getCustomerSendOTPTemplate = EmailService::customerSendOTPTemplete($code, $customerMailData, $request->language_code);

                        if ($getCustomerSendOTPTemplate['status'] == 'false') {
                            return back()->with('error', $getCustomerSendOTPTemplate['error']);
                        } else {
                            $sendOTPSubject = $getCustomerSendOTPTemplate['data']['subject'];
                            $sendOTPMailData = $getCustomerSendOTPTemplate['data']['mailData'];
                            $customerEmail = $email;
                            $files = [];

                            // set data in sendEmail function
                            $data = EmailService::sendEmail($customerEmail, $sendOTPSubject, $sendOTPMailData, $files, $siteName);
                        }
                        if (!empty($otp_setting_data)) {
                            if ($otp_setting_data[0]['value'] == "on") {
                                $otp_verification = true;
                                //send OTP SMS code

                                $this->sendEmail($email, $sendOTPSubject, $sendOTPMailData);
                            }
                        }

                        AdminUserOtp::updateOrCreate($emailAddress, $temp);

                        $success['otp'] = $otp;
                        $success['otp_verify'] = $otp_verification;
                        $response = array($success);
                        if ($request['type'] == 'mobile') {
                            return $this->sendResponse($response, "OTP send successfully to your registered Mobile Number");
                        } else {
                            return $this->sendResponse($response, "OTP send successfully to your registered Email Address");
                        }
                    }
                } else if ($getCustomerDetails->status == CustomerStatus::Deleted->value) {
                    return $this->sendError('You no longer access your acount. Please create a new one to continue', [], 200);
                } else {
                    return $this->sendError('Customer is Inactive or terminated', [], 200);
                }
            } else {
                $success = [];
                return $this->sendError('Customer not found', $success, 200);
            }


            // if (!empty($user)) {
            //     return $this->sendResponse([], 'User Find Successfully, please check otp send api');
            // } else {
            //     $success = [];
            //     return $this->sendError('User Not Found.', $success, 200);
            // }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/customer/verification-otp/verify",
     *   tags={"Customer"},
     *   summary="OTP Verify Using Mobile or Email",
     *   description="pass type either mobile or email<be>pass ISD Code Ex: +358, it's required when type value is mobile<br>Pass mobile_or_email value Ex: 8989898989 or example@gmail.com<br>enter otp what ever you got on mobile or email",
     *   operationId="OtpVerify",
     *   @OA\RequestBody(
     *     required=true,
     *     description="The request body for otp verification",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"language_code","type","isd_code","mobile_or_email","otp"},
     *           @OA\Property(property="language_code",default="en", type="string"),
     *             @OA\Property(property="type", type="string", title="type",description="enter type either mobile or email"),
     *             @OA\Property(property="isd_code", type="string", title="isd_code",description="enter exisiting isd code"),
     *             @OA\Property(property="mobile_or_email", type="string", title="mobile",description="enter exisiting mobile or email number on which you got otp"),
     *             @OA\Property(property="otp", type="string",title="otp", description="enter a otp which you got on registered mobile number or email address"),
     *      )
     *    ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * OTP Verify
     *
     * @return \Illuminate\Http\Response
     */
    public function OtpVerify(Request $request)
    {


        try {
            //check validation for either mobile or email using extends method
            Validator::extend('check_type', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition]; // Get the value of the condition field

                if ($conditionValue === 'mobile') {
                    // Check if the value is a valid mobile number
                    return preg_match('/^[0-9]+$/', $value);
                } elseif ($conditionValue === 'email') {
                    // Check if the value is a valid email address
                    return filter_var($value, FILTER_VALIDATE_EMAIL);
                }

                return false; // Invalid condition value
            }, 'The customer :conditionValue is in-valid');
            Validator::replacer('check_type', function ($message, $attribute, $rule, $parameters) use ($request) {
                $signupWith = $request->input('type'); // Access signup_with from the request
                return str_replace(':conditionValue', $signupWith, $message);
            });


            //check mobile length must be as per mentioned in admin panel country module
            Validator::extend('check_mobile_length', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition];
                $conditionForISD = $parameters[1]; // Get the condition field name
                $conditionISDValue = $validator->getData()[$conditionForISD];

                if ($conditionValue == 'mobile') {
                    // Check if the value is a valid mobile number
                    $query = Country::where('isd_code', $conditionISDValue)->first();
                    if (!empty($query)) {

                        $mobileLength = $query['max_mobile_number_length'];
                        return (strlen($value) == $mobileLength);
                    } else {
                        return true;
                    }
                } else {
                    return true;
                }

                return false; // Invalid condition value
            }, 'The customer mobile number must be contain :mobileLength digits');

            //replacer validation message for mobile number length
            Validator::replacer('check_mobile_length', function ($message, $attribute, $rule, $parameters) use ($request) {
                $isdCode = $request->input('isd_code'); // Access signup_with from the request
                $query = Country::where('isd_code', $isdCode)->first();
                return str_replace(':mobileLength', $query['max_mobile_number_length'], $message);
            });


            //isd code validation
            Validator::extend('valid_isd_code', function ($attribute, $value, $parameters, $validator) {
                // Implement the validation logic for ISD code
                return preg_match('/^\+\d+([- ]\d+)?$/', $value);
            }, "The :attribute must start with + followed by numbers, and may be followed by - or space and more numbers.");

            $validator = Validator::make($request->all(), [
                'language_code' => 'required|in:ar,en',
                'type' => 'required|in:email,mobile',
                'isd_code' => 'nullable|required_if:type,mobile|' . ($request->input('type') == 'mobile' ? 'valid_isd_code' : ''),
                'mobile_or_email' => 'required|check_type:type|check_mobile_length:type,isd_code',
                'otp' => 'required|numeric',

            ], [
                'type.in' => 'entered type must be either email or mobile',

            ]);
            $validator->sometimes('isd_code', 'valid_isd_code', function ($input) {
                return $input->type == 'mobile';
            });

            if ($validator->fails()) {
                return $this->sendError('Validation failed', $validator->errors(), 422);
            }
            if ($request['type'] != 'email') {

                $checker = Customer::select('*', 'id as user_id')->where('mobile', '=', $request->isd_code . ' ' . $request->mobile_or_email)->where('status', CustomerStatus::Active->value)->where('status', '!=', CustomerStatus::Deleted->value)->first();
            } else {
                $checker = Customer::select('*', 'id as user_id')->where('email', '=',  $request->mobile_or_email)->where('status', CustomerStatus::Deleted->value)->where('status', '!=', CustomerStatus::Active->value)->first();
            }
            if (!empty($checker)) {
                if ($request['type'] != 'email') {

                    $userOtp = AdminUserOtp::where('mobile', '=', $request->isd_code . ' ' . $request->mobile_or_email)
                        ->where('otp', '=', $request->otp)
                        ->first();
                } else {
                    $userOtp = AdminUserOtp::Where('mobile', $request->mobile_or_email)
                        ->where('otp', '=', $request->otp)
                        ->first();
                }

                if (!empty($userOtp)) {

                    $exipreDate = $userOtp['expired'];
                    $currentDate = date('Y-m-d H:i:s');

                    if (strtotime($currentDate) < strtotime($exipreDate)) {

                        AdminUserOtp::where('mobile', '=', $request->isd_code . ' ' . $request->mobile_or_email)->orWhere('mobile', $request->mobile_or_email)
                            ->where('otp', '=', $request->otp)
                            ->delete();


                        Auth::guard('appuser')->loginUsingId($checker->id);
                        $userData = Auth::guard('appuser')->user();
                        $success = $checker;
                        $success['token'] =  $userData->createToken('AuthToken')->accessToken;
                        $updateCustomer = Customer::query();
                        if ($request['type'] != 'email') {

                            $updateCustomer->where('mobile', $request->isd_code . ' ' . $request->mobile_or_email)->update([
                                'is_mobile_verified' => '1'
                            ]);
                        } else {

                            $updateCustomer->where('email',  $request->mobile_or_email)->update([
                                'is_email_verified' => '1'
                            ]);
                        }
                        return $this->sendResponse($success, 'OTP Verified Successfully');
                    } else {

                        $success = [];
                        return $this->sendError('OTP expired!', $success, 200);
                    }
                } else {
                    $success = [];
                    return $this->sendError($request['type'] . ' and otp does not match', $success, 200);
                }
            } else {
                $success = [];
                return $this->sendError('Customer not found', $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }


    /**
     * @OA\Post(
     *   path="/v1/customer/resend-activation-mail",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Resend activation mail to customer",
     *   description="send activation mail to customer<br>Pass language code value either en or ar",
     *   operationId="resend-mail",
     *   @OA\RequestBody(
     *     required=true,
     *     description="The request body for resend activation mail",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"language_code"},
     *           @OA\Property(property="language_code",default="en", type="string"),
     *      )
     *    ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function customerResendMail(Request $request)
    {
        try {

            if (AUth::user()) {
                $validator = Validator::make($request->all(), [
                    'language_code' => 'required|in:ar,en',
                ]);
                if ($validator->fails()) {
                    return $this->sendError('Validation failed', $validator->errors(), 422);
                }
                $customer = Customer::where('id', Auth::id())->first();

                $isMail = Setting::select('value')->where('config_key', '=', 'mail|smtp|server')->first();
                if (isset($isMail) && $isMail->value == '0') {
                    return $this->sendResponse([$customer], 'Unable to send mail now, please check your SMTP setting');
                } else {
                    if ($customer['email'] != "") {

                        $token = Str::random(60);
                        $updateCustomerToken = DB::table('customer_activation_tokens')
                            ->where(['email' => $customer['email']])
                            ->first();
                        if (!$updateCustomerToken) {
                            \DB::table('customer_activation_tokens')->insert(
                                ['email' => $customer['email'], 'token' => $token, 'created_at' => Carbon::now()]
                            );
                        } else {
                            DB::table('customer_activation_tokens')->where(['email' => $customer['email']])->update(
                                ['token' => $updateCustomerToken->token]
                            );
                            $token = $updateCustomerToken->token;
                        }
                        $code = 'CUSTOMER_SIGN_UP';
                        if ($customer['agency_id'] != 0) {
                            $agencyLogo = Agency::where('id', $customer['agency_id'])->value('logo');
                            $agencyName = Agency::where('id', $customer['agency_id'])->value('full_name');
                        } else {
                            $agencyLogo = Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'] ?? Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'];
                            $agencyName = count(Setting::where('config_key', 'general|basic|siteName')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'] : "Rehlte";
                        }



                        $b2cUrl  = Setting::select('value')->where('config_key', 'general|b2cUrl')->first();
                        $customerEmail = ucwords($customer['email']);

                        $link = $b2cUrl->value . 'email-verification/' . $token;

                        $data = array(
                            'first_name' => "roshan",
                            'site_name' => $agencyName,
                            'agency_name' => $agencyName,
                            'agency_logo' => $agencyLogo,
                            'email' => $customerEmail,
                            'activation_link' => $link
                        );


                        $getCustomerSignUp = $this->customerSignUp($code, $data, $request->language_code);

                        $mailData = $getCustomerSignUp['data']['mailData'];
                        $subject = $getCustomerSignUp['data']['subject'];
                        $mailData = $getCustomerSignUp['data']['mailData'];
                        $toEmail = $customer['email'];
                        $files = [];

                        // set data in sendEmail function
                        $data = $this->sendEmail($toEmail, $subject, $mailData, $files, $agencyName, $code);
                    }
                    return $this->sendResponse([$customer], 'Varification mail send successfully');
                }
            } else {
                $success = [];
                return $this->sendError('Customer not found', $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }
    /**
     * @OA\Post(
     *   path="/v1/customer/forgot-password-otp/send",
     *   tags={"Customer"},
     *   summary="Send request for forgot password",
     *   description="Pass type value either mobile or email<br>if type mobile then isd_code required else not<br>Pass an Mobile Number and check mobile is exists or not<br>Or<br>
                    Pass email address to get otp<br>Pass language code either ar for arabic  or en for english",
     *   operationId="forgot-password",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",
      required={"type","isd_code","mobile_or_email","language_code"},
     *           @OA\Property(property="type", type="string",description="pass 'mobile' or 'email'"),
     *           @OA\Property(property="isd_code", type="string",description="pass isd code if you want to get otp on mobile"),
     *           @OA\Property(property="mobile_or_email", type="string"),
     *           @OA\Property(property="language_code",default="en", type="string"),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Check Users Exists api
     *
     * @return \Illuminate\Http\Response
     */

    public function generateRandomPassword($length = 8)
    {
        if ($length < 4) {
            throw new \Exception("Password length must be at least 4 characters.");
        }

        $lowercase = 'abcdefghijklmnopqrstuvwxyz';
        $uppercase = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $numbers = '0123456789';
        $specials = '!@';

        $password = '';
        $password .= $lowercase[rand(0, strlen($lowercase) - 1)];
        $password .= $uppercase[rand(0, strlen($uppercase) - 1)];
        $password .= $numbers[rand(0, strlen($numbers) - 1)];
        $password .= $specials[rand(0, strlen($specials) - 1)];

        $all = $lowercase . $uppercase . $numbers . $specials;
        for ($i = 4; $i < $length; $i++) {
            $password .= $all[rand(0, strlen($all) - 1)];
        }

        return str_shuffle($password);
    }

    public static function checkInputType($input)
    {
        $input = trim($input);

        // Check for valid email
        if (filter_var($input, FILTER_VALIDATE_EMAIL)) {
            return "email";
        }


        $normalized = preg_replace('/[^0-9+]/', '', $input);

        if (preg_match('/^\+?[0-9]{9,15}$/', $normalized)) {
            return "mobile";
        }

        return "Invalid Input";
    }


    public function forgotPassword(Request $request)
    {
        try {
            //check validation for either mobile or email using extends method
            Validator::extend('check_type', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition]; // Get the value of the condition field

                if ($conditionValue === 'mobile') {
                    // Check if the value is a valid mobile number
                    return preg_match('/^[0-9]+$/', $value);
                } elseif ($conditionValue === 'email') {
                    // Check if the value is a valid email address
                    return filter_var($value, FILTER_VALIDATE_EMAIL);
                }

                return false; // Invalid condition value
            }, 'The customer :conditionValue is in-valid');
            Validator::replacer('check_type', function ($message, $attribute, $rule, $parameters) use ($request) {
                $signupWith = $request->input('type'); // Access signup_with from the request
                return str_replace(':conditionValue', $signupWith, $message);
            });


            //check mobile length must be as per mentioned in admin panel country module
            // Validator::extend('check_mobile_length', function ($attribute, $value, $parameters, $validator) {
            //     $condition = $parameters[0]; // Get the condition field name
            //     $conditionValue = $validator->getData()[$condition];
            //     $conditionForISD = $parameters[1]; // Get the condition field name
            //     $conditionISDValue = $validator->getData()[$conditionForISD];

            //     if ($conditionValue == 'mobile') {
            //         // Check if the value is a valid mobile number
            //         $query = Country::where('isd_code', $conditionISDValue)->first();
            //         if (!empty($query)) {

            //             $mobileLength = $query['max_mobile_number_length'];
            //             return (strlen($value) == $mobileLength);
            //         } else {
            //             return true;
            //         }
            //     } else {
            //         return true;
            //     }

            //     return false; // Invalid condition value
            // }, 'The customer mobile number must be contain :mobileLength digits');

            // //replacer validation message for mobile number length
            // Validator::replacer('check_mobile_length', function ($message, $attribute, $rule, $parameters) use ($request) {
            //     $isdCode = $request->input('isd_code'); // Access signup_with from the request
            //     $query = Country::where('isd_code', $isdCode)->first();
            //     return str_replace(':mobileLength', $query['max_mobile_number_length'], $message);
            // });


            //isd code validation
            // Validator::extend('valid_isd_code', function ($attribute, $value, $parameters, $validator) {
            //     // Implement the validation logic for ISD code
            //     return preg_match('/^\+\d+([- ]\d+)?$/', $value);
            // }, "The :attribute must start with + followed by numbers, and may be followed by - or space and more numbers.");

            $validator = Validator::make($request->all(), [
                'type' => 'required|in:email,mobile',
                'isd_code' => 'nullable',
                'mobile_or_email' => 'required',
                'language_code' => 'required|in:ar,en',

            ], [
                'type.in' => 'entered type must be either email or mobile',
                'language_code.in' => 'entered language code is wrong, it must be either ar or en'
            ]);
            // $validator->sometimes('isd_code', 'valid_isd_code', function ($input) {
            //     return $input->type == 'mobile';
            // });

            if ($validator->fails()) {
                return $this->sendError('Validation failed', $validator->errors(), 422);
            }
            $inputType = self::checkInputType($request->mobile_or_email);
            if ($inputType == "mobile") {
                // $getCustomerDetails = Customer::where('mobile', $request->isd_code . ' ' . $request->mobile_or_email)->first();
                $getCustomerDetails = Customer::whereRaw("TRIM(SUBSTRING_INDEX(mobile, ' ', -1)) = ?", [$request->mobile_or_email])->where('status', '!=', CustomerStatus::Deleted->value)
                    ->first();
            } else {
                $getCustomerDetails = Customer::where('email', $request->mobile_or_email)->where('status', '!=', CustomerStatus::Deleted->value)->first();
            }


            if ($getCustomerDetails) {
                if ($getCustomerDetails['first_name'] != "" && $getCustomerDetails['last_name'] != "") {
                    $customerName = $getCustomerDetails['first_name'] . ' ' . $getCustomerDetails['last_name'];
                } else {
                    $customerName = 'Customer';
                }
                //get OTP verification setting key value 'general|otp|phoneVerification'
                $otp_verification = false;
                $otp_setting_data = \App\Models\Setting::where('config_key', 'general|otp|phoneVerification')->get()->toArray();

                if ($getCustomerDetails->status == CustomerStatus::Active->value) {
                    if (false) {
                        // $otp = rand(100000, 999999);
                        $otp = $this->generateRandomPassword();
                        $currentDate = date('Y-m-d H:i:s');
                        $datee = strtotime($currentDate);
                        $otp_expire_minute = 5;
                        $date1 = strtotime("+" . $otp_expire_minute . "minute", $datee);
                        $otp_expire = date('Y-m-d H:i:s', $date1);

                        $temp = array(
                            "otp" => $otp,
                            "mobile" => $request->isd_code . ' ' . $request->mobile_or_email,
                            "expired" => $otp_expire,
                        );

                        $mobile_number = ['mobile' => $request->isd_code . ' ' . $request->mobile_or_email];
                        $message = "$otp is your password to proceed on Web Site. After Login Change your password for security reasons";

                        if (!empty($otp_setting_data)) {
                            if ($otp_setting_data[0]['value'] == "on") {
                                $otp_verification = true;
                                //send OTP SMS code
                                $mobile_number = $request->isd_code . $request->mobile_or_email;
                                $this->sendSms($mobile_number, $message);
                                $getCustomerDetails->password = Hash::make($otp);
                                $getCustomerDetails->save();
                            }
                        }

                        AdminUserOtp::updateOrCreate(['mobile' => $mobile_number], $temp);


                        // $success['otp'] = $otp;
                        $success['otp_verify'] = $otp_verification;
                        $response = array($success);

                        return $this->sendResponse($response, "OTP send successfully to your registered Mobile");
                    } else {

                        $email = $getCustomerDetails->email;
                        // $otp = rand(100000, 999999);
                        $otp = $this->generateRandomPassword();

                        $currentDate = date('Y-m-d H:i:s');
                        $datee = strtotime($currentDate);
                        $otp_expire_minute = 5;
                        $date1 = strtotime("+" . $otp_expire_minute . "minute", $datee);
                        $otp_expire = date('Y-m-d H:i:s', $date1);
                        $mobile_number = ['mobile' => $request->mobile_or_email];
                        $temp = array(
                            "otp" => $otp,
                            "mobile" => $request->mobile_or_email,
                            "expired" => $otp_expire,
                        );
                        AdminUserOtp::updateOrCreate(['mobile' => $mobile_number], $temp);

                        $emailAddress = ['mobile' => $request->mobile_or_email];
                        $code = 'FORGOT_PASSWORD';
                        $siteName = count(Setting::where('config_key', 'general|basic|siteName')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'] : "Rehlte Tourism & Travel";
                        $otp = $otp;
                        $otp_expire_minute = $otp_expire_minute;
                        $agencyLogo = Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'] ?? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'];
                        $agencyName = Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'];
                        $customerMailData = [
                            'customer_name' => $customerName,
                            'agencyLogo' => $agencyLogo,
                            'agencyName' => $agencyName,
                            'otp' => $otp,
                            'otp_expire_minute' => $otp_expire_minute,
                            'site_name' => $siteName
                        ];

                        $getCustomerSendOTPTemplate = EmailService::customerSendOTPTemplete($code, $customerMailData, $request->language_code);
                        if ($getCustomerSendOTPTemplate['status'] == 'false') {
                            return back()->with('error', $getCustomerSendOTPTemplate['error']);
                        } else {
                            $sendOTPSubject = $getCustomerSendOTPTemplate['data']['subject'];
                            $sendOTPMailData = $getCustomerSendOTPTemplate['data']['mailData'];
                            $customerEmail = $email;
                            $files = [];

                            // set data in sendEmail function
                            $data = EmailService::sendEmail($customerEmail, $sendOTPSubject, $sendOTPMailData, $files, $siteName);
                            $getCustomerDetails->password = Hash::make($otp);
                            $getCustomerDetails->save();
                            $success['email'] = $customerEmail;
                            $response = array($success);
                        }

                        if ($request['type'] != 'email') {

                            return $this->sendResponse($response, "OTP send successfully to your registered Mobile Number");
                        } else {

                            return $this->sendResponse($response, "OTP send successfully to your registered Email Address");
                        }
                    }
                } else {
                    return $this->sendError('You no longer access your acount. Please create a new one to continue', [], 200);
                }
            } else {
                $success = [];
                return $this->sendError('No Account found for this Email or Mobile Number', $success, 200);
            }


            // if (!empty($user)) {
            //     return $this->sendResponse([], 'User Find Successfully, please check otp send api');
            // } else {
            //     $success = [];
            //     return $this->sendError('User Not Found.', $success, 200);
            // }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }
    /**
     * @OA\Post(
     *   path="/v1/customer/forgot-password-otp/verify",
     *   tags={"Customer"},
     *   summary="OTP Verify Using Mobile or Email",
     *   description="pass type either mobile or email<be>pass ISD Code Ex: +358, it's required when type value is mobile<br>Pass mobile_or_email value Ex: 8989898989 or example@gmail.com<br>enter otp what ever you got on mobile or email",
     *   operationId="ForgotPasswordOtpVerify",
     *   @OA\RequestBody(
     *     required=true,
     *     description="The request body for otp verification",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"language_code","type","isd_code","mobile_or_email","otp"},
     *             @OA\Property(property="language_code", type="string",default="en"),
     *             @OA\Property(property="type", type="string", title="type",description="enter type either mobile or email"),
     *             @OA\Property(property="isd_code", type="string", title="isd_code",description="enter exisiting isd code"),
     *             @OA\Property(property="mobile_or_email", type="string", title="mobile",description="enter exisiting mobile or email number on which you got otp"),
     *             @OA\Property(property="otp", type="string",title="otp", description="enter a otp which you got on registered mobile number or email address"),
     *      )
     *    ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * OTP Verify
     *
     * @return \Illuminate\Http\Response
     */
    public function forgotPasswordOtpVerify(Request $request)
    {

        try {
            //check validation for either mobile or email using extends method
            Validator::extend('check_type', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition]; // Get the value of the condition field

                if ($conditionValue === 'mobile') {
                    // Check if the value is a valid mobile number
                    return preg_match('/^[0-9]+$/', $value);
                } elseif ($conditionValue === 'email') {
                    // Check if the value is a valid email address
                    return filter_var($value, FILTER_VALIDATE_EMAIL);
                }

                return false; // Invalid condition value
            }, 'The customer :conditionValue is in-valid');
            Validator::replacer('check_type', function ($message, $attribute, $rule, $parameters) use ($request) {
                $signupWith = $request->input('type'); // Access signup_with from the request
                return str_replace(':conditionValue', $signupWith, $message);
            });


            //check mobile length must be as per mentioned in admin panel country module
            Validator::extend('check_mobile_length', function ($attribute, $value, $parameters, $validator) {
                $condition = $parameters[0]; // Get the condition field name
                $conditionValue = $validator->getData()[$condition];
                $conditionForISD = $parameters[1]; // Get the condition field name
                $conditionISDValue = $validator->getData()[$conditionForISD];

                if ($conditionValue == 'mobile') {
                    // Check if the value is a valid mobile number
                    $query = Country::where('isd_code', $conditionISDValue)->first();
                    if (!empty($query)) {

                        $mobileLength = $query['max_mobile_number_length'];
                        return (strlen($value) == $mobileLength);
                    } else {
                        return true;
                    }
                } else {
                    return true;
                }

                return false; // Invalid condition value
            }, 'The customer mobile number must be contain :mobileLength digits');

            //replacer validation message for mobile number length
            Validator::replacer('check_mobile_length', function ($message, $attribute, $rule, $parameters) use ($request) {
                $isdCode = $request->input('isd_code'); // Access signup_with from the request
                $query = Country::where('isd_code', $isdCode)->first();
                return str_replace(':mobileLength', $query['max_mobile_number_length'], $message);
            });


            //isd code validation
            Validator::extend('valid_isd_code', function ($attribute, $value, $parameters, $validator) {
                // Implement the validation logic for ISD code
                return preg_match('/^\+\d+([- ]\d+)?$/', $value);
            }, "The :attribute must start with + followed by numbers, and may be followed by - or space and more numbers.");

            $validator = Validator::make($request->all(), [
                'language_code' => 'required|in:ar,en',
                'type' => 'required|in:email,mobile',
                'isd_code' => 'nullable|required_if:type,mobile|' . ($request->input('type') == 'mobile' ? 'valid_isd_code' : ''),
                'mobile_or_email' => 'required|check_type:type|check_mobile_length:type,isd_code',
                'otp' => 'required|numeric',

            ], [
                'type.in' => 'entered type must be either email or mobile',

            ]);
            $validator->sometimes('isd_code', 'valid_isd_code', function ($input) {
                return $input->type == 'mobile';
            });

            if ($validator->fails()) {
                return $this->sendError('Validation failed', $validator->errors(), 422);
            }
            if ($request['type'] != 'email') {

                $checker = Customer::select('*', 'id as user_id')->where('mobile', '=', $request->isd_code . ' ' . $request->mobile_or_email)->where('status', CustomerStatus::Active->value)->where('status', '!=', CustomerStatus::Deleted->value)->first();
            } else {
                $checker = Customer::select('*', 'id as user_id')->where('email', '=',  $request->mobile_or_email)->where('status', CustomerStatus::Active->value)->where('status', '!=', CustomerStatus::Deleted->value)->first();
            }
            if (!empty($checker)) {
                if ($request['type'] != 'email') {

                    $userOtp = AdminUserOtp::where('mobile', '=', $request->mobile_or_email)
                        ->where('otp', '=', $request->otp)
                        ->first();
                } else {
                    $userOtp = AdminUserOtp::Where('mobile', $request->mobile_or_email)
                        ->where('otp', '=', $request->otp)
                        ->first();
                }

                if (!empty($userOtp)) {

                    $exipreDate = $userOtp['expired'];
                    $currentDate = date('Y-m-d H:i:s');

                    if (strtotime($currentDate) < strtotime($exipreDate)) {

                        AdminUserOtp::where('mobile', '=', $request->isd_code . ' ' . $request->mobile_or_email)->orWhere('mobile', $request->mobile_or_email)
                            ->where('otp', '=', $request->otp)
                            ->delete();


                        Auth::guard('appuser')->loginUsingId($checker->id);
                        $userData = Auth::guard('appuser')->user();
                        $success = $checker;
                        $success['token'] =  $userData->createToken('AuthToken')->accessToken;
                        // $success['token'] = Str::random(60);


                        return $this->sendResponse($success, 'OTP Verified Successfully');
                    } else {

                        $success = [];
                        return $this->sendError('OTP expired!', $success, 200);
                    }
                } else {
                    $success = [];
                    return $this->sendError($request['type'] . ' and otp does not match', $success, 200);
                }
            } else {
                $success = [];
                return $this->sendError('Customer not found', $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }
    /**
     * @OA\Post(
     *   path="/v1/customer/reset-password",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Send request for reset password",
     *   description="Pass language code either ar for arabic  or en for english<br>Enter new password and confirm password both are must be same",
     *   operationId="reset-password",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",
                required={"language_code","new_password","confirm_password"},
     *           @OA\Property(property="language_code", type="string",default="en"),

     *           @OA\Property(property="new_password", type="string"),
     *           @OA\Property(property="confirm_password", type="string"),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Check Users Exists api
     *
     * @return \Illuminate\Http\Response
     */
    public function resetForgotPassword(Request $request)
    {
        try {
            if (Auth::user()) {
                $customerDetails = Customer::where('id', Auth::id())->first();


                $validator = Validator::make($request->all(), [
                    'language_code' => 'required|in:ar,en',

                    'new_password' => 'required',
                    'confirm_password' => 'required|same:new_password'

                ], [
                    'language_code.in' => 'entered language code is wrong, it must be either ar or en'
                ]);


                if ($validator->fails()) {
                    return $this->sendError('Validation failed', $validator->errors(), 422);
                }

                $data['key'] = 'password';
                $data['password'] = $request->new_password;
                $responseData = resetPassword($data);

                if ($responseData['valid'] != '1') {
                    return $this->sendError($responseData, [$responseData], 422);
                }
                if ($customerDetails['email'] != "") {
                    $updatePassword = Customer::where('email', $customerDetails['email'])->update([
                        'password' => Hash::make($request->new_password)
                    ]);
                } else {
                    $updatePassword = Customer::where('mobile', $customerDetails['mobile'])->update([
                        'password' => Hash::make($request->new_password)
                    ]);
                }
                $success = [];
                return $this->sendResponse($updatePassword, 'Password Reset Successfully');
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/customer/change-password",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Change Password",
     *   description="end request for Customer Change Password",
     *   operationId="changePassword",
     *  @OA\RequestBody(
     *     required=true,
     *     description="Customer Change Password",
     *     @OA\MediaType(
     *       mediaType="multipart/form-data",
     *       @OA\Schema(
     *             required={"language_code","old_password","new_password","confirm_password"},
     *           @OA\Property(property="language_code", default="en",type="string",description="need to pass language code either 'en' or 'ar'"),
     *           @OA\Property(property="old_password", type="string",description="need to pass Old Password"),
     *           @OA\Property(property="new_password", type="string",description="need to pass New Password"),
     *           @OA\Property(property="confirm_password", type="string",description="need to pass Confirm Password")
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * OTP Verify
     *
     * @return \Illuminate\Http\Response
     */
    public function changePassword(Request $request)
    {
        try {

            if ($request->filled('type') == 'check_password') {
                $validator = Validator::make($request->all(), [
                    'old_password' => 'required',
                ]);
                if ($validator->fails()) {
                    return $this->sendError('Invalid Request', [$validator->errors()->first()]);
                }
                $validatedData = $validator->validated();
                if (!\Hash::check($validatedData['old_password'], auth()->user()->password)) {
                    return $this->sendError('The old password is incorrect.', [], 200);
                } else {
                    return $this->sendResponse('Old password is correct.', [], true);
                }
            } else {
                $customer = Customer::where('id', Auth::id())->first();
                Validator::extend('custom_rule_password', function ($attribute, $value) {
                    $query = Customer::where('password', $value)
                        ->where('status', CustomerStatus::Active->value)->where('id', Auth::id())->get();
                    return !$query->count();
                }, 'The old password has already been taken');

                if ($customer->isPasswordSet) {

                    $validator = Validator::make($request->all(), [
                        'language_code' => 'required|in:ar,en',
                        'old_password' =>  [
                            'required',
                            function ($attribute, $value, $fail) {
                                // Check if the provided old password matches the authenticated user's current password
                                if (!\Hash::check($value, auth()->user()->password)) {
                                    $fail("The $attribute is incorrect.");
                                }
                            },
                        ],
                        'new_password' => 'required',
                        'confirm_password' => 'required|same:new_password',
                    ]);
                } else {
                    $validator = Validator::make($request->all(), [
                        'language_code' => 'required|in:ar,en',

                        'new_password' => 'required',
                        'confirm_password' => 'required|same:new_password',
                    ]);
                }

                $data['key'] = 'password';
                $data['password'] = $request->new_password;
                $responseData = resetPassword($data);

                if ($responseData['valid'] != '1') {
                    return $this->sendError($responseData, $responseData, 200);
                }
                if ($validator->fails()) {
                    return $this->sendError('Validation Failed', [$validator->errors()], 422);
                }
                $requestData = $request->all();
                if (Auth::user()) {
                    $data = [
                        'password' => Hash::make($requestData['new_password']),
                    ];
                    $response = Customer::where('id', Auth::id())->update($data);
                    if (!empty($response)) {
                        $checkNotifyEnable = Setting::where('config_key', 'passwordSecurity|changePasswordNotify')->get('value')[0]['value'];
                        if ($checkNotifyEnable == '1') {

                            $customerDetail = Customer::where('id', Auth::id())->get()->toArray();

                            $agencyName = Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'];
                            $agencyLogo = Setting::where('config_key', 'general|basic|colorLogo')->get('value')[0]['value'] ?? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'];
                            $language_code = $requestData['language_code'];

                            $code = 'CHANGE_PASSWORD';
                            $siteName = count(Setting::where('config_key', 'general|basic|siteName')->get('value')) > 0 ? Setting::where('config_key', 'general|basic|siteName')->get('value')[0]['value'] : "Amar Infotech";
                            $customerName = ucwords($customerDetail[0]['first_name'] . ' ' . $customerDetail[0]['last_name']);

                            $data = array(
                                'customer_name' => $customerName,
                                'site_name' => $siteName,
                                'agency_name' => $agencyName,
                                'agency_logo' => $agencyLogo
                            );

                            $getTemplateData = $this->changePasswordMailTemplate($code, $data, $language_code);

                            if ($getTemplateData['status'] == 'false') {
                                return back()->with('error', $getTemplateData['error']);
                            } else {
                                $subject = $getTemplateData['data']['subject'];
                                $mailData = $getTemplateData['data']['mailData'];
                                $toEmail = $customerDetail[0]['email'];
                                $files = [];

                                // set data in sendEmail function
                                $data = $this->sendEmail($toEmail, $subject, $mailData, $files, $siteName, $code);
                                $otp_verification = false;
                                $otp_setting_data = \App\Models\Setting::where('config_key', 'general|otp|phoneVerification')->get()->toArray();

                                $currentDate = date('Y-m-d H:i:s');
                                $message = "Dear " . $customerName . ",Your " . $agencyName . " Account Password was Changed on" . $currentDate .
                                    "If you did this, you can safely disregard this sms. If you did not do this, please secure your account or Contact Us.Thanks," . $agencyName . " Team!! ";
                                if (!empty($otp_setting_data)) {
                                    if ($otp_setting_data[0]['value'] == "on") {
                                        $otp_verification = true;
                                        //send OTP SMS code
                                        $this->sendSms($customerDetail[0]['mobile'], $message);
                                    }
                                }
                            }
                        }
                        return $this->sendResponse($success = [], 'Your Password changed Successfully');
                    }
                } else {
                    $success = [];
                    return $this->sendError('User not found', $success, 200);
                }
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }
    /**
     * @OA\Post(
     *   path="/v1/customer/email-verify",
     *  security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Verify customer email",
     *   description="Initiates the email verification process for a customer.</br>",
     *   operationId="emailVerify",
     *   @OA\RequestBody(
     *     required=true,
     *     description="The request body for email verification",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             required={"email","token"},
     *             @OA\Property(property="email", type="string", example="test@gmail.com", description="The user's email address."),
     *             @OA\Property(property="token", type="string", example="abcdef123456", description="The verification token for email confirmation."),
     *      )
     *    ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Login Send
     *
     * @return \Illuminate\Http\Response
     */
    public function emailVerify(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email',
                'token' => 'required',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid Request', [$validator->errors()->first()]);
            }

            $checkEmailAlreadyIsVerify = Customer::select('email', 'is_email_verified')->where('email', $request->email)->first();
            if ($checkEmailAlreadyIsVerify) {
                if ($checkEmailAlreadyIsVerify->is_email_verified == 0) {
                    $checkEmailORTokenIsNotExpired = DB::table('customer_activation_tokens')
                        ->where('email', $request->email)
                        ->where('token', $request->token)
                        ->first();

                    if ($checkEmailORTokenIsNotExpired) {

                        $currentTimestamp = Carbon::now()->timestamp;
                        $expiryTimestamp = Carbon::parse($checkEmailORTokenIsNotExpired->created_at)->addHours(24)->timestamp;

                        if ($currentTimestamp <= $expiryTimestamp) {

                            $updateEmailIsVerify = Customer::where(['email' => $checkEmailORTokenIsNotExpired->email])->update(
                                ['is_email_verified' => true]
                            );

                            $deleteRecords = DB::table('customer_activation_tokens')->where(['email' => $checkEmailORTokenIsNotExpired->email])
                                ->delete();

                            $success = [];
                            return $this->sendResponse($success, 'Email Is Verified Successfully');
                        } else {
                            $success = [];
                            return $this->sendResponse($success, 'Token has expired');
                        }
                    } else {
                        $success = [];
                        return $this->sendError('Invalid email or token', $success, 200);
                    }
                } else {
                    $success = [];
                    return $this->sendError('Email Is Already Verified!', $success, 200);
                }
            } else {
                $success = [];
                return $this->sendError('Invalid Email', $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }


    /**
     * @OA\Post(
     *   path="/v1/customer/close-account",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Close Account",
     *   description="",
     *   operationId="closeAccount",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",
     *            required={"language_code","deleted_reason"},
     *            @OA\Property(property="language_code", type="string", default="en" ),
     *            @OA\Property(property="deleted_reason", type="string",  ),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Close Account
     *
     * @return \Illuminate\Http\Response
     */
    public function closeAccount(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'language_code' => 'required|in:ar,en',
                'deleted_reason' => 'required',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Failed', [$validator->errors()->first()], 422);
            }
            Customer::where('id', Auth::user()->id)->update([
                'deleted_reason' => $request->deleted_reason,
                'status' => CustomerStatus::Deleted->value,
                'is_email_verified' => '0',
                'is_mobile_verified' => '0',
            ]);
            // $this->logout();
            if (Auth::guard('appuser-api')->check()) {
                $auth = "appuser-api";
            }

            if ($auth) {
                Auth::guard($auth)->user()->token()->revoke();
            }
            $success = [];
            return $this->sendResponse($success, 'Your account has been Deleted Successfully.');
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }


    /**
     * @OA\Post(
     *   path="/v1/customer/logout-customer",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Logout",
     *   description="logout",
     *   operationId="logoutCustomer",
     *   @OA\RequestBody(
     *     required=false,
     *     description="Logout",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             @OA\Property(property="device_id", type="string", example="mobileorwebsitedeviceid"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Logout
     *
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {

        $auth = null;

        $user_id = Auth::user()->id;

        $requestData = $request->all();

        if (isset($requestData['device_id']) && $requestData['device_id'] != '') {
            UserLoginHistory::where('user_id', $user_id)
                ->where('device_id', $requestData['device_id'])
                ->delete();
        }

        if (Auth::guard('appuser-api')->check()) {
            $auth = "appuser-api";
        }

        if ($auth) {
            //Auth::guard($auth)->user()->token()->revoke();
            Auth::user()->tokens->each(function ($token, $key) {
                $token->delete();
            });
        }

        $success = [];
        return $this->sendResponse($success, 'User Logout Successfully');
    }

    public function deleteCustomer()
    {
        $id = Auth::id();
        $result = Customer::deleteCustomer($id);
        if ($result['status'] == 1) {
            return $this->sendResponse([], $result['message'], true);
        } else {
            return $this->sendResponse([], $result['message'], false);
        }
    }

    public function updateCustomerData(Request $request)
    {
        try {
            $id = Auth::id();
            $requestData = $request->all();
            return Customer::updateCustomerData($id, $requestData);
        } catch (\Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    public function getWalletList()
    {
        try {
            $id = Auth::id();
            $walletList = Wallet::getCustomerWalletList($id);
            return $this->sendResponse($walletList, 'Customer wallet list have listed successfully');
        } catch (\Exception $e) {
            $success = [];
            return $this->sendError('Something went wrong', ['error' => $e], 500);
        }
    }

    public function getCouponList()
    {
        try {
            $id = Auth::id();
            $usedCoupons = [];
            $bookings = Bookings::whereCustomer_id($id)
                ->where('coupon_code', '!=', '')
                ->select('booking_ref', 'rehtle_ref', 'coupon_amount', 'coupon_code', 'created_at')
                ->get();
            if ($bookings) {
                $bookings = $bookings->toArray();
                $couponCodes = [];
                foreach ($bookings as $booking) {
                    if (!in_array($booking['coupon_code'], $couponCodes)) {
                        array_push($couponCodes, $booking['coupon_code']);
                    }
                }
                $coupons = Coupon::whereIn('coupon_code', $couponCodes)->get()->toArray();
                $tempCoupons = [];
                foreach ($coupons as $coupon) {
                    $tempCoupons[$coupon['coupon_code']] = $coupon;
                }
                foreach ($bookings as $booking) {
                    array_push($usedCoupons, [
                        'booking_ref' => $booking['booking_ref'],
                        'rehlte_ref' => $booking['rehtle_ref'],
                        'coupon_code' => $booking['coupon_code'],
                        'discount_value' => $booking['coupon_amount'],
                        'issued_date' => $tempCoupons[$booking['coupon_code']]['from_date'],
                        'expire_date' => $tempCoupons[$booking['coupon_code']]['to_date'],
                        'redemption_date' => $booking['created_at'],
                    ]);
                }
            }

            return $this->sendResponse(
                $usedCoupons,
                'Customer used coupons have listed successfully',
                true
            );
        } catch (\Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    public function handleFavoriteHotel(Request $request)
    {
        $requestData = $request->all();
        try {
            $id = Auth::id();
            $customerDetails = Customer::whereId($id)
                ->select('favorite_hotels')->first();


            $favoriteHotel = $customerDetails->favorite_hotels;
            if ($favoriteHotel == '') {
                $favoriteHotels = [];
            } else {
                $favoriteHotels = explode(',', $favoriteHotel);
            }
            if ($requestData['mode'] == 'add') {
                array_push($favoriteHotels, $requestData['hotel_id']);
                $message = 'Hotel has been added to favorites successfully';
            } else {
                if (($key = array_search($requestData['hotel_id'], $favoriteHotels)) !== false) {
                    unset($favoriteHotels[$key]);
                }
                $message = 'Hotel has been removed from favorites successfully';
            }
            $favoriteHotel = implode(',', $favoriteHotels);
            $result = Customer::updateCustomerData($id, ['favorite_hotels' => $favoriteHotel]);
            if ($result['status'] == true) {
                return $this->sendResponse([], $message, $result['status']);
            } else {
                return $this->sendResponse([], 'Unable to process request. Please try again.', $result['status']);
            }
        } catch (\Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    public function dashboardData(Request $request)
    {
        try {
            $id = Auth::id();
            $currency = $request->input('currency', Currency::SAR->value);
            $currentDate = now()->format('Y-m-d');

            // Total trips
            $allTrips = Bookings::where('customer_id', $id)
                ->where('booking_status',  BookingStatus::Confirmed->value)
                ->count();

            // Upcoming trips
            $upcomingTrips = Bookings::where('customer_id', $id)
                ->where('booking_status', BookingStatus::Confirmed->value)
                ->where('booking_start_date', '>=', $currentDate)
                ->count();

            // Reward points: only for bookings after service end date, and not used
            $rewardPoints = CustomerLoyaltyPoints::getTotalRewardPoints($id);

            // Wallet amount
            $walletAmount = Wallet::getTotalWalletAmount($id);
            $walletAmount = convertCurrencyExchangeRate($walletAmount, Currency::SAR->value, $currency);

            $result = [
                'all_trips' => $allTrips,
                'upcoming_trips' => $upcomingTrips,
                'reward_points' => $rewardPoints,
                'wallet_amount' => $walletAmount['data']['convertedRate'] ?? 0,
            ];

            return $this->sendResponse($result, 'Dashboard data fetched successfully', true);
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }


    // Google Sign In Using email (For Mobile App)
    public function googleSignInWithEmail(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'locale' => 'string|in:ar,en',
                'email' => 'required|email',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Validation Error', $validator->errors()->first(), 422);
            }

            $customerDetails = Customer::where('email', $request->email)->first();

            if (self::checkValidCustomer($customerDetails)) {
                return $this->sendResponse([], 'You no longer access your account. Please create new one to continue', false);
            }

            if ($customerDetails) {
                $customerDetails->token = $customerDetails->createToken('AuthToken')->accessToken;
                return $this->sendResponse($customerDetails, "Welcome Back. You've successfully logged in");
            } else {
                $customer = Customer::create([
                    'email' => $request->email,
                    'status' => CustomerStatus::Active->value,
                    'is_email_verified' => 1,
                ]);
                $customerDetails = Customer::where('email', $request->email)->first();
                $customerDetails->token = $customerDetails->createToken('AuthToken')->accessToken;
                return $this->sendResponse($customerDetails, "Welcome Back. You've successfully logged in");
            }
        } catch (\Exception $e) {
            $success = [];
            return $this->sendError('Something went wrong', ['error' => $e], 500);
        }
    }

    // Verify Guest Customer Mobile Number at Traveller Info Page

    public function verifyCustomerwithMobile(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'customer_mobile' => 'required|string',
                'isd_code' => 'required|string',
                'isGuestUser' => 'required|boolean',
                'deviceId' => 'required',
            ]);


            if ($validator->fails()) {
                $this->sendError('Validation Error', $validator->errors()->first(), 200);
            }

            $otp_setting_data_all = \App\Models\Setting::where('config_key', 'general|otp|phoneVerification')->get()->toArray();
             // Temporarily Disable Mobile Verification
            $otp_setting_data_booking = \App\Models\Setting::where('config_key', 'general|otp|mobileBookingVerification')->get()->toArray();
            if (!empty($otp_setting_data_all) && !empty($otp_setting_data_booking)) {
                if ($otp_setting_data_booking[0]['value'] == 'off' || $otp_setting_data_booking[0]['value'] == 'off') {
                   
                    $data = [
                        'type' => 'account_verified',
                    ];
                    return $this->sendResponse($data, "Customer Already Verified");
                }
            }


            $mobile = trim($request->isd_code . ' ' . $request->customer_mobile);
            $customer = Customer::where('mobile', $mobile)->latest()->first();
            if (self::checkValidCustomer($customer)) {
                $data = [
                    'type' => 'account_deleted',
                ];
                return $this->sendResponse($data, "Your Account has been " . $customer?->status . ". Please contact customersupport@rehlte.com.");
            }
            $isDeviceIdExists = CustomerDeviceId::where('mobile_no', $request->customer_mobile)
                ->where(function ($query) use ($request) {
                    if (is_null($request->deviceId)) {
                        $query->whereNull('device_id');
                    } else {
                        $query->where('device_id', $request->deviceId);
                    }
                })
                ->exists();

            if (!$request->isGuestUser && $customer?->is_mobile_verified) {
                $data = [
                    'type' => 'account_verified',
                ];
                return $this->sendResponse($data, "Customer Already Verified");
            } else {
                if ($isDeviceIdExists) {
                    $data = [
                        'type' => 'account_verified',
                    ];
                    return $this->sendResponse($data, "Customer Already Verified");
                } else {
                    $data = [
                        'type' => 'not_verified',
                    ];
                    return $this->sendResponse($data, "Mobile number not verified. Please Verify your mobile number");
                }
            }
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()]);
        }
    }

    /**
     * Admin Login as Customer function 
     */
    public function loginAsCustomer(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'customer_id' => 'required',
                'token' => 'required'
            ]);
            $success = [];
            if ($validator->fails()) {
                $this->sendError('Validation Error', $validator->errors()->first(), 422);
            }

            $validatedData = $validator->validated();
            $customerLoginToken = CustomerLoginToken::whereCustomer_id($validatedData['customer_id'])->whereToken($validatedData['token'])->first();

            if (!$customerLoginToken || $customerLoginToken->expires_at < now() || $customerLoginToken->used) {
                return $this->sendResponse([], 'Invalid or expired token', false);
            }

            $customer = Customer::find($validatedData['customer_id']);
            if (!$customer) {
                return $this->sendResponse([], 'Customer Not Found', false);
            }
            $customerLoginToken->used = true;
            $customerLoginToken->save();
            $success = $customer;
            $success['token'] = $customer->createToken('AuthToken')->accessToken;
            $success['isPasswordSet'] = $customer->password ? true : false;
            return $this->sendResponse($success, 'You have successfully logged in as ' . ($customer?->first_name ?? 'Customer'));
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', ['error' => $e->getMessage()]);
        }
    }



    /**
     * Social Login function
     * */

    public function googleSignIn(Request $request)
    {

        try {
            $requestData = $request->json()->all();


            if (empty($requestData['email']) || empty($requestData['name'])) {
                return $this->sendError('Something went wrong. No response received.', [], 422);
            }


            $finduser = Customer::where('email', $requestData['email'])->where('status', '!=', CustomerStatus::Deleted->value)->latest()->first();


            if ($finduser) {
                if (Self::checkValidCustomer($finduser)) {
                    return $this->sendResponse([], 'You no longer access your acount. Please contact customersupport@rehlte.com.', false);
                }

                Auth::login($finduser);

                $token = $finduser->createToken('auth_token')->accessToken;
                $finduser->token = $token;
                $finduser->isPasswordSet = $finduser->password ? true : false;
                return $this->sendResponse($finduser, 'Welcome!. You have logged in successfully');
            } else {

                $newUser = Customer::create([
                    'first_name' => $requestData['name'],
                    'email' => $requestData['email'],
                    'google_id' => $requestData['id'] ?? Str::random(24),
                    'is_email_verified' => 1,
                    'password' => null,
                ]);


                $newUser = Customer::where('email', $requestData['email'])->first();
                $newUser->makeHidden(['password']);


                Auth::login($newUser);
                $token = $newUser->createToken('auth_token')->accessToken;
                $newUser->token = $token;
                $newUser->isPasswordSet = $newUser->password ? true : false;
                return $this->sendResponse($newUser, 'Welcome!. You have logged in successfully');
            }
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', [$e->getMessage()], 500);
        }
    }


    public function facebookSignIn(Request $request)
    {
        try {
            $requestData = $request->json()->all();


            if (empty($requestData['email']) || empty($requestData['first_name'])) {
                return $this->sendError('Something went wrong. No response received.', [], 422);
            }


            $finduser = Customer::where('email', $requestData['email'])->where('status', '!=', CustomerStatus::Deleted->value)->latest()->first();


            if ($finduser) {
                if (self::checkValidCustomer($finduser)) {
                    return $this->sendResponse([], 'You no longer access your acount. Please create a new one to continue', false);
                }
                Auth::login($finduser);

                $token = $finduser->createToken('auth_token')->accessToken;
                $finduser->token = $token;
                $finduser->isPasswordSet = $finduser->password ? true : false;
                return $this->sendResponse($finduser, 'Welcome!. You have logged in successfully');
            } else {

                $newUser = Customer::create([
                    'first_name' => $requestData['first_name'],
                    'last_name' => $requestData['last_name'],
                    'email' => $requestData['email'],
                    'facebook_id' => $requestData['id'] ?? Str::random(24),
                    'is_email_verified' => 1,
                    'password' => null,
                ]);


                $newUser = Customer::where('email', $requestData['email'])->first();
                $newUser->makeHidden(['password']);

                Auth::login($newUser);
                $token = $newUser->createToken('auth_token')->accessToken;
                $newUser->token = $token;
                $newUser->isPasswordSet = $newUser->password ? true : false;
                return $this->sendResponse($newUser, 'Welcome!. You have logged in successfully');
            }
        } catch (\Exception $e) {
            return $this->sendError('Something went wrong', [$e->getMessage()], 500);
        }
    }

    private static function checkValidCustomer($customer)
    {
        return $customer?->status == CustomerStatus::InActive->value || $customer?->status == CustomerStatus::Terminated->value;
    }
}
