<?php

namespace App\Http\Controllers\API\V1;

use Exception;
use Carbon\Carbon;
use App\Models\Page;
use App\Enums\Locale;
use App\Models\Coupon;
use App\Models\Wallet;
use App\Enums\Currency;
use App\Models\Airline;
use App\Models\Payment;
use App\Models\Setting;
use App\Models\Bookings;
use App\Traits\Gimmonix;
use App\Models\PageI18ns;
use App\Traits\ActiveLog;
use App\Enums\ServiceType;
use App\Models\StateI18ns;
use App\Traits\SmsService;
use App\Models\SupplierLog;
use App\Traits\EditBooking;
use App\Enums\BookingStatus;
use App\Enums\PaymentStatus;
use App\Models\HotelAddress;
use App\Traits\EmailService;
use Illuminate\Http\Request;
use App\Traits\BookingHelper;
use App\Traits\CommonService;
use App\Traits\TraacsService;
use App\Models\BookingHistory;
use App\Traits\AmadeusService;
use App\Traits\BookingService;
use App\Traits\PaymentService;
use App\Models\BookingCheckout;
use App\Models\EditHotelDetail;
use App\Models\MostBookedHotel;

use App\Traits\WhatsappService;
use App\Jobs\MostBookedHotelJob;
use App\Models\EditFlightDetail;
use App\Models\CustomerTraveller;
use Illuminate\Support\Facades\DB;
use App\Models\LoyaltyPointSetting;
use App\Traits\LoyaltyPointsService;
use Illuminate\Support\Facades\Auth;
use App\Models\CustomerLoyaltyPoints;
use App\Models\BookingAdditionalDetail;
use Illuminate\Support\Facades\Validator;
use Google\Service\ShoppingContent\LoyaltyPoints;
use App\Http\Controllers\API\V1\BaseController as BaseController;
use App\Services\TranslateBookingDetails;

class BookingController extends BaseController
{
    use AmadeusService, BookingHelper, CommonService, EmailService, SmsService, WhatsappService, ActiveLog, BookingService, LoyaltyPointsService, PaymentService, TraacsService, Gimmonix, EditBooking;

    public function __construct()
    {
        $this->perPage = count(Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')) > 0 ? Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')[0]['value'] : "20";

        //set AMADEUS API configuration from config key
        $this->amadeusAPIEnvironment = env('AMADEUS_API_CREDENTIALS');

        if ($this->amadeusAPIEnvironment == 'test') {
            $this->amadeusAPIEndPoint = env('AMADEUS_API_TEST_API_ENDPOINT');
            $this->amadeusAPIClientID = env('AMADEUS_API_TEST_CLIENT_ID');
            $this->amadeusAPIClientSecret = env('AMADEUS_API_TEST_CLIENT_SECRET');
            $this->amadeusAPIGrantType = env('AMADEUS_API_TEST_GRANT_TYPE');
        } else {
            $this->amadeusAPIEndPoint = env('AMADEUS_API_LIVE_API_ENDPOINT');
            $this->amadeusAPIClientID = env('AMADEUS_API_LIVE_CLIENT_ID');
            $this->amadeusAPIClientSecret = env('AMADEUS_API_LIVE_CLIENT_SECRET');
            $this->amadeusAPIGrantType = env('AMADEUS_API_LIVE_GRANT_TYPE');
        }

        $this->amadeusAPISecret = env('AMADEUS_API_SECRET');
    }
    /**
     * @OA\Get(
     *   path="/v1/customer/get-booking-list",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Get Booking List",
     *   description="get Booking List",
     *   operationId="getBooking",

     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * get driver status
     *
     * @return \Illuminate\Http\Response
     */
    public function getBookings(Request $request)
    {
        try {
            $showBookingData = false;
            $isSingleResult = false;
            $locale = $request->input('locale');

            if ($request->input('id') != NULL || $request->input('ref') != NULL) {
                $showBookingData = true;
                $isSingleResult = true;
            }
            if ($request->input('email') != NULL) {
                $showBookingData = true;
            }
            if ($request->input('booking_type') != NULL) {
                $showBookingData = true;
            }
            if (Auth::check()) {
                $showBookingData = true;
            }

            if ($showBookingData) {
                if ($isSingleResult && $request->input('ref') != NULL) {
                    $bookingData = Bookings::with(['payment', 'passengers'])
                        ->where('booking_ref', $request->input('ref'))
                        ->first();
                    if ($bookingData) {
                        $bookingData = $bookingData->toArray();
                    }
                } elseif ($isSingleResult && $request->input('id') != NULL) {
                    $bookingData = Bookings::with(['payment', 'passengers'])
                        ->where('id', $request->input('id'))
                        ->first();
                    if ($bookingData) {
                        $bookingData = $bookingData->toArray();
                    }
                } else {
                    $query = Bookings::with(['payment', 'passengers']);
                    if ($request->input('email') != NULL) {
                        $query = $query->where('email', '=', $request->input('email'));
                    }
                    if (Auth::check()) {
                        $query = $query->where('customer_id', '=', Auth::id());
                    }
                    if ($request->input('booking_type') != NULL) {
                        $query = $query->where('booking_type', '=', $request->input('booking_type'));
                    }
                    if ($request->input('future_bookings') != '') {
                        $query = $query->where('booking_start_date', '>', date('Y-m-d'))
                            ->where('booking_status', '=', BookingStatus::Confirmed->value);
                    }

                    $bookingData = $query->orderBy('created_at', 'DESC')->get()->toArray();
                }

                if ($bookingData) {
                    $success = true;
                    $serviceTypes = $this->getServiceTypes();
                    $supplierNames = $this->getSuppliers();

                    if ($isSingleResult) {
                        $bookingData['supplier'] = $supplierNames[$bookingData['supplier_id']] ?? '';
                        $bookingData['service'] = $serviceTypes[$bookingData['service_id']] ?? '';
                        $bookingData['agency'] = '';

                        // eager loaded already
                        $bookingData['payment'] = $bookingData['payment'] ?? [];
                        $bookingData['passengers'] = $bookingData['passengers'] ?? [];
                    } else {
                        foreach ($bookingData as $key => $booking) {
                            $bookingData[$key]['supplier'] = $supplierNames[$booking['supplier_id']] ?? '';
                            $bookingData[$key]['service'] = $serviceTypes[$booking['service_id']] ?? '';
                            $bookingData[$key]['agency'] = '';

                            // eager loaded already
                            $bookingData[$key]['payment'] = $booking['payment'] ?? [];
                            $bookingData[$key]['passengers'] = $booking['passengers'] ?? [];
                        }
                    }

                    if ($locale == Locale::Arabic->value) {
                        $bookingData = array_map(function ($booking) {
                            if ($booking['booking_status'] == BookingStatus::Confirmed->value) {
                                $booking['booking_status_ar'] = 'مؤكد';
                            }
                            if ($booking['booking_status'] == BookingStatus::Failed->value) {
                                $booking['booking_status_ar'] = 'فشل';
                            }
                            if ($booking['booking_status'] == BookingStatus::Pending->value) {
                                $booking['booking_status_ar'] = 'قيد الانتظار';
                            }

                            if ($booking['booking_type'] == ServiceType::Hotel->value) {
                                $booking['booking_type_ar'] = 'الفندق';
                                $booking['service_ar'] = 'الفندق';
                            }
                            if ($booking['booking_type'] == ServiceType::Flight->value) {
                                $booking['booking_type_ar'] = 'رحلة جوية';
                                $booking['service_ar'] = 'رحلة جوية';
                            }

                            return $booking;
                        }, is_array($bookingData) && isset($bookingData[0]) ? $bookingData : [$bookingData]);

                        // if it was single result, return first item
                        if ($isSingleResult && isset($bookingData[0])) {
                            $bookingData = $bookingData[0];
                        }
                    }

                    return $this->sendResponse($bookingData, 'Booking Listed Successfully!', $success);
                } else {
                    $success = [];
                    return $this->sendError(no_record_found, $success, 200);
                }
            } else {
                return $this->sendError(no_record_found, [], 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrongs ' . $e->getMessage(), 500);
        }
    }


    public function processViewBooking($bookingData)
    {
        if ($bookingData['service_provider'] == 'GIMMONIX') {
            $bookingAdditionalDetails = BookingAdditionalDetail::where('booking_id', $bookingData['id'])->first()->toArray();
            $bookingPayloadDetails = json_decode($bookingAdditionalDetails['booking_details'], true);
        } else {
            $bookingPayloadDetails = json_decode($bookingData['booking_details'], true);
        }
        $bookingData['booking_details'] = $bookingPayloadDetails;
        if ($bookingData['booking_type'] == ServiceType::Flight->value) {
            $bookingData['flightRoute'] = $bookingData['additional_details'];
        } elseif ($bookingData['booking_type'] == ServiceType::Hotel->value) {
            $bookedRooms = [];
            if ($bookingData['booking_type'] == 'HOTEL_BEDS') {
                $bookingDetails = $bookingPayloadDetails;
                $hotelDetails = $bookingDetails['hotelDetails']['hotelDetails'];
                $selectedRooms = $bookingDetails['hotelDetails']['selectedRooms'];
                $roomNameBoardDetails = [];
                foreach ($hotelDetails['rooms'] as $key => $room) {
                    foreach ($room['rates'] as $rate) {
                        $roomNameBoardDetails[$rate['rateKey']] = [
                            'roomName' => ucwords($room['name']),
                            'boardCode' => $rate['boardCode'],
                            'boardName' => $rate['boardName'],
                        ];
                    }
                }
                foreach ($selectedRooms['rooms'] as $key => $selectedRoom) {
                    $bookedRooms[$key] = [
                        'room' => ('Room ' . ($key + 1)),
                        'roomName' => $roomNameBoardDetails[$selectedRoom['rateKey']]['roomName'],
                        'boardName' => $roomNameBoardDetails[$selectedRoom['rateKey']]['boardName'],
                        'paxes' => $selectedRoom['paxes'],
                        'rateComments' => ''
                    ];
                    if (isset($selectedRoom['rateComments'])) {
                        $bookedRooms[$key]['rateComments'] = str_replace(' .  ', '&nbsp;&#x2022;&nbsp;&nbsp;', $selectedRoom['rateComments']);
                    }
                }
            } elseif ($bookingData['booking_type'] == 'GIMMONIX') {
                $selectedRooms = $bookingPayloadDetails['hotelDetails']['selectedRooms'];
                foreach ($bookingPayloadDetails['searchDetails']['occupancies'] as $key => $occupancy) {
                    $bookedRoom = [
                        'room' => ('Room ' . ($key + 1)),
                        'roomName' => $selectedRooms['RoomName'],
                        'boardName' => $selectedRooms['RoomBasis'],
                        'paxes' => '',
                        'rateComments' => (array_key_exists('Remarks', $selectedRooms)) ? implode(',', $selectedRooms['Remarks']) : ''
                    ];
                    array_push($bookedRooms, $bookedRoom);
                }
            }
            $bookingData['bookedRooms'] = $bookedRooms;
        }
        return $bookingData;
    }

    public function getBookingPassengers($bookingType, $bookingId)
    {
        $travelers = CustomerTraveller::select('customers_traveller.*')->where('booking_id', $bookingId)->get();
        if ($bookingType == ServiceType::Hotel->value) {
            $passengers = [];
            if ($travelers && count($travelers) > 0) {
                foreach ($travelers as $traveler) {
                    $passenger = [
                        'booking_id' => $traveler['booking_id'],
                        'first_name' => $traveler['first_name'],
                        'last_name' => $traveler['last_name'],
                        'room' => '',
                        'type' => '',
                        'roomType' => '',
                        'boardName' => '',
                    ];
                    if ($traveler['additional_details'] != '') {
                        $additionalDetails = json_decode($traveler['additional_details'], true);
                        $passenger['room'] = $additionalDetails['room'];
                        $passenger['type'] = $additionalDetails['paxType'];
                        $passenger['roomType'] = $additionalDetails['roomName'];
                        $passenger['boardName'] = $additionalDetails['boardName'];
                    }
                    array_push($passengers, $passenger);
                }
            }
            $travelers = $passengers;
        }
        return $travelers;
    }
    /**
     * @OA\Get(
     *   path="/v1/customer/get-booking-list",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Get Booking List",
     *   description="get Booking List",
     *   operationId="getBooking",

     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * get driver status
     *
     * @return \Illuminate\Http\Response
     */
    public function getBooking(Request $request)
    {
        try {
            $perPage = Setting::where('config_key', 'general|setting|pagePerAPIRecords')->value('value');
            $filter = array(
                'per_page' => $perPage,
            );
            $locale = $request->input('locale');
            $showBookingData = false;
            $isSingleResult = false;
            if (request()->input('id') != NULL) {
                $filter['id'] = request()->input('id');
                $showBookingData = true;
                $isSingleResult = true;
            }
            // if (request()->input('email') != NULL) {
            //     $filter['where'][] = ['email', '=', request()->input('email')];
            //     $showBookingData = true;
            // }
            // if (request()->input('ref') != NULL) {
            //     $filter['where'][] = ['booking_ref', '=', request()->input('ref')];
            //     $showBookingData = true;
            //     $isSingleResult = true;
            // }
            if (
                request()->filled('email') &&
                request()->filled('ref')
            ) {
                $filter['where'][] = ['email', '=', request()->input('email')];
                $filter['where'][] = ['booking_ref', '=', request()->input('ref')];
                $showBookingData = true;
                $isSingleResult = true;
            }
            if ($showBookingData) {
                $bookingDataList = Bookings::getBookings($filter);
                $bookingData = $bookingDataList['data'];

                if ($bookingData) {
                    $additionalbookingDetails = BookingAdditionalDetail::where('booking_id', $bookingData['id'])->first();

                    $success = true;
                    $serviceTypes = $this->getServiceTypes();
                    $supplierNames = $this->getSuppliers();
                    if ($isSingleResult) {
                        $bookingData['supplier'] = $supplierNames[$bookingData['supplier_id']];
                        $bookingData['service'] = $serviceTypes[$bookingData['service_id']];
                        $bookingData['agency'] = '';
                        $bookingData['payment'] = Payment::select('payments.*')->where('booking_id', $bookingData['id'])->first();
                        $bookingData['passengers'] = $this->getBookingPassengers($bookingData['booking_type'], $bookingData['id'], $bookingData['service_provider']);

                        $bookingData = $this->processViewBooking($bookingData);
                    } else {
                        foreach ($bookingData as $key => $booking) {
                            $bookingData[$key]['supplier'] = $supplierNames[$booking['supplier_id']];
                            $bookingData[$key]['service'] = $serviceTypes[$booking['service_id']];
                            $bookingData[$key]['agency'] = '';

                            // $bookingData[$key]['payment'] = Payment::select('payments.*')->where('booking_id', $booking['id'])->first();
                            $bookingData[$key]['payment'] = $booking['payment'] ?? [];
                            $bookingData[$key]['passengers'] = $this->getBookingPassengers($booking['booking_type'], $booking['id'], $booking['service_provider']);

                            $bookingData[$key] = $this->processViewBooking($bookingData[$key]);
                        }
                    }

                    $pages = Page::where('page_code', 'Important_Information')->pluck('id')->first();
                    if (!empty($pages)) {
                        $importantInformation = PageI18ns::where('page_id', $pages)->get();
                        $bookingData['importantInformation'] = $importantInformation->toArray();
                    }

                    $editedbookingData = $this->getEditedBooking($bookingData['id'], $bookingData['booking_type']);
                    if (!empty($editedbookingData)) {
                        if ($bookingData instanceof \App\Models\Bookings) {
                            $bookingData = $bookingData->toArray();
                        }
                        /*** Booking Details json in empty in Booking Table for Failed , Processing and Pending Status booking, so we get data from additional booking details table 
                         ***/
                        if (!$bookingData['booking_details']) {

                            $bookingData['booking_details']['booking_details'] = json_decode($additionalbookingDetails['booking_details'], true) ?? '';
                        }
                        $bookingData['first_name'] = $editedbookingData['first_name'] ?? '';
                        $bookingData['last_name'] = $editedbookingData['last_name'] ?? '';
                        $bookingData['email'] = $editedbookingData['email'] ?? '';
                        $bookingData['booking_date'] = $editedbookingData['booking_date'] ?? '';
                        $bookingData['booking_start_date'] = $editedbookingData['booking_start_date'] ?? '';

                        $editedProcessedPrice = json_decode($editedbookingData['processedPrice'], true);
                        $editedBillingDetails = json_decode($editedbookingData['billingDetails'], true) ?? '';
                        if ($bookingData['booking_type'] == ServiceType::Hotel->value) {
                            $editedHotelDetails = json_decode($editedbookingData['hotelDetails'], true) ?? '';
                            $editedSearchDetails = json_decode($editedbookingData['searchDetails'], true) ?? '';
                        } else if ($bookingData['booking_type'] == ServiceType::Flight->value) {
                            $editedFlightDetails = json_decode($editedbookingData['flightDetails'], true) ?? '';
                            $editedPassengerDetails = json_decode($editedbookingData['passengerDetails'], true) ?? '';
                            $completeFlightDetails = json_decode($editedbookingData['completeflightDetails'], true) ?? [];


                            $airLineListData = $bookingData['booking_details']['booking_details']['searchDetails']['airlineList'];
                            $bookingData['booking_details']['booking_details']['searchDetails']['airlineList'] =  $this->fetchAirlineData($airLineListData);
                        }

                        if ($bookingData['booking_details']) {
                            // Update Edited  Processed Price to Outer Booking Details Json
                            $bookingData['booking_details']['processedPrice'] = $editedProcessedPrice  ?? '';
                            $bookingData['booking_details']['processedPrice']['chargeableSeatPrice'] = $editedProcessedPrice['selectedSeatsTotalPrice'] ?? '';


                            // Update Edited  Processed Price to inner Booking Details Json
                            $bookingData['booking_details']['booking_details']['processedPrice'] = $editedProcessedPrice ?? '';
                            $bookingData['booking_details']['booking_details']['processedPrice']['chargeableSeatPrice'] = $editedProcessedPrice['selectedSeatsTotalPrice'] ?? '';


                            // Update Edited Billing Details to Booking Details Json Billing Details Array
                            $bookingData['booking_details']['billingDetails']['address'] = $editedBillingDetails['address'] ?? '';
                            $bookingData['booking_details']['billingDetails']['address1'] = $editedBillingDetails['address1'] ?? '';
                            $bookingData['booking_details']['billingDetails']['city'] = $editedBillingDetails['city'] ?? '';
                            $bookingData['booking_details']['billingDetails']['state'] = $editedBillingDetails['state'] ?? '';
                            $bookingData['booking_details']['billingDetails']['billing_country'] = $editedBillingDetails['country'] ?? '';
                            $bookingData['booking_details']['billingDetails']['zipcode'] = $editedBillingDetails['zipcode'] ?? '';

                            $bookingData['booking_details']['booking_details']['billingDetails']['address'] = $editedBillingDetails['address'] ?? '';
                            $bookingData['booking_details']['booking_details']['billingDetails']['address1'] = $editedBillingDetails['address1'] ?? '';
                            $bookingData['booking_details']['booking_details']['billingDetails']['city'] = $editedBillingDetails['city'] ?? '';
                            $bookingData['booking_details']['booking_details']['billingDetails']['state'] = $editedBillingDetails['state'] ?? '';
                            $bookingData['booking_details']['booking_details']['billingDetails']['billing_country'] = $editedBillingDetails['country'] ?? '';
                            $bookingData['booking_details']['booking_details']['billingDetails']['zipcode'] = $editedBillingDetails['zipcode'] ?? '';


                            // Update Edited Passenger Details to Booking Details Json Passenger Details Array
                            if (
                                isset($bookingData['booking_details']['booking_details']['passengerDetails']) &&
                                is_array($bookingData['booking_details']['booking_details']['passengerDetails']) &&
                                is_array($editedPassengerDetails)
                            ) {
                                foreach ($bookingData['booking_details']['booking_details']['passengerDetails'] as &$passenger) {
                                    foreach ($editedPassengerDetails as $edited) {
                                        if (isset($edited['id']) && isset($passenger['id']) && $edited['id'] === $passenger['id']) {
                                            foreach ($edited as $field => $value) {
                                                $passenger[$field] = $value;
                                            }
                                            break;
                                        }
                                    }
                                }
                                unset($passenger);
                            }


                            // Update Edited Hotel Details to Booking Details Json Hotel Details Array
                            if (isset($bookingData['booking_details']['hotelDetails']['hotelDetails'])) {
                                $bookingData['booking_details']['hotelDetails']['hotelDetails']['DisplayName'] =
                                    $editedHotelDetails['hotelDetails']['DisplayName'] ?? '';
                                $bookingData['description'] =  $editedHotelDetails['hotelDetails']['DisplayName'] ?? '';
                            }



                            if (isset($bookingData['booking_details']['searchDetails'])) {

                                $bookingData['booking_details']['searchDetails']['checkInDate'] = $editedSearchDetails['checkInDate'] ?? '';
                                $bookingData['booking_details']['searchDetails']['checkOutDate'] = $editedSearchDetails['checkOutDate'] ?? '';
                            }
                        }
                        $bookingData['sub_total'] = $editedProcessedPrice['totalPrice'] ?? '';
                        $bookingData['tax'] = $editedProcessedPrice['vat'] ?? '';
                        $bookingData['s_markup_fee'] = $editedProcessedPrice['markupFee'] ?? '';
                        $bookingData['s_service_fee'] = $editedProcessedPrice['serviceFee'] ?? '';
                        $bookingData['s_markup_service_fee'] = $editedProcessedPrice['markupServiceFee'] ?? '';
                        $bookingData['s_tax'] = $editedProcessedPrice['totalTaxAmount'] ?? '';
                        $bookingData['chargeable_seat_price'] = $editedProcessedPrice['chargeableSeatPrice'] ?? '';
                        $bookingData['t_markup'] = $editedProcessedPrice['markupValue'] ?? '';
                        $bookingData['total'] = $editedProcessedPrice['grandTotal'] ?? '';


                        $bookingData['billing_address1'] = $editedBillingDetails['address'] ?? '';
                        $bookingData['billing_address2'] = $editedBillingDetails['address1'] ?? '';
                        $bookingData['billing_city'] = $editedBillingDetails['city'] ?? '';
                        $bookingData['billing_state'] = $editedBillingDetails['state'] ?? '';
                        $bookingData['billing_country'] = $editedBillingDetails['country'] ?? '';
                        $bookingData['billing_zip'] = $editedBillingDetails['zipcode'] ?? '';

                        if (!empty($completeFlightDetails)) {

                            $bookingData['booking_details']['booking_details']['flightDetails'] = $completeFlightDetails;
                            $bookingData['booking_details']['booking_details']['processedPrice'] = $editedProcessedPrice;
                        }
                    }

                    if ($locale == Locale::Arabic->value) {

                        if ($bookingData['booking_status'] == BookingStatus::Confirmed->value) {
                            $bookingData['booking_status_ar'] = 'مؤكد';
                        }
                        if ($bookingData['booking_status'] == BookingStatus::Failed->value) {
                            $bookingData['booking_status_ar'] = 'فشل';
                        }
                        if ($bookingData['booking_status'] == BookingStatus::Pending->value) {
                            $bookingData['booking_status_ar'] = 'قيد الانتظار';
                        }
                    }
                    $pnrDetails = json_decode($additionalbookingDetails->service_provider_booking_response, true) ?? [];
                    $airlinePnr = $gdsPnr = [];
                    if (!empty($pnrDetails)) {
                        $bookingData['pnrDetails'] =  $pnrDetails['data']['associatedRecords'] ?? [];
                        foreach ($bookingData['pnrDetails'] as $record) {
                            if ($record['originSystemCode'] === 'GDS') {
                                $gdsPnr[] = $record['reference'];
                            } else {
                                $airlinePnr[] = $record['reference'];
                            }
                        }
                        $bookingData['airlinePnr'] = $airlinePnr;
                        $bookingData['gdsPnr'] = $gdsPnr;
                    }

                    if ($bookingData['booking_type'] == ServiceType::Flight->value) {
                        $searchDetails = $bookingData['booking_details']['booking_details']['searchDetails'];
                        if (isset($searchDetails)) {
                            $userBookedlang = $searchDetails['languageCode'] ?? 'en';
                            if ($userBookedlang != $locale) {
                                $bookingData = TranslateBookingDetails::translateFlightBookingDetails($bookingData, $locale);
                            }
                        }
                    } else {
                        $bookingDetails = $bookingData['booking_details']['bookingDetails'];
                        if (isset($bookingDetails)) {
                            $userBookedlang = $bookingDetails['customer_language_code'] ?? 'en';
                            if ($userBookedlang != $locale) {
                                $bookingData = TranslateBookingDetails::translateHotelBookingDetails($bookingData, $locale);
                            }
                        }
                    }

                    return $this->sendResponse($bookingData, 'Booking Details Successfully!', $success);
                } else {
                    $success = [];
                    return $this->sendError(no_record_found, $success, 200);
                }
            } else {
                return $this->sendError(no_record_found, [], 200);
            }
        } catch (Exception $e) {
            return $this->sendError('Something went wrong. ' . $e->getMessage(), ['error' => $e . $e->getMessage()], 500);
        }
    }

    /* Process Booking starts here */
    public function preAuthorizeBookingPayment($paymentDetails)
    {
        $message = '';
        $successPattern = '/^(000\.000\.|000\.100\.1|000\.[36])/';
        $successManualReviewCodePattern = '/^(000\.400\.0|000\.400\.100)/';

        $paymentId = $paymentDetails['paymentId'];
        $paymentBrand = $paymentDetails['paymentBrand'];
        $paymentData = $paymentDetails['paymentData'];

        $preAuthorizeStatus = false;
        $preAuthorizeData = '';
        $preAuthorizeResponse = $this->getPaymentStatus($paymentBrand, $paymentId);
        if ($preAuthorizeResponse['status'] == 'true') {
            $preAuthorizeData = $preAuthorizeResponse['data'];
            $preAuthResultCode = $preAuthorizeData['result']['code'];
            if ($preAuthResultCode != '200.300.404' && (preg_match($successPattern, $preAuthResultCode) || preg_match($successManualReviewCodePattern, $preAuthResultCode) || $preAuthResultCode == '000.200.100')) {
                $preAuthorizeStatus = true;
            } else {
                $message = $preAuthorizeData['result']['description'];
            }
        } else {
            $message = $preAuthorizeResponse['message'];
        }
        $paymentData = $this->updatePaymentData('pre_authorization', $preAuthorizeStatus, $paymentData, $preAuthorizeData, true);

        return [
            'message' => $message,
            'paymentData' => $paymentData,
            'preAuthorizeStatus' => $preAuthorizeStatus,
            'preAuthorizeData' => $preAuthorizeData
        ];
    }

    public function captureBookingPayment($paymentDetails, $preAuthorizeData)
    {
        $message = '';
        $successPattern = '/^(000\.000\.|000\.100\.1|000\.[36])/';
        $successManualReviewCodePattern = '/^(000\.400\.0|000\.400\.100)/';

        $paymentId = $paymentDetails['paymentId'];
        $paymentBrand = $paymentDetails['paymentBrand'];
        $paymentData = $paymentDetails['paymentData'];

        $preAuthId = $preAuthorizeData['id'];
        $amount = $preAuthorizeData['amount'];
        $currency = $preAuthorizeData['currency'];

        $captureStatus = false;
        $captureResponseData = '';
        $captureResponse = $this->capturePayment($preAuthId, $paymentBrand, $amount, $currency);
        if ($captureResponse['status'] == 'true') {
            $captureResponseData = $captureResponse['data'];
            $captureResultCode = $captureResponseData['result']['code'];
            if ($captureResultCode != '200.300.404' && (preg_match($successPattern, $captureResultCode) || preg_match($successManualReviewCodePattern, $captureResultCode) || $captureResultCode == '000.200.100')) {
                $captureStatus = true;
            } else {
                $message = $captureResponse['data']['result']['description'];
            }
        } else {
            $message = $captureResponse['message'];
        }
        $paymentData = $this->updatePaymentData('capture', $captureStatus, $paymentData, $captureResponseData);
        if ($captureStatus == false) {
            $revertStatus = false;
            $revertResponseData = '';
            $revertResponse = $this->revertPayment($preAuthId, $paymentBrand);
            if ($revertResponse['status'] == 'true') {
                $revertStatus = true;
                $revertResponseData = $revertResponse['data'];
            }
            $paymentData = $this->updatePaymentData('revert', $revertStatus, $paymentData, $revertResponseData);
        }

        return [
            'message' => $message,
            'paymentData' => $paymentData,
            'captureStatus' => $captureStatus
        ];
    }

    public function debitBookingPayment($paymentDetails)
    {
        $message = '';
        $successPattern = '/^(000\.000\.|000\.100\.1|000\.[36])/';
        $successManualReviewCodePattern = '/^(000\.400\.0|000\.400\.100)/';

        $paymentId = $paymentDetails['paymentId'];
        $paymentData = $paymentDetails['paymentData'];
        $paymentBrand = $paymentDetails['paymentBrand'];

        $debitStatus = false;
        $debitResponseData = '';
        $debitResponse = $this->getPaymentStatus($paymentBrand, $paymentId);
        if ($debitResponse['status'] == 'true') {
            $debitResponseData = $debitResponse['data'];
            $debitResultCode = $debitResponseData['result']['code'];
            if ($debitResultCode != '200.300.404' && (preg_match($successPattern, $debitResultCode) || preg_match($successManualReviewCodePattern, $debitResultCode) || $debitResultCode == '000.200.100')) {
                $debitStatus = true;
            } else {
                $message = $debitResponseData['result']['description'];
            }
        }
        $paymentData = $this->updatePaymentData('debit', $debitStatus, $paymentData, $debitResponseData, true);

        return [
            'message' => $message,
            'paymentData' => $paymentData,
            'debitStatus' => $debitStatus
        ];
    }

    public function revertBookingPayment($paymentId, $paymentBrand, $paymentData)
    {
        $message = '';

        $revertStatus = false;
        $revertResponseData = '';
        $revertResponse = $this->revertPayment($paymentId, $paymentBrand);
        if ($revertResponse['status'] == 'true') {
            $revertStatus = true;
            $revertResponseData = $revertResponse['data'];
        } else {
            $message = $revertResponse['message'];
        }
        $paymentData = $this->updatePaymentData('revert', $revertStatus, $paymentData, $revertResponseData);

        return [
            'message' => $message,
            'paymentData' => $paymentData,
            'revertStatus' => $revertStatus
        ];
    }


    public function processPayment($type, $paymentData, $paymentResponseData = [])
    {
        $result = [
            'status' => false,
            'data' => [],
            'message' => ''
        ];

        if ($type == 'PA' || $type == 'DB' || $type == 'CP') {
            $paymentPayload = [
                'paymentId' => $paymentData['payment_id'],
                'paymentBrand' => $paymentData['payment_brand'],
                'paymentData' => $paymentData,
                'paymentSuccess' => false
            ];
        }
        switch ($type) {
            case 'PA':
                $preAuthorizeBooking = $this->preAuthorizeBookingPayment($paymentPayload);
                $result = [
                    'status' => $preAuthorizeBooking['preAuthorizeStatus'],
                    'message' => $preAuthorizeBooking['message'],
                    'data' => $preAuthorizeBooking['preAuthorizeData'],
                    'paymentData' => $preAuthorizeBooking['paymentData']
                ];
                break;
            case 'DB':
                $debitBooking = $this->debitBookingPayment($paymentPayload);
                $result = [
                    'status' => $debitBooking['debitStatus'],
                    'message' => $debitBooking['message'],
                    'data' => ['id' => $debitBooking['paymentData']['debit_id']],
                    'paymentData' => $debitBooking['paymentData']
                ];
                break;
            case 'CP':
                $captureBooking = $this->captureBookingPayment($paymentPayload, $paymentResponseData);
                $result = [
                    'status' => $captureBooking['captureStatus'],
                    'message' => $captureBooking['message'],
                    'data' => [],
                    'paymentData' => $captureBooking['paymentData']
                ];
                break;
            case 'RV':
                $revertBooking = $this->revertBookingPayment($paymentResponseData['id'], $paymentData['payment_brand'], $paymentData);
                $result = [
                    'status' => $revertBooking['revertStatus'],
                    'message' => $revertBooking['message'],
                    'data' => [],
                    'paymentData' => $revertBooking['paymentData']
                ];
                break;
        }
        if ($type == 'PA' || $type == 'DB') {
            $paymentData = $this->processPaymentData($result['paymentData']);
            unset($paymentData['is_capture_payment']);
            $paymentSaveResponse = Payment::createPayment($paymentData);

            $result['paymentData']['id'] = $paymentSaveResponse['data']['id'];
        } elseif ($type == 'CP' || $type == 'RV') {
            $paymentData = $this->processPaymentData($result['paymentData']);
            $paymentSaveResponse = Payment::updatePayment($paymentData['id'], $paymentData);
        }
        return $result;
    }

    public function updateBookingDetails($id, $type, $data, $paymentSuccess = false)
    {
        $result = [
            'sub_status' => '',
            'message' => ''
        ];
        if ($type == 'payment') {
            $updateBookingPayload = [
                'booking_status' => ($data['status'] ? BookingStatus::Processing->value : BookingStatus::Failed->value),
                'booking_sub_status' => ($data['status'] ? PaymentStatus::Payment_Success->value : PaymentStatus::Payment_Failed->value),
                'booking_status_message' => $data['message']
            ];
            Bookings::updateBooking($id, $updateBookingPayload);

            $result['status'] = $updateBookingPayload['booking_status'];
            $result['sub_status'] = $updateBookingPayload['booking_sub_status'];
            $result['message'] = $updateBookingPayload['booking_status_message'];
        } elseif ($type == 'payment-and-supplier') {
            $updateBookingPayload = [
                'booking_status' => (($paymentSuccess) ? BookingStatus::Confirmed->value : BookingStatus::Failed->value),
                'booking_sub_status' => (($paymentSuccess) ? BookingStatus::Supplier_Success->value : BookingStatus::Supplier_Success_Payment_Failed->value),
                'booking_status_message' => (($paymentSuccess) ? '' : $data['message']),
            ];
            Bookings::updateBooking($id, $updateBookingPayload);

            $result['status'] = $updateBookingPayload['booking_status'];
            $result['sub_status'] = $updateBookingPayload['booking_sub_status'];
            $result['message'] = $updateBookingPayload['booking_status_message'];
        } elseif ($type == 'revert-payment') {
            $updateBookingPayload = [
                'booking_status' => BookingStatus::Failed->value,
                'booking_sub_status' => (($data['status']) ? BookingStatus::Booking_Failed_Payment_Reversed->value : BookingStatus::Booking_Failed_Payment_Not_Reversed->value),
                'booking_status_message' => $data['message'],
            ];
            Bookings::updateBooking($id, $updateBookingPayload);

            $result['status'] = $updateBookingPayload['booking_status'];
            $result['sub_status'] = $updateBookingPayload['booking_sub_status'];
            $result['message'] = $updateBookingPayload['booking_status_message'];
        }
        return $result;
    }

    public function createSupplierOrder($referenceDetails, $bookingDetails)
    {
        $locale = $referenceDetails['booking_details']['bookingDetails']['customer_language_code'];
        $createBookingResponse = [
            'status' => 'false',
            'message' => (($locale == Locale::Arabic->value) ? 'غير قادر على معالجة الحجز.' : 'Unable to process booking.')
        ];
        $orderCreateLogName = $supplierBookingReference = $bookingSegmentId = $supplierName = '';
        $createBookingPayload = $createBookingResponse = [];
        switch ($referenceDetails['booking_type']) {
            case ServiceType::Flight->value:
                $orderCreateLogName = 'flight-order';
                $supplierName = 'Amadeus';
                $amaClientRef = '';
                $createBookingPayload = $this->generateFlightBookingPayload($referenceDetails['booking_details']);
                $createBookingResponse = $this->flightNewOrderCreate($createBookingPayload, $amaClientRef);
                if ($createBookingResponse['status'] == 'true') {
                    // Add Original Flight Offers to Amadeus Booking Response
                    //$createBookingResponse = $this->addOriginalFlightOffersToBooking($referenceDetails['booking_details'], $createBookingResponse);
                    $supplierBookingReference = $createBookingResponse['data']['id'];
                } else {
                    $errorMessages = json_decode($createBookingResponse['message'], true);
                    $tempErrorMessages = [];
                    if (is_array($errorMessages)) {
                        foreach ($errorMessages as $key => $errorMessage) {
                            $tempMessage = '';
                            if (isset($errorMessage['title'])) {
                                $tempMessage = $errorMessage['title'];
                            }
                            if (isset($errorMessage['detail'])) {
                                $tempMessage = (($tempMessage != '') ? ': ' : '') . $errorMessage['detail'];
                            }
                            array_push($tempErrorMessages, $tempMessage);
                        }
                    }
                    if (count($tempErrorMessages) > 0) {
                        $createBookingResponse['message'] = implode(', ', $tempErrorMessages);
                    } else {
                        $createBookingResponse['message'] = $errorMessages;
                    }
                }
                break;
            case ServiceType::Hotel->value:
                if ($referenceDetails['service_provider'] == 'HOTEL_BEDS') {
                    $createBookingPayload = $this->generateHotelBookingPayload($bookingDetails);
                    $createBookingResponse = $this->hotelNewOrderCreate($createBookingPayload);
                    if ($createBookingResponse['status'] == 'true') {
                        $supplierBookingReference = $createBookingResponse['data']['reference'];
                    } else {
                        $errorMessageArr = [];
                        if (isset($createBookingResponse['message']['code'])) {
                            array_push($errorMessageArr, $createBookingResponse['message']['code']);
                        }
                        if (isset($createBookingResponse['message']['message'])) {
                            array_push($errorMessageArr, $createBookingResponse['message']['message']);
                        }
                        if (count($errorMessageArr) > 0) {
                            $errorMessage = implode(":", $errorMessageArr);
                        } else {
                            $errorMessage = (($locale == Locale::Arabic->value) ? 'غير قادر على إنشاء قسيمة الفندق.' : 'Unable to create hotel voucher.');
                        }
                        $createBookingResponse['message'] = $errorMessage;
                    }
                } elseif ($referenceDetails['service_provider'] == 'GIMMONIX') {
                    $createBookingPayload = $this->generateGimmonixBookingPayload($referenceDetails);
                    $createBookingResponse = $this->createGimmonixOrder($createBookingPayload);
                    if ($createBookingResponse['status'] == true) {
                        $supplierBookingReference = $createBookingResponse['data']['HotelSegments'][0]['BookingReference'];
                        $bookingSegmentId = $createBookingResponse['data']['HotelSegments'][0]['SegmentId'];
                    } else {
                        $errorMessageArr = [];
                        if (is_array($createBookingResponse['message']) && count($createBookingResponse['message']) > 0) {
                            foreach ($createBookingResponse['message'] as $error) {
                                $tempError = [];
                                if (array_key_exists('ErrorCode', $error)) {
                                    array_push($tempError, $error['ErrorCode']);
                                }
                                if (array_key_exists('ErrorText', $error)) {
                                    array_push($tempError, $error['ErrorText']);
                                }
                                if (array_key_exists('Message', $error)) {
                                    array_push($tempError, $error['Message']);
                                }
                                $errorMessage = implode(':', $tempError);
                                array_push($errorMessageArr, $errorMessage);
                            }
                        }
                        if (count($errorMessageArr) > 0) {
                            $errorMessage = implode(",", $errorMessageArr);
                        } else {
                            $errorMessage = (($locale == Locale::Arabic->value) ? 'غير قادر على إنشاء قسيمة الفندق.' : 'Unable to create hotel voucher.');
                        }
                        $createBookingResponse['message'] = $errorMessage;
                    }
                }
                break;
        }
        $additionalDetails = [
            'booking_id' => $bookingDetails['id'],
            'service_provider_booking_request' => json_encode($createBookingPayload),
            'service_provider_booking_response' => json_encode($createBookingResponse)
        ];
        BookingAdditionalDetail::updateBookingAdditionalDetail($additionalDetails);

        $createOrderSupplierStatus = ($createBookingResponse['status'] == 'true');
        $supplierLogPayload = [
            'ama_client_ref' => ($bookingDetails['booking_type'] == ServiceType::Flight->value) ? $amaClientRef : '',
            'booking_id' => $bookingDetails['id'],
            'log_parent_id' => $referenceDetails['log_id'],
            'type' => $bookingDetails['booking_type'],
            'supplier' => $supplierName,
            'supplier_reference' => ($supplierBookingReference != '') ? $supplierBookingReference : BookingStatus::Failed->value,
            'log_type' => $orderCreateLogName,
            'customer_id' => $referenceDetails['customer_id'],
            'request' => json_encode($createBookingPayload),
            'response' => json_encode($createBookingResponse)
        ];
        $this->createSupplierLogForBooking($supplierLogPayload);

        // Update order status
        $updateBookingPayload = [
            'booking_status' => ($createOrderSupplierStatus ? BookingStatus::Confirmed->value : BookingStatus::Failed->value),
            'booking_sub_status' => ($createOrderSupplierStatus ? 'Supplier success' : 'Supplier failed'),
            'booking_status_message' => $createBookingResponse['message']
        ];
        if ($createOrderSupplierStatus) {
            $updateBookingPayload = $this->getUpdateOrderPayload($updateBookingPayload, $referenceDetails, $createBookingResponse['data'], $supplierBookingReference, $bookingSegmentId);
        }

        Bookings::updateBooking($bookingDetails['id'], $updateBookingPayload);


        /*if ($bookingDetails['booking_type'] === ServiceType::Flight->value && $createBookingResponse['status'] != 'false') {
            $createBookingIssuanceResponse = $this->flightNewOrderIssuanceCreate($supplierBookingReference, $amaClientRef);
            $this->getAndUpdateAmadeusBookingOrderIssuance(
                $amaClientRef,
                $bookingDetails['id'],
                $referenceDetails['customer_id'],
                $referenceDetails['log_id'],
                $supplierBookingReference,
                $supplierName,
                $createBookingIssuanceResponse
            );
        }*/

        return [
            'status' => $createOrderSupplierStatus,
            'message' => $createBookingResponse['message']
        ];
    }

    public function addCustomerBookingPoints($bookingDetails, $referenceDetails)
    {
        $data = [
            'booking_id' => $bookingDetails['id'],
            'customer_id' => $referenceDetails['customer_id'],
            'points_earned' => $bookingDetails['booking_points']
        ];

        if (isset($bookingDetails['paymentDetails']['redeem_details'])) {
            $data['points_redeemed'] = $referenceDetails['booking_details']['paymentDetails']['redeem']['points'];
            $data['redeemed_amount'] = $referenceDetails['booking_details']['paymentDetails']['redeem']['amount'];
        }

        CustomerLoyaltyPoints::createCustomerLoyaltyPoints($data);
    }

    public function updateCustomerCouponCode($bookingId, $referenceDetails)
    {
        if (isset($referenceDetails['booking_details']['paymentDetails']['coupon_details']['status'])) {
            $couponDetails = Coupon::whereId($referenceDetails['booking_details']['paymentDetails']['coupon_details']['id'])->first()->toArray();

            if ($couponDetails) {
                $updateBookingPayload = [
                    'coupon_code' => $referenceDetails['booking_details']['paymentDetails']['coupon_details']['code']
                ];
                Bookings::updateBooking($bookingId, $updateBookingPayload);

                if ($couponDetails['redeem_count'] == '') {
                    $couponDetails['redeem_count'] = 0;
                }
                $redeemCount = $couponDetails['redeem_count'] + 1;
                Coupon::updateCouponData($couponDetails['id'], ['redeem_count' => $redeemCount]);
            }
        }
    }

    public function createBookingChildren($bookingId, $referenceDetails)
    {
        $savedBookingDetails = Bookings::where('id', $bookingId)->first()->toArray();
        if ($referenceDetails['customer_id'] != '') {
            $this->addCustomerBookingPoints($savedBookingDetails, $referenceDetails);
            //$this->updateCustomerCouponCode($savedBookingDetails['id'], $referenceDetails);
        }
        if ($referenceDetails['booking_type'] == ServiceType::Hotel->value) {
            $traacsResponse = $this->createVoucherInTraacs($bookingId);
        }
        $this->sendBookingEmail($savedBookingDetails);
        $this->sendBookingSms($savedBookingDetails);
        $this->sendBookingWhatsapp($savedBookingDetails);
    }

    public function processBookingInitialResult($request)
    {
        $requestData =  $request->all();
        $tempRequestData = [
            'ref_id' => $requestData['ref_id'],
            'payment_id' => $requestData['id'],
            'payment_brand' => $requestData['payment_brand'],
            'booking_via' => $request->input('request_from', 'web'),
            'booking_type' => '',
        ];
        $result = array_merge($tempRequestData, [
            'status' => BookingStatus::Failed->value,
            'sub_status' => '',
            'message' => '',
            'booking_ref_id' => ''
        ]);
        return $result;
    }

    public function processBooking(Request $request)
    {
        $result = $this->processBookingInitialResult($request);

        // Get Reference Details
        $referenceDetailsResponse = $this->getReferenceDetails($result['ref_id']);
        if ($referenceDetailsResponse['status'] == true) {
            $result['booking_type'] = $referenceDetailsResponse['data']['booking_type'];
            $referenceDetails = array_merge($referenceDetailsResponse['data'], $result);
            $result['language'] = $referenceDetails['booking_details']['bookingDetails']['customer_language_code'];

            // Create / Update Booking
            $createBookingResponse = $this->createOrUpdateBooking($referenceDetails);

            if ($createBookingResponse['status'] == true) {
                $bookingDetails = $createBookingResponse['data'];

                // Save Travelers
                if ($createBookingResponse['type'] == 'create') {
                    $this->saveBookingTravelers($referenceDetails, $bookingDetails);
                }

                // Payment Details
                $paymentData = $this->getPaymentData($referenceDetails['amount'], $result['payment_brand'], $result['payment_id'], $bookingDetails['id']);
                $type = ($paymentData['is_capture_payment'] ? 'PA' : 'DB');
                $paymentResponse = $this->processPayment($type, $paymentData);
                $paymentData = $paymentResponse['paymentData'];
                $paymentSuccess = $paymentResponse['status'];

                // Update Booking for Payment Response
                $updateResponse = $this->updateBookingDetails($bookingDetails['id'], 'payment', $paymentResponse);
                $result['status'] = $updateResponse['status'];
                $result['sub_status'] = $updateResponse['sub_status'];
                $result['message'] = $updateResponse['message'];

                // If payment response is success
                if ($paymentSuccess) {
                    // Create supplier order
                    $supplierBookingResponse = $this->createSupplierOrder($referenceDetails, $bookingDetails);

                    if ($supplierBookingResponse['status']) {
                        // Capture payment
                        if ($paymentData['is_capture_payment']) {
                            $capturePaymentResponse = $this->processPayment('CP', $paymentData, $paymentResponse['data']);
                            $paymentSuccess = $capturePaymentResponse['status'];
                        }
                        // Update booking final status confirmed or failed
                        $updateResponse = $this->updateBookingDetails($bookingDetails['id'], 'payment-and-supplier', $paymentResponse, $paymentSuccess);
                        $result['status'] = $updateResponse['status'];
                        $result['sub_status'] = $updateResponse['sub_status'];
                        $result['message'] = $updateResponse['message'];

                        // If order is confirmed, call booking related updates like loyalty poitns, traccs, email, sms
                        $walletUsages = [];
                        if ($paymentSuccess) {

                            $result['status'] = BookingStatus::Confirmed->value;
                            $result['booking_ref_id'] = $bookingDetails['booking_ref'];
                            $this->createBookingChildren($bookingDetails['id'], $referenceDetails);
                            $walletUsages = $referenceDetails['booking_details']['processedPrice'] ?? [];
                            if (!empty($walletUsages)) {
                                foreach ($walletUsages as $walletUsage) {
                                    if (!empty($walletUsage['wallet_id']) && isset($walletUsage['original_used'])) {
                                        $walletDetails = Wallet::find($walletUsage['wallet_id']);

                                        if ($walletDetails) {
                                            $originalUsed = floatval($walletUsage['original_used']);
                                            $walletDetails->type = 'debit';
                                            $walletDetails->wallet_used_status = 'used';
                                            $walletDetails->balance = max(0, $walletDetails->balance - $originalUsed);

                                            $walletDetails->save();
                                        }
                                    }
                                }
                            }
                        }
                    } elseif ($supplierBookingResponse['status'] == false && $paymentData['is_capture_payment'] == false) {
                        // If order is failed, revert the payment
                        $revertPaymentResponse = $this->processPayment('RV', $paymentData, $paymentResponse['data']);
                        $updateResponse = $this->updateBookingDetails($bookingDetails['id'], 'revert-payment', $revertPaymentResponse);
                        $result['status'] = $updateResponse['status'];
                        $result['sub_status'] = $updateResponse['sub_status'];
                        $result['message'] = $updateResponse['message'];
                        $walletUsages = $referenceDetails['booking_details']['processedPrice'] ?? [];
                        if (!empty($walletUsages)) {
                            foreach ($walletUsages as $walletUsage) {
                                if (!empty($walletUsage['wallet_id'])) {
                                    Wallet::where('id', $walletUsage['wallet_id'])->update([
                                        'wallet_used_status' => null,
                                        'used_temp_amount' => null,
                                        'temp_amount_currency' => null,
                                        'converted_used_amount' => null,
                                    ]);
                                }
                            }
                        }
                    } else {
                        $result['message'] = $supplierBookingResponse['message'];
                    }
                }
            }
        }

        // Get return result based on web or app
        $returnResult = $this->processReturnResult($result);
        if ($result['booking_via'] == 'mobile') {
            return $this->sendResponse($returnResult['data'], $returnResult['message'], $returnResult['success']);
        } else {
            return redirect()->to($returnResult);
        }
    }
    /* Process booking ends here */



    public function resendEmailBookingDetails(Request $request)
    {
        try {
            $result = [
                'success' => false,
                'message' => 'Unable to find booking details.',
            ];
            $requestData =  $request->all();
            $bookingDetails = Bookings::where('booking_ref', $requestData['reference'])
                ->where('email', $requestData['email'])
                ->first()
                ->toArray();

            if ($bookingDetails) {
                $result = $this->sendBookingEmail($bookingDetails);
            }
            return $this->sendResponse([], $result['message'], $result['success']);
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }



    public function getCustomerRehltePointsHistory(Request $request)
    {
        $currency = $request->input('currency', Currency::SAR->value);
        $userId = auth()->id();

        if (!$userId) {
            return $this->sendError([], 'User not authenticated', 401);
        }

        $history = CustomerLoyaltyPoints::getRehltePointsHistory($userId, $currency);

        if ($history['status'] == 0) {
            return $this->sendError([], $history['message'], 404);
        }

        return $this->sendResponse($history['data'], $history['message'], 200);
    }





    public function processBookingReference($referenceDetails, $bookingDetails, $paymentDetails)
    {
        return [
            'ref_id' => $referenceDetails['ref_id'],
            'booking_via' => $referenceDetails['booking_via'],
            'status' => $bookingDetails['booking_status'],
            'sub_status' => $bookingDetails['booking_sub_status'],
            'message' => $bookingDetails['booking_status_message'],
            'language' => $bookingDetails['customer_language_code'],
            'booking_type' => $referenceDetails['booking_type'],
            'booking_ref_id' => $bookingDetails['booking_ref'],
            'payment_brand' => $paymentDetails['payment_brand'],
        ];
    }

    public function processBooking3(Request $request)
    {
        $requestData =  $request->all();
        $id = $requestData['id'];
        $refId = $requestData['ref_id'];
        $referenceDetails = $this->getReferenceDetails($refId);
        if ($referenceDetails['status'] == 1) {
            $referenceDetails = $referenceDetails['data'];
            $bookingDetails = Bookings::whereRef_id($refId)->first();
            if ($bookingDetails) {
                $bookingDetails = $bookingDetails->toArray();
            }
            $paymentDetails = Payment::whereBooking_id($bookingDetails['id'])->first();
            if ($paymentDetails) {
                $paymentDetails = $paymentDetails->toArray();
            }
            $result = $this->processBookingReference($referenceDetails, $bookingDetails, $paymentDetails);

            // Get return result based on web or app
            $returnResult = $this->processReturnResult($result);
            if ($result['booking_via'] == 'web') {
                return redirect()->to($returnResult);
            } else {
                return $this->sendResponse($returnResult['data'], $returnResult['message'], $returnResult['success']);
            }
        } else {
            echo "Unable to process booking";
        }
    }

    public function processBookingPayment(Request $request)
    {
        $refId = $amount = $checkStatusUrl = $bookingVia = '';
        $paymentId = $request->input('id');
        $paymentBrand = $request->input('payment_brand');
        if ($paymentId != '' && $paymentBrand != '') {
            $paymentResponse = $this->getPaymentStatus($paymentBrand, $paymentId);
            if ($paymentResponse['status'] == true) {
                if (array_key_exists('customParameters', $paymentResponse['data'])) {
                    $refId = $paymentResponse['data']['customParameters']['SHOPPER_ref_id'];
                    $amount = $paymentResponse['data']['customParameters']['SHOPPER_amount'];
                }
            }
            if ($refId != '') {
                $referenceDetails = BookingCheckout::whereRef_id($refId)->first();
                $bookingVia = $referenceDetails->booking_via;
                $checkStatusUrl = $referenceDetails->checkout_url . '/payment/payment-success?ref_id=' . $refId;
            }

            $statusUrl = env('API_BASE_URL') . 'booking/get-booking-status?';
            $data = [
                'payment_id' => $paymentId,
                'ref_id' => $refId,
                'amount' => $amount,
                'status_url' => $statusUrl,
                'booking_via' => $bookingVia,
                'check_status_url' => $checkStatusUrl
            ];
            if ($bookingVia == 'web') {
                return redirect()->to($checkStatusUrl);
            } else {
                return view('admin.booking.payment')->with(['data' => $data]);
            }
        } else {
            return redirect()->to(env('WEBSITE_URL'));
        }
    }

    public function getLoyaltyPointExpiryDate($service_end_date)
    {
        $loyaltyPointSetting = LoyaltyPointSetting::first();
        if ($loyaltyPointSetting) {

            $expiry_year = $loyaltyPointSetting->expiry_year ?? 1;
            $expiry_month = $loyaltyPointSetting->expiry_month ?? 0;
            $expiry_day = $loyaltyPointSetting->expiry_day ?? 0;

            $checkoutDate = Carbon::parse($service_end_date);

            $addYears = $expiry_year;
            $addMonths = ($expiry_month > 0) ? $expiry_month  : 0;
            $addDays = ($expiry_day > 0) ? $expiry_day : 0;

            $expiryDate = $checkoutDate->copy()
                ->addYears($addYears)
                ->addMonths($addMonths)
                ->addDays($addDays)
                ->format('Y-m-d');
        } else {

            $expiry_year =  1;
            $expiry_month =  0;
            $expiry_day =  0;

            $checkoutDate = Carbon::parse($service_end_date);

            $addYears = $expiry_year;
            $addMonths = ($expiry_month > 0) ? $expiry_month  : 0;
            $addDays = ($expiry_day > 0) ? $expiry_day : 0;

            $expiryDate = $checkoutDate->copy()
                ->addYears($addYears)
                ->addMonths($addMonths)
                ->addDays($addDays)
                ->format('Y-m-d');
        }
        return $expiryDate;
    }

    public function getBookingStatus(Request $request)
    {
        $result = [
            'success' => false,
            'data' => [
                'status' => '',
                'url' => ''
            ],
            'message' => ''
        ];
        try {
            $requestData =  $request->all();
            $refId = $request->input('ref_id');
            if ($refId != '') {
                $result = [
                    'success' => true,
                    'data' => [
                        'status' => 'in_progress',
                        'url' => ''
                    ],
                    'message' => ''
                ];
                $referenceDetails = BookingCheckout::whereRef_id($refId)->first();
                if ($referenceDetails) {
                    $booking_details = json_decode($referenceDetails->booking_details, true);
                    $walletUsages = $booking_details['processedPrice']['walletUsage'] ?? [];
                    if ($referenceDetails) {
                        $bookingDetails = Bookings::whereRef_id($refId)->first();
                        if ($bookingDetails) {
                            $result['data']['email'] = $bookingDetails->email;
                            $result['data']['status'] = $bookingDetails->booking_status;
                            $result['data']['url'] = $referenceDetails->checkout_url;
                            $bookingType = strtolower($bookingDetails->booking_type);
                            $booking_details = json_decode($referenceDetails->booking_details, true);
                            $walletUsages = $booking_details['processedPrice']['walletUsage'] ?? [];
                            $redeemUsages = $booking_details['processedPrice']['loyaltyPointsUsage'] ?? [];
                            if ($bookingDetails->booking_status == BookingStatus::Confirmed->value) {
                                if ($referenceDetails['booking_type'] == ServiceType::Hotel->value && $referenceDetails['traacs_voucher_created'] == 0) {
                                    $traacsResponse = $this->createVoucherInTraacs($referenceDetails['booking_id']);
                                    if ($traacsResponse['status'] == true) {
                                        BookingCheckout::where('ref_id', $refId)->update(['traacs_voucher_created' => 1]);
                                        Bookings::where('ref_id', $refId)->update(['traacs_created' => 1]);
                                    }

                                    $hotelId = $booking_details['hotelDetails']['selectedRooms']['HotelId'] ?? '';
                                    if ($hotelId != '') {
                                        dispatch(new MostBookedHotelJob(
                                            $booking_details['hotelDetails']['selectedRooms']['HotelId'] ?? '',
                                            $booking_details['hotelDetails']['hotelDetails']['DisplayName'] ?? '',
                                            $referenceDetails['booking_id'] ?? ''
                                        ));
                                    }
                                }
                                // Wallet Update
                                if (!empty($walletUsages)) {
                                    // Get all wallet IDs to update
                                    $walletIds = collect($walletUsages)
                                        ->pluck('wallet_id')
                                        ->filter()
                                        ->toArray();

                                    $wallets = Wallet::whereIn('id', $walletIds)->get()->keyBy('id');

                                    $today = date('Y-m-d');

                                    foreach ($walletUsages as $walletUsage) {
                                        $walletId = $walletUsage['wallet_id'] ?? null;
                                        $originalUsed = isset($walletUsage['original_used']) ? floatval($walletUsage['original_used']) : 0;

                                        if ($walletId && isset($wallets[$walletId])) {
                                            $wallet = $wallets[$walletId];

                                            $wallet->type = 'debit';
                                            $wallet->wallet_used_status = 'used';
                                            $wallet->wallet_used_amount = $originalUsed;
                                            $wallet->used_date = $today;
                                            $wallet->save();
                                        }
                                    }
                                }


                                if ($bookingDetails->booking_type == ServiceType::Hotel->value) {
                                    $service_end_date = $booking_details['searchDetails']['stay']['checkOut'];
                                    $searchFrom = $booking_details['searchDetails']['originDevice'] ?? 'web';
                                    if ($searchFrom != 'web' && $bookingDetails->booking_processed == 0) {
                                        Bookings::where('ref_id', $refId)->update([
                                            'booking_from' => $searchFrom
                                        ]);
                                    }
                                }
                                if ($bookingDetails->booking_type == ServiceType::Flight->value) {
                                    $service_end_date_raw = $booking_details['flightDetails']['airlines'][0]['flights'][0]['segments'][0]['arrival']['at'] ?? null;

                                    $service_end_date = $service_end_date_raw
                                        ? Carbon::parse($service_end_date_raw)->format('Y-m-d')
                                        : null;
                                }

                                $expiryDate = $this->getLoyaltyPointExpiryDate($service_end_date);

                                CustomerLoyaltyPoints::where('booking_id', $bookingDetails->id)->update([
                                    'service_end_date' => $service_end_date,
                                    'expiry_date' => $expiryDate,
                                ]);
                                //Loyalty Point Update
                                if (!empty($redeemUsages)) {
                                    $redeemIds = collect($redeemUsages)->pluck('id')->filter()->toArray();
                                    if (!empty($redeemIds)) {
                                        CustomerLoyaltyPoints::whereIn('id', $redeemIds)->update([
                                            'used_status' => 'used',
                                            'service_end_date' => $service_end_date,
                                            'expiry_date' => $expiryDate,
                                        ]);
                                    }
                                }
                                $savedBookingDetails = Bookings::whereRef_id($refId)->first()->toArray();

                                if ($bookingDetails->booking_processed == 0) {
                                    if ($savedBookingDetails['email_notification'] == 0) {
                                        dispatch(new \App\Jobs\SendBookingEmailJob($savedBookingDetails, $refId));
                                        // $isEmailSent =  $this->sendBookingEmail($savedBookingDetails);
                                        // if ($isEmailSent['success'] == true) {
                                        // 	Bookings::where('ref_id', $refId)->update(['email_notification' => 1]);
                                        // }
                                    }

                                    if ($savedBookingDetails['sms_notification'] == 0) {
                                        dispatch(new \App\Jobs\SendBookingSmsJob($savedBookingDetails, $refId));
                                        // $isSmsSent = $this->sendBookingSms($savedBookingDetails);
                                        // if ($isSmsSent) {
                                        // 	Bookings::where('ref_id', $refId)->update(['sms_notification' => 1]);
                                        // }
                                    }
                                    if ($savedBookingDetails['whatsapp_notification'] == 0) {
                                        dispatch(new \App\Jobs\SendBookingWhatsappJob($savedBookingDetails, $refId));
                                        // $iswhatsappSent = $this->sendBookingWhatsapp($savedBookingDetails);
                                        // if ($iswhatsappSent) {
                                        // 	Bookings::where('ref_id', $refId)->update(['whatsapp_notification' => 1]);
                                        // }
                                    }
                                    Bookings::where('ref_id', $refId)->update(['booking_processed' => 1]);
                                }

                                if ($referenceDetails->booking_via == 'web-app') {
                                    $result['data']['url'] = str_replace(['/en', '/ar'], ['', ''], $result['data']['url']);
                                }
                                $result['data']['url'] .= '/booking/' . $bookingType . '-booking?booking_reference=' . $bookingDetails->booking_ref . '&email=' . $bookingDetails->email . '&reference=create';
                            } else {
                                $failurePage = '/payment';
                                if ($referenceDetails->booking_via == 'web-app' && $referenceDetails->booking_type == ServiceType::Flight->value) {
                                    $failurePage = '/flightCheckout';
                                    $result['data']['url'] = str_replace(['/en', '/ar'], ['', ''], $result['data']['url']);
                                } elseif ($referenceDetails->booking_via == 'web-app' && $referenceDetails->booking_type == ServiceType::Hotel->value) {
                                    $result['data']['url'] = str_replace(['/en', '/ar'], ['', ''], $result['data']['url']);
                                    $failurePage = '/hotelCheckout';
                                }
                                $result['data']['url'] .= $failurePage . '?ref_id=' . $refId . '&payment_brand=VISA&status=' . $bookingDetails->booking_status . '&sub_status=' . $bookingDetails->booking_sub_status . '&message=' . $bookingDetails->booking_status_description;
                            }
                        }
                    }
                }
            } else {
                $result['message'] = 'Invalid Reference';
            }
            return $this->sendResponse($result['data'], $result['message'], $result['success']);
        } catch (\Exception $e) {
            return $this->sendError([], 'Something went wrong' . $e->getFile() . $e->getLine(), ['error' => $e->getMessage()], 500);
        }
    }


    public function checkBookingExist(Request $request)
    {
        $result = [
            'status' => false,
            'message' => 'Booking not found',
            'data' => []
        ];
        $validated = Validator::make(
            $request->all(),
            [
                'ref_id' => 'required',
                'email' => 'required|email'
            ],
        );
        if ($validated->fails()) {
            return $this->sendError('Validation Error', [$validated->errors()], 200);
        }

        try {

            $validatedData = $validated->validated();

            $bookings = Bookings::where('booking_ref', $validatedData['ref_id'])
                ->where('email', $validatedData['email'])
                ->first();
            if ($bookings) {
                $result['status'] = true;
                $result['message'] = 'Booking found';
                $result['data'] = [
                    'booking_id' => $bookings->id,
                    'booking_ref' => $bookings->booking_ref,
                    'first_name' => $bookings->first_name,
                    'last_name' => $bookings->last_name,
                    'email' => $bookings->email,
                    'phone' => $bookings->phone,
                    'booking_type' => $bookings->booking_type,
                    'service_provider' => $bookings->service_provider
                ];

                return $this->sendResponse($result, '', 200);
            }
            return $this->sendResponse($result, '', 200);
        } catch (\Exception $e) {
            return $this->sendError([], 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    public function getPaymentStatusByPaymentId(Request $request)
    {
        $result = [
            'success' => false,
            'message' => 'Status not found',
            'data' => []
        ];

        $refId = $amount = '';
        $paymentId = $request->input('id');
        $paymentBrand = $request->input('payment_brand');
        if ($paymentId != '' && $paymentBrand != '') {
            $paymentResponse = $this->getPaymentStatus($paymentBrand, $paymentId);
            if ($paymentResponse['status'] == true) {
                if (array_key_exists('customParameters', $paymentResponse['data'])) {
                    $refId = $paymentResponse['data']['customParameters']['SHOPPER_ref_id'];
                    $amount = $paymentResponse['data']['customParameters']['SHOPPER_amount'];
                    $result['success'] = true;
                }
            }

            $result['data'] = [
                'payment_id' => $paymentId,
                'ref_id' => $refId,
                'amount' => $amount
            ];
        }
        return $this->sendResponse($result['data'], $result['message'], $result['success']);
    }

    public function getBookingStatusByRefId(Request $request)
    {
        $result = [
            'success' => false,
            'message' => 'Booking not found',
            'data' => []
        ];
        $refId = $request->input('ref_id');
        $bookings = Bookings::where('ref_id', $refId)->first();
        if ($bookings) {
            $result['success'] = true;
            $result['message'] = 'Booking found';
            $result['data'] = [
                'booking_id' => $bookings->id,
                'booking_ref' => $bookings->booking_ref,
                'booking_status' => $bookings->booking_status,
                'booking_sub_status' => $bookings->booking_sub_status,
                'booking_status_message' => $bookings->booking_status_message,
                'booking_status_description' => $bookings->booking_status_description
            ];

            return $this->sendResponse($result['data'], $result['message'], $result['success']);
        }
        return $this->sendResponse($result['data'], $result['message'], $result['success']);
    }

    /**
     * For Testing purpose
     */

    // public function testWhatsapp(Request $request)
    // {
    //     $savedBookingDetails = Bookings::whereRef_id($request->input('ref_id'))->first()->toArray();


    //     return $this->sendBookingWhatsapp($savedBookingDetails);
    // }
    public function testEmailService(Request $request)
    {
        $request->validate([
            'ref_id' => 'required',
        ]);
        $savedBookingDetails = Bookings::whereRef_id($request->input('ref_id'))->first()->toArray();
        dispatch(new \App\Jobs\SendBookingEmailJob($savedBookingDetails, $request->input('ref_id')));
        return $this->sendResponse([], 'Booking Email Sent Successfully', true);
        // return $this->sendBookingEmail($savedBookingDetails);
    }
    public function testWhatsappService(Request $request)
    {
        $request->validate([
            'ref_id' => 'required',
        ]);
        $savedBookingDetails = Bookings::whereRef_id($request->input('ref_id'))->first()->toArray();
        // dispatch(new \App\Jobs\SendBookingWhatsappJob($savedBookingDetails, $request->input('ref_id')));
        // return $this->sendBookingWhatsapp($savedBookingDetails);
        return $this->sendResponse([], 'Booking Email Sent Successfully', true);

        return $this->sendBookingWhatsapp($savedBookingDetails);
    }

    /**
     * Delete booking checkouts older than 7 days that do not have a booking ID.
     */
    public function deleteBookingCheckout()
    {
        $dateBefore = Carbon::today()->subDays('7')->format('Y-m-d H:i:s');
        $bookingDeleted =   BookingCheckout::where('created_at', '<=', $dateBefore)
            ->where(function ($query) {
                $query->whereNull('booking_id')
                    ->orWhere('booking_id', '');
            })
            ->delete();

        if ($bookingDeleted == 0) {
            return $this->sendResponse('No Records to delete', $bookingDeleted, 200);
        }
        return $this->sendResponse('Booking checkouts deleted successfully', $bookingDeleted, 200);
    }

    /**
     * Delete supplier logs older than 7 days that do not have a booking ID.
     */

    public function deleteSupplierLog()
    {
        $dateBefore = Carbon::today()->subDays(7)->format('Y-m-d H:i:s');

        $supplierLogDeleted = SupplierLog::where('created_at', '<=', $dateBefore)
            ->where(function ($query) {
                $query->whereNull('booking_id')
                    ->orWhere('booking_id', '');
            })
            ->delete();

        if ($supplierLogDeleted == 0) {
            return $this->sendResponse('No Records to delete', $supplierLogDeleted, 200);
        }
        return $this->sendResponse('Supplier Log deleted successfully', $supplierLogDeleted, 200);
    }
}
