<?php

namespace App\Http\Controllers\API\V1;

use Exception;
use Carbon\Carbon;
use App\Models\Page;
use App\Enums\Locale;
use App\Models\Coupon;
use App\Models\Wallet;
use App\Enums\Currency;
use App\Models\Airline;
use App\Models\Payment;
use App\Models\Setting;
use App\Models\Bookings;
use App\Traits\Gimmonix;
use App\Models\PageI18ns;
use App\Traits\ActiveLog;
use App\Enums\ServiceType;
use App\Models\StateI18ns;
use App\Traits\SmsService;
use App\Models\SupplierLog;
use App\Traits\EditBooking;
use App\Enums\BookingStatus;
use App\Enums\PaymentStatus;
use App\Models\HotelAddress;
use App\Traits\EmailService;
use Illuminate\Http\Request;
use App\Traits\BookingHelper;
use App\Traits\CommonService;
use App\Traits\TraacsService;
use App\Models\BookingHistory;
use App\Traits\AmadeusService;
use App\Traits\BookingService;
use App\Traits\PaymentService;
use App\Models\BookingCheckout;
use App\Models\EditHotelDetail;
use App\Models\MostBookedHotel;

use App\Traits\WhatsappService;
use App\Jobs\MostBookedHotelJob;
use App\Models\EditFlightDetail;
use App\Models\CustomerTraveller;
use Illuminate\Support\Facades\DB;
use App\Models\LoyaltyPointSetting;
use App\Traits\LoyaltyPointsService;
use Illuminate\Support\Facades\Auth;
use App\Models\CustomerLoyaltyPoints;
use App\Models\BookingAdditionalDetail;
use Illuminate\Support\Facades\Validator;
use Google\Service\ShoppingContent\LoyaltyPoints;
use App\Http\Controllers\API\V1\BaseController as BaseController;
use App\Services\TranslateBookingDetails;

class BookingController extends BaseController
{
    use AmadeusService, BookingHelper, CommonService, EmailService, SmsService, WhatsappService, ActiveLog, BookingService, LoyaltyPointsService, PaymentService, TraacsService, Gimmonix, EditBooking;

    public function __construct()
    {
        $this->perPage = count(Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')) > 0 ? Setting::where('config_key', 'general|setting|pagePerAPIRecords')->get('value')[0]['value'] : "20";

        //set AMADEUS API configuration from config key
        $this->amadeusAPIEnvironment = env('AMADEUS_API_CREDENTIALS');

        if ($this->amadeusAPIEnvironment == 'test') {
            $this->amadeusAPIEndPoint = env('AMADEUS_API_TEST_API_ENDPOINT');
            $this->amadeusAPIClientID = env('AMADEUS_API_TEST_CLIENT_ID');
            $this->amadeusAPIClientSecret = env('AMADEUS_API_TEST_CLIENT_SECRET');
            $this->amadeusAPIGrantType = env('AMADEUS_API_TEST_GRANT_TYPE');
        } else {
            $this->amadeusAPIEndPoint = env('AMADEUS_API_LIVE_API_ENDPOINT');
            $this->amadeusAPIClientID = env('AMADEUS_API_LIVE_CLIENT_ID');
            $this->amadeusAPIClientSecret = env('AMADEUS_API_LIVE_CLIENT_SECRET');
            $this->amadeusAPIGrantType = env('AMADEUS_API_LIVE_GRANT_TYPE');
        }

        $this->amadeusAPISecret = env('AMADEUS_API_SECRET');
    }
    /**
     * @OA\Get(
     *   path="/v1/customer/get-booking-list",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Get Booking List",
     *   description="get Booking List",
     *   operationId="getBooking",

     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * get driver status
     *
     * @return \Illuminate\Http\Response
     */
    public function getBookings(Request $request)
    {
        try {
            $showBookingData = false;
            $isSingleResult = false;
            $locale = $request->input('locale');

            if ($request->input('id') != NULL || $request->input('ref') != NULL) {
                $showBookingData = true;
                $isSingleResult = true;
            }
            if ($request->input('email') != NULL) {
                $showBookingData = true;
            }
            if ($request->input('booking_type') != NULL) {
                $showBookingData = true;
            }
            if (Auth::check()) {
                $showBookingData = true;
            }

            if ($showBookingData) {
                if ($isSingleResult && $request->input('ref') != NULL) {
                    $bookingData = Bookings::with(['payment', 'passengers'])
                        ->where('booking_ref', $request->input('ref'))
                        ->first();
                    if ($bookingData) {
                        $bookingData = $bookingData->toArray();
                    }
                } elseif ($isSingleResult && $request->input('id') != NULL) {
                    $bookingData = Bookings::with(['payment', 'passengers'])
                        ->where('id', $request->input('id'))
                        ->first();
                    if ($bookingData) {
                        $bookingData = $bookingData->toArray();
                    }
                } else {
                    $query = Bookings::with(['payment', 'passengers']);
                    if ($request->input('email') != NULL) {
                        $query = $query->where('email', '=', $request->input('email'));
                    }
                    if (Auth::check()) {
                        $query = $query->where('customer_id', '=', Auth::id());
                    }
                    if ($request->input('booking_type') != NULL) {
                        $query = $query->where('booking_type', '=', $request->input('booking_type'));
                    }
                    if ($request->input('future_bookings') != '') {
                        $query = $query->where('booking_start_date', '>', date('Y-m-d'))
                            ->where('booking_status', '=', BookingStatus::Confirmed->value);
                    }

                    $bookingData = $query->orderBy('created_at', 'DESC')->get()->toArray();
                }

                if ($bookingData) {
                    $success = true;
                    $serviceTypes = $this->getServiceTypes();
                    $supplierNames = $this->getSuppliers();

                    if ($isSingleResult) {
                        $bookingData['supplier'] = $supplierNames[$bookingData['supplier_id']] ?? '';
                        $bookingData['service'] = $serviceTypes[$bookingData['service_id']] ?? '';
                        $bookingData['agency'] = '';

                        // eager loaded already
                        $bookingData['payment'] = $bookingData['payment'] ?? [];
                        $bookingData['passengers'] = $bookingData['passengers'] ?? [];
                    } else {
                        foreach ($bookingData as $key => $booking) {
                            $bookingData[$key]['supplier'] = $supplierNames[$booking['supplier_id']] ?? '';
                            $bookingData[$key]['service'] = $serviceTypes[$booking['service_id']] ?? '';
                            $bookingData[$key]['agency'] = '';

                            // eager loaded already
                            $bookingData[$key]['payment'] = $booking['payment'] ?? [];
                            $bookingData[$key]['passengers'] = $booking['passengers'] ?? [];
                        }
                    }

                    if ($locale == Locale::Arabic->value) {
                        $bookingData = array_map(function ($booking) {
                            if ($booking['booking_status'] == BookingStatus::Confirmed->value) {
                                $booking['booking_status_ar'] = 'مؤكد';
                            }
                            if ($booking['booking_status'] == BookingStatus::Failed->value) {
                                $booking['booking_status_ar'] = 'فشل';
                            }
                            if ($booking['booking_status'] == BookingStatus::Pending->value) {
                                $booking['booking_status_ar'] = 'قيد الانتظار';
                            }
                            if ($booking['booking_status'] == BookingStatus::Cancelled->value) {
                                $booking['booking_status_ar'] = 'تم الإلغاء';
                            }
                            if ($booking['booking_status'] == BookingStatus::Cancelled_Not_Refunded->value) {
                                $booking['booking_status_ar'] = 'تم الإلغاء ولم يتم استرداد المبلغ';
                            }
                            if ($booking['booking_status'] == BookingStatus::Cancelled_And_Refunded->value) {
                                $booking['booking_status_ar'] = 'إلغاء واسترداد المبلغ';
                            }

                            if ($booking['booking_type'] == ServiceType::Hotel->value) {
                                $booking['booking_type_ar'] = 'الفندق';
                                $booking['service_ar'] = 'الفندق';
                            }
                            if ($booking['booking_type'] == ServiceType::Flight->value) {
                                $booking['booking_type_ar'] = 'رحلة جوية';
                                $booking['service_ar'] = 'رحلة جوية';
                            }

                            return $booking;
                        }, is_array($bookingData) && isset($bookingData[0]) ? $bookingData : [$bookingData]);

                        // if it was single result, return first item
                        if ($isSingleResult && isset($bookingData[0])) {
                            $bookingData = $bookingData[0];
                        }
                    }

                    return $this->sendResponse($bookingData, 'Booking Listed Successfully!', $success);
                } else {
                    $success = [];
                    return $this->sendError(no_record_found, $success, 200);
                }
            } else {
                return $this->sendError(no_record_found, [], 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrongs ' . $e->getMessage(), 500);
        }
    }


    public function processViewBooking($bookingData)
    {
        if ($bookingData['service_provider'] == 'GIMMONIX') {
            $bookingAdditionalDetails = BookingAdditionalDetail::where('booking_id', $bookingData['id'])->first()->toArray();
            $bookingPayloadDetails = json_decode($bookingAdditionalDetails['booking_details'], true);
        } else {
            $bookingPayloadDetails = json_decode($bookingData['booking_details'], true);
        }
        $bookingData['booking_details'] = $bookingPayloadDetails;
        if ($bookingData['booking_type'] == ServiceType::Flight->value) {
            $bookingData['flightRoute'] = $bookingData['additional_details'];
        } elseif ($bookingData['booking_type'] == ServiceType::Hotel->value) {
            $bookedRooms = [];
            if ($bookingData['booking_type'] == 'HOTEL_BEDS') {
                $bookingDetails = $bookingPayloadDetails;
                $hotelDetails = $bookingDetails['hotelDetails']['hotelDetails'];
                $selectedRooms = $bookingDetails['hotelDetails']['selectedRooms'];
                $roomNameBoardDetails = [];
                foreach ($hotelDetails['rooms'] as $key => $room) {
                    foreach ($room['rates'] as $rate) {
                        $roomNameBoardDetails[$rate['rateKey']] = [
                            'roomName' => ucwords($room['name']),
                            'boardCode' => $rate['boardCode'],
                            'boardName' => $rate['boardName'],
                        ];
                    }
                }
                foreach ($selectedRooms['rooms'] as $key => $selectedRoom) {
                    $bookedRooms[$key] = [
                        'room' => ('Room ' . ($key + 1)),
                        'roomName' => $roomNameBoardDetails[$selectedRoom['rateKey']]['roomName'],
                        'boardName' => $roomNameBoardDetails[$selectedRoom['rateKey']]['boardName'],
                        'paxes' => $selectedRoom['paxes'],
                        'rateComments' => ''
                    ];
                    if (isset($selectedRoom['rateComments'])) {
                        $bookedRooms[$key]['rateComments'] = str_replace(' .  ', '&nbsp;&#x2022;&nbsp;&nbsp;', $selectedRoom['rateComments']);
                    }
                }
            } elseif ($bookingData['booking_type'] == 'GIMMONIX') {
                $selectedRooms = $bookingPayloadDetails['hotelDetails']['selectedRooms'];
                foreach ($bookingPayloadDetails['searchDetails']['occupancies'] as $key => $occupancy) {
                    $bookedRoom = [
                        'room' => ('Room ' . ($key + 1)),
                        'roomName' => $selectedRooms['RoomName'],
                        'boardName' => $selectedRooms['RoomBasis'],
                        'paxes' => '',
                        'rateComments' => (array_key_exists('Remarks', $selectedRooms)) ? implode(',', $selectedRooms['Remarks']) : ''
                    ];
                    array_push($bookedRooms, $bookedRoom);
                }
            }
            $bookingData['bookedRooms'] = $bookedRooms;
        }
        return $bookingData;
    }

    public function getBookingPassengers($bookingType, $bookingId)
    {
        $travelers = CustomerTraveller::select('customers_traveller.*')->where('booking_id', $bookingId)->get();
        if ($bookingType == ServiceType::Hotel->value) {
            $passengers = [];
            if ($travelers && count($travelers) > 0) {
                foreach ($travelers as $traveler) {
                    $passenger = [
                        'booking_id' => $traveler['booking_id'],
                        'first_name' => $traveler['first_name'],
                        'last_name' => $traveler['last_name'],
                        'room' => '',
                        'type' => '',
                        'roomType' => '',
                        'boardName' => '',
                    ];
                    if ($traveler['additional_details'] != '') {
                        $additionalDetails = json_decode($traveler['additional_details'], true);
                        $passenger['room'] = $additionalDetails['room'];
                        $passenger['type'] = $additionalDetails['paxType'];
                        $passenger['roomType'] = $additionalDetails['roomName'];
                        $passenger['boardName'] = $additionalDetails['boardName'];
                    }
                    array_push($passengers, $passenger);
                }
            }
            $travelers = $passengers;
        }
        return $travelers;
    }
    /**
     * @OA\Get(
     *   path="/v1/customer/get-booking-list",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Get Booking List",
     *   description="get Booking List",
     *   operationId="getBooking",

     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * get driver status
     *
     * @return \Illuminate\Http\Response
     */
    public function getBooking(Request $request)
    {
        try {
            $perPage = Setting::where('config_key', 'general|setting|pagePerAPIRecords')->value('value');
            $filter = array(
                'per_page' => $perPage,
            );
            $locale = $request->input('locale');
            $showBookingData = false;
            $isSingleResult = false;
            if (request()->input('id') != NULL) {
                $filter['id'] = request()->input('id');
                $showBookingData = true;
                $isSingleResult = true;
            }
            // if (request()->input('email') != NULL) {
            //     $filter['where'][] = ['email', '=', request()->input('email')];
            //     $showBookingData = true;
            // }
            // if (request()->input('ref') != NULL) {
            //     $filter['where'][] = ['booking_ref', '=', request()->input('ref')];
            //     $showBookingData = true;
            //     $isSingleResult = true;
            // }
            if (
                request()->filled('email') &&
                request()->filled('ref')
            ) {
                $filter['where'][] = ['email', '=', request()->input('email')];
                $filter['where'][] = ['booking_ref', '=', request()->input('ref')];
                $showBookingData = true;
                $isSingleResult = true;
            }
            if ($showBookingData) {
                $bookingDataList = Bookings::getBookings($filter);
                $bookingData = $bookingDataList['data'];

                if ($bookingData) {
                    $additionalbookingDetails = BookingAdditionalDetail::where('booking_id', $bookingData['id'])->first();

                    $success = true;
                    $serviceTypes = $this->getServiceTypes();
                    $supplierNames = $this->getSuppliers();
                    if ($isSingleResult) {
                        $bookingData['supplier'] = $supplierNames[$bookingData['supplier_id']];
                        $bookingData['service'] = $serviceTypes[$bookingData['service_id']];
                        $bookingData['agency'] = '';
                        $bookingData['payment'] = Payment::select('payments.*')->where('booking_id', $bookingData['id'])->first();
                        $bookingData['passengers'] = $this->getBookingPassengers($bookingData['booking_type'], $bookingData['id'], $bookingData['service_provider']);

                        $bookingData = $this->processViewBooking($bookingData);
                    } else {
                        foreach ($bookingData as $key => $booking) {
                            $bookingData[$key]['supplier'] = $supplierNames[$booking['supplier_id']];
                            $bookingData[$key]['service'] = $serviceTypes[$booking['service_id']];
                            $bookingData[$key]['agency'] = '';

                            // $bookingData[$key]['payment'] = Payment::select('payments.*')->where('booking_id', $booking['id'])->first();
                            $bookingData[$key]['payment'] = $booking['payment'] ?? [];
                            $bookingData[$key]['passengers'] = $this->getBookingPassengers($booking['booking_type'], $booking['id'], $booking['service_provider']);

                            $bookingData[$key] = $this->processViewBooking($bookingData[$key]);
                        }
                    }

                    $pages = Page::where('page_code', 'Important_Information')->pluck('id')->first();
                    if (!empty($pages)) {
                        $importantInformation = PageI18ns::where('page_id', $pages)->get();
                        $bookingData['importantInformation'] = $importantInformation->toArray();
                    }

                    $editedbookingData = $this->getEditedBooking($bookingData['id'], $bookingData['booking_type']);
                    if (!empty($editedbookingData)) {
                        if ($bookingData instanceof \App\Models\Bookings) {
                            $bookingData = $bookingData->toArray();
                        }
                        /*** Booking Details json is empty in Booking Table for Failed , Processing and Pending Status booking, so we get data from additional booking details table 
                         ***/
                        if (empty($bookingData['booking_details'])) {
                            if($additionalbookingDetails && $additionalbookingDetails->booking_details){
                                 $bookingData['booking_details']['booking_details'] = json_decode($additionalbookingDetails->booking_details, true) ?? '';
                            }
                           
                        }
                        /**
                         * Fetch booking Details json data Edit booking Table and assign it ot booking data
                         */
                        $editedBookingFullData = json_decode($editedbookingData['booking_details'], true) ?? [];
                        if (!empty($editedBookingFullData)) {
                            $bookingData['booking_details']['booking_details'] = $editedBookingFullData;

                            // Assign Edited Processed price details to external price details
                            $editedProcessedPrice = $editedBookingFullData['processedPrice'] ?? [];
                            if (!empty($editedProcessedPrice)) {
                                $bookingData['sub_total'] = $editedProcessedPrice['totalPrice'] ?? '';
                                $bookingData['tax'] = $editedProcessedPrice['vat'] ?? '';
                                $bookingData['s_markup_fee'] = $editedProcessedPrice['markupFee'] ?? '';
                                $bookingData['s_service_fee'] = $editedProcessedPrice['serviceFee'] ?? '';
                                $bookingData['s_markup_service_fee'] = $editedProcessedPrice['markupServiceFee'] ?? '';
                                $bookingData['s_tax'] = $editedProcessedPrice['totalTaxAmount'] ?? '';
                                $bookingData['chargeable_seat_price'] = $editedProcessedPrice['chargeableSeatPrice'] ?? '';
                                $bookingData['t_markup'] = $editedProcessedPrice['markupValue'] ?? '';
                                $bookingData['total'] = $editedProcessedPrice['grandTotal'] ?? '';
                            }

                            // Assign Edited billing details to external billing details
                            $editedBillingDetails = $editedBookingFullData['billingDetails'] ?? [];
                            if (!empty($editedBillingDetails)) {
                                $bookingData['billing_address1'] = $editedBillingDetails['address'] ?? '';
                                $bookingData['billing_address2'] = $editedBillingDetails['address1'] ?? '';
                                $bookingData['billing_city'] = $editedBillingDetails['city'] ?? '';
                                $bookingData['billing_state'] = $editedBillingDetails['state'] ?? '';
                                $bookingData['billing_country'] = $editedBillingDetails['country'] ?? '';
                                $bookingData['billing_zip'] = $editedBillingDetails['zipcode'] ?? '';
                            }
                        }
                        $bookingData['first_name'] = $editedbookingData['first_name'] ?? '';
                        $bookingData['last_name'] = $editedbookingData['last_name'] ?? '';
                        $bookingData['email'] = $editedbookingData['email'] ?? '';
                        $bookingData['booking_date'] = $editedbookingData['booking_date'] ?? '';
                        $bookingData['booking_start_date'] = $editedbookingData['booking_start_date'] ?? '';
                        if ($bookingData['booking_type'] == ServiceType::Flight->value) {
                            $airLineListData = $bookingData['booking_details']['booking_details']['searchDetails']['airlineList'] ?? [];
                            if (!empty($airLineListData)) {
                                $bookingData['booking_details']['booking_details']['searchDetails']['airlineList'] =  $this->fetchAirlineData($airLineListData);
                            }
                        }
                    }

                    if ($locale == Locale::Arabic->value) {
                        if ($bookingData['booking_status'] == BookingStatus::Confirmed->value) {
                            $bookingData['booking_status_ar'] = 'مؤكد';
                        } else if ($bookingData['booking_status'] == BookingStatus::Failed->value) {
                            $bookingData['booking_status_ar'] = 'فشل';
                        } else if ($bookingData['booking_status'] == BookingStatus::Pending->value) {
                            $bookingData['booking_status_ar'] = 'قيد الانتظار';
                        } else if ($bookingData['booking_status'] == BookingStatus::Cancelled->value) {
                            $bookingData['booking_status_ar'] = 'تم الإلغاء';
                        } else if ($bookingData['booking_status'] == BookingStatus::Cancelled_Not_Refunded->value) {
                            $bookingData['booking_status_ar'] = 'تم الإلغاء ولم يتم استرداد المبلغ';
                        } else if ($bookingData['booking_status'] == BookingStatus::Cancelled_And_Refunded->value) {
                            $bookingData['booking_status_ar'] = 'إلغاء واسترداد المبلغ';
                        }
                    }
                    if($additionalbookingDetails && $additionalbookingDetails->service_provider_booking_response){
                        $pnrDetails = json_decode($additionalbookingDetails->service_provider_booking_response, true) ?? [];
                        $airlinePnr = $gdsPnr = [];
                        if (!empty($pnrDetails)) {
                            $bookingData['pnrDetails'] =  $pnrDetails['data']['associatedRecords'] ?? [];
                            foreach ($bookingData['pnrDetails'] as $record) {
                                if ($record['originSystemCode'] === 'GDS') {
                                    $gdsPnr[] = $record['reference'];
                                } else {
                                    $airlinePnr[] = $record['reference'];
                                }
                            }
                            $bookingData['airlinePnr'] = $airlinePnr;
                            $bookingData['gdsPnr'] = $gdsPnr;
                        }
                    }

                    if ($bookingData['booking_type'] == ServiceType::Flight->value) {
                        $searchDetails = $bookingData['booking_details']['booking_details']['searchDetails'];
                        if (isset($searchDetails)) {
                            $userBookedlang = $searchDetails['languageCode'] ?? Locale::English->value;
                            if ($userBookedlang != $locale) {
                                $bookingData = TranslateBookingDetails::translateFlightBookingDetails($bookingData, $locale);
                            }
                        }
                    } else {
                        $bookingDetails = $bookingData['booking_details']['bookingDetails'];
                        if (isset($bookingDetails)) {
                            $userBookedlang = $bookingDetails['customer_language_code'] ?? Locale::English->value;
                            if ($userBookedlang != $locale) {
                                $bookingData = TranslateBookingDetails::translateHotelBookingDetails($bookingData, $locale);
                            }
                        }
                    }

                    return $this->sendResponse($bookingData, 'Booking Details fetched Successfully!', $success);
                } else {
                    $success = [];
                    return $this->sendError(no_record_found, $success, 200);
                }
            } else {
                return $this->sendError(no_record_found, [], 200);
            }
        } catch (Exception $e) {
            return $this->sendError('Something went wrong. ' . $e->getMessage(), ['error' => $e . $e->getMessage()], 500);
        }
    }

    public function resendEmailBookingDetails(Request $request)
    {
        try {
            $result = [
                'success' => false,
                'message' => 'Unable to find booking details.',
            ];
            $requestData =  $request->all();
            $bookingDetails = Bookings::where('booking_ref', $requestData['reference'])
                ->where('email', $requestData['email'])
                ->first()
                ->toArray();

            if ($bookingDetails) {
                $result = $this->sendBookingEmail($bookingDetails);
            }
            return $this->sendResponse([], $result['message'], $result['success']);
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    public function getCustomerRehltePointsHistory(Request $request)
    {
        $currency = $request->input('currency', Currency::SAR->value);
        $userId = auth()->id();

        if (!$userId) {
            return $this->sendError([], 'User not authenticated', 401);
        }

        $history = CustomerLoyaltyPoints::getRehltePointsHistory($userId, $currency);

        if ($history['status'] == 0) {
            return $this->sendError([], $history['message'], 404);
        }

        return $this->sendResponse($history['data'], $history['message'], 200);
    }

    public function processBookingPayment(Request $request)
    {
        $refId = $amount = $checkStatusUrl = $bookingVia = '';
        $paymentId = $request->input('id');
        $paymentBrand = $request->input('payment_brand');
        if ($paymentId != '' && $paymentBrand != '') {
            $paymentResponse = $this->getPaymentStatus($paymentBrand, $paymentId);
            if ($paymentResponse['status'] == true) {
                if (array_key_exists('customParameters', $paymentResponse['data'])) {
                    $refId = $paymentResponse['data']['customParameters']['SHOPPER_ref_id'];
                    $amount = $paymentResponse['data']['customParameters']['SHOPPER_amount'];
                }
            }
            if ($refId != '') {
                $referenceDetails = BookingCheckout::whereRef_id($refId)->first();
                $bookingVia = $referenceDetails->booking_via;
                $checkStatusUrl = $referenceDetails->checkout_url . '/payment/payment-success?ref_id=' . $refId;
            }

            $statusUrl = env('API_BASE_URL') . 'booking/get-booking-status?';
            $data = [
                'payment_id' => $paymentId,
                'ref_id' => $refId,
                'amount' => $amount,
                'status_url' => $statusUrl,
                'booking_via' => $bookingVia,
                'check_status_url' => $checkStatusUrl
            ];
            if ($bookingVia == 'web') {
                return redirect()->to($checkStatusUrl);
            } else {
                return view('admin.booking.payment')->with(['data' => $data]);
            }
        } else {
            return redirect()->to(env('WEBSITE_URL'));
        }
    }

    public function getLoyaltyPointExpiryDate($service_end_date)
    {
        $loyaltyPointSetting = LoyaltyPointSetting::first();
        if ($loyaltyPointSetting) {

            $expiry_year = $loyaltyPointSetting->expiry_year ?? 1;
            $expiry_month = $loyaltyPointSetting->expiry_month ?? 0;
            $expiry_day = $loyaltyPointSetting->expiry_day ?? 0;

            $checkoutDate = Carbon::parse($service_end_date);

            $addYears = $expiry_year;
            $addMonths = ($expiry_month > 0) ? $expiry_month  : 0;
            $addDays = ($expiry_day > 0) ? $expiry_day : 0;

            $expiryDate = $checkoutDate->copy()
                ->addYears($addYears)
                ->addMonths($addMonths)
                ->addDays($addDays)
                ->format('Y-m-d');
        } else {

            $expiry_year =  1;
            $expiry_month =  0;
            $expiry_day =  0;

            $checkoutDate = Carbon::parse($service_end_date);

            $addYears = $expiry_year;
            $addMonths = ($expiry_month > 0) ? $expiry_month  : 0;
            $addDays = ($expiry_day > 0) ? $expiry_day : 0;

            $expiryDate = $checkoutDate->copy()
                ->addYears($addYears)
                ->addMonths($addMonths)
                ->addDays($addDays)
                ->format('Y-m-d');
        }
        return $expiryDate;
    }

    public function getBookingStatus(Request $request)
    {
        $result = [
            'success' => false,
            'data' => [
                'status' => '',
                'url' => ''
            ],
            'message' => ''
        ];
        try {
            $requestData =  $request->all();
            $refId = $request->input('ref_id');
            if ($refId != '') {
                $result = [
                    'success' => true,
                    'data' => [
                        'status' => 'in_progress',
                        'url' => ''
                    ],
                    'message' => ''
                ];
                $referenceDetails = BookingCheckout::whereRef_id($refId)->first();
                if ($referenceDetails) {
                    $booking_details = json_decode($referenceDetails->booking_details, true);
                    $walletUsages = $booking_details['processedPrice']['walletUsage'] ?? [];
                    if ($referenceDetails) {
                        $bookingDetails = Bookings::whereRef_id($refId)->first();
                        if ($bookingDetails) {
                            $result['data']['email'] = $bookingDetails->email;
                            $result['data']['status'] = $bookingDetails->booking_status;
                            $result['data']['url'] = $referenceDetails->checkout_url;
                            $bookingType = strtolower($bookingDetails->booking_type);
                            $booking_details = json_decode($referenceDetails->booking_details, true);
                            $walletUsages = $booking_details['processedPrice']['walletUsage'] ?? [];
                            $redeemUsages = $booking_details['processedPrice']['loyaltyPointsUsage'] ?? [];
                            if ($bookingDetails->booking_status == BookingStatus::Confirmed->value) {
                                if ($referenceDetails['booking_type'] == ServiceType::Hotel->value && $referenceDetails['traacs_voucher_created'] == 0) {
                                    $traacsResponse = $this->createVoucherInTraacs($referenceDetails['booking_id']);
                                    if ($traacsResponse['status'] == true) {
                                        BookingCheckout::where('ref_id', $refId)->update(['traacs_voucher_created' => 1]);
                                        Bookings::where('ref_id', $refId)->update(['traacs_created' => 1]);
                                    }

                                    $hotelId = $booking_details['hotelDetails']['selectedRooms']['HotelId'] ?? '';
                                    if ($hotelId != '') {
                                        dispatch(new MostBookedHotelJob(
                                            $booking_details['hotelDetails']['selectedRooms']['HotelId'] ?? '',
                                            $booking_details['hotelDetails']['hotelDetails']['DisplayName'] ?? '',
                                            $referenceDetails['booking_id'] ?? ''
                                        ));
                                    }
                                }
                                // Wallet Update
                                if (!empty($walletUsages)) {
                                    // Get all wallet IDs to update
                                    $walletIds = collect($walletUsages)
                                        ->pluck('wallet_id')
                                        ->filter()
                                        ->toArray();

                                    $wallets = Wallet::whereIn('id', $walletIds)->get()->keyBy('id');

                                    $today = date('Y-m-d');

                                    foreach ($walletUsages as $walletUsage) {
                                        $walletId = $walletUsage['wallet_id'] ?? null;
                                        $originalUsed = isset($walletUsage['original_used']) ? floatval($walletUsage['original_used']) : 0;

                                        if ($walletId && isset($wallets[$walletId])) {
                                            $wallet = $wallets[$walletId];

                                            $wallet->type = 'debit';
                                            $wallet->wallet_used_status = 'used';
                                            $wallet->wallet_used_amount = $originalUsed;
                                            $wallet->used_date = $today;
                                            $wallet->save();
                                        }
                                    }
                                }


                                if ($bookingDetails->booking_type == ServiceType::Hotel->value) {
                                    $service_end_date = $booking_details['searchDetails']['stay']['checkOut'];
                                    $searchFrom = $booking_details['searchDetails']['originDevice'] ?? 'web';
                                    if ($searchFrom != 'web' && $bookingDetails->booking_processed == 0) {
                                        Bookings::where('ref_id', $refId)->update([
                                            'booking_from' => $searchFrom
                                        ]);
                                    }
                                }
                                if ($bookingDetails->booking_type == ServiceType::Flight->value) {
                                    $service_end_date_raw = $booking_details['flightDetails']['airlines'][0]['flights'][0]['segments'][0]['arrival']['at'] ?? null;

                                    $service_end_date = $service_end_date_raw
                                        ? Carbon::parse($service_end_date_raw)->format('Y-m-d')
                                        : null;
                                }

                                $expiryDate = $this->getLoyaltyPointExpiryDate($service_end_date);

                                CustomerLoyaltyPoints::where('booking_id', $bookingDetails->id)->update([
                                    'service_end_date' => $service_end_date,
                                    'expiry_date' => $expiryDate,
                                ]);
                                //Loyalty Point Update
                                if (!empty($redeemUsages)) {
                                    $redeemIds = collect($redeemUsages)->pluck('id')->filter()->toArray();
                                    if (!empty($redeemIds)) {
                                        CustomerLoyaltyPoints::whereIn('id', $redeemIds)->update([
                                            'used_status' => 'used',
                                            'service_end_date' => $service_end_date,
                                            'expiry_date' => $expiryDate,
                                        ]);
                                    }
                                }
                                $savedBookingDetails = Bookings::whereRef_id($refId)->first()->toArray();

                                if ($bookingDetails->booking_processed == 0) {
                                    if ($savedBookingDetails['email_notification'] == 0) {
                                        dispatch(new \App\Jobs\SendBookingEmailJob($savedBookingDetails, $refId));
                                        // $isEmailSent =  $this->sendBookingEmail($savedBookingDetails);
                                        // if ($isEmailSent['success'] == true) {
                                        // 	Bookings::where('ref_id', $refId)->update(['email_notification' => 1]);
                                        // }
                                    }

                                    if ($savedBookingDetails['sms_notification'] == 0) {
                                        dispatch(new \App\Jobs\SendBookingSmsJob($savedBookingDetails, $refId));
                                        // $isSmsSent = $this->sendBookingSms($savedBookingDetails);
                                        // if ($isSmsSent) {
                                        // 	Bookings::where('ref_id', $refId)->update(['sms_notification' => 1]);
                                        // }
                                    }
                                    if ($savedBookingDetails['whatsapp_notification'] == 0) {
                                        dispatch(new \App\Jobs\SendBookingWhatsappJob($savedBookingDetails, $refId));
                                        // $iswhatsappSent = $this->sendBookingWhatsapp($savedBookingDetails);
                                        // if ($iswhatsappSent) {
                                        // 	Bookings::where('ref_id', $refId)->update(['whatsapp_notification' => 1]);
                                        // }
                                    }
                                    Bookings::where('ref_id', $refId)->update(['booking_processed' => 1]);
                                }

                                if ($referenceDetails->booking_via == 'web-app') {
                                    $result['data']['url'] = str_replace(['/en', '/ar'], ['', ''], $result['data']['url']);
                                }
                                $result['data']['url'] .= '/booking/' . $bookingType . '-booking?booking_reference=' . $bookingDetails->booking_ref . '&email=' . $bookingDetails->email . '&reference=create';
                            } else {
                                $failurePage = '/payment';
                                if ($referenceDetails->booking_via == 'web-app' && $referenceDetails->booking_type == ServiceType::Flight->value) {
                                    $failurePage = '/flightCheckout';
                                    $result['data']['url'] = str_replace(['/en', '/ar'], ['', ''], $result['data']['url']);
                                } elseif ($referenceDetails->booking_via == 'web-app' && $referenceDetails->booking_type == ServiceType::Hotel->value) {
                                    $result['data']['url'] = str_replace(['/en', '/ar'], ['', ''], $result['data']['url']);
                                    $failurePage = '/hotelCheckout';
                                }
                                $result['data']['url'] .= $failurePage . '?ref_id=' . $refId . '&payment_brand=VISA&status=' . $bookingDetails->booking_status . '&sub_status=' . $bookingDetails->booking_sub_status . '&message=' . $bookingDetails->booking_status_description;
                            }
                        }
                    }
                }
            } else {
                $result['message'] = 'Invalid Reference';
            }
            return $this->sendResponse($result['data'], $result['message'], $result['success']);
        } catch (\Exception $e) {
            return $this->sendError([], 'Something went wrong' . $e->getFile() . $e->getLine(), ['error' => $e->getMessage()], 500);
        }
    }

    public function checkBookingExist(Request $request)
    {
        $result = [
            'status' => false,
            'message' => 'Booking not found',
            'data' => []
        ];
        $validated = Validator::make(
            $request->all(),
            [
                'ref_id' => 'required',
                'email' => 'required|email'
            ],
        );
        if ($validated->fails()) {
            return $this->sendError('Validation Error', [$validated->errors()], 200);
        }

        try {

            $validatedData = $validated->validated();

            $bookings = Bookings::where('booking_ref', $validatedData['ref_id'])
                ->where('email', $validatedData['email'])
                ->first();
            if ($bookings) {
                $result['status'] = true;
                $result['message'] = 'Booking found';
                $result['data'] = [
                    'booking_id' => $bookings->id,
                    'booking_ref' => $bookings->booking_ref,
                    'first_name' => $bookings->first_name,
                    'last_name' => $bookings->last_name,
                    'email' => $bookings->email,
                    'phone' => $bookings->phone,
                    'booking_type' => $bookings->booking_type,
                    'service_provider' => $bookings->service_provider
                ];

                return $this->sendResponse($result, '', 200);
            }
            return $this->sendResponse($result, '', 200);
        } catch (\Exception $e) {
            return $this->sendError([], 'Something went wrong', ['error' => $e->getMessage()], 500);
        }
    }

    public function getPaymentStatusByPaymentId(Request $request)
    {
        $result = [
            'success' => false,
            'message' => 'Status not found',
            'data' => []
        ];

        $refId = $amount = '';
        $paymentId = $request->input('id');
        $paymentBrand = $request->input('payment_brand');
        if ($paymentId != '' && $paymentBrand != '') {
            $paymentResponse = $this->getPaymentStatus($paymentBrand, $paymentId);
            if ($paymentResponse['status'] == true) {
                if (array_key_exists('customParameters', $paymentResponse['data'])) {
                    $refId = $paymentResponse['data']['customParameters']['SHOPPER_ref_id'];
                    $amount = $paymentResponse['data']['customParameters']['SHOPPER_amount'];
                    $result['success'] = true;
                }
            }

            $result['data'] = [
                'payment_id' => $paymentId,
                'ref_id' => $refId,
                'amount' => $amount
            ];
        }
        return $this->sendResponse($result['data'], $result['message'], $result['success']);
    }

    public function getBookingStatusByRefId(Request $request)
    {
        $result = [
            'success' => false,
            'message' => 'Booking not found',
            'data' => []
        ];
        $refId = $request->input('ref_id');
        $bookings = Bookings::where('ref_id', $refId)->first();
        if ($bookings) {
            $result['success'] = true;
            $result['message'] = 'Booking found';
            $result['data'] = [
                'booking_id' => $bookings->id,
                'booking_ref' => $bookings->booking_ref,
                'booking_status' => $bookings->booking_status,
                'booking_sub_status' => $bookings->booking_sub_status,
                'booking_status_message' => $bookings->booking_status_message,
                'booking_status_description' => $bookings->booking_status_description
            ];

            return $this->sendResponse($result['data'], $result['message'], $result['success']);
        }
        return $this->sendResponse($result['data'], $result['message'], $result['success']);
    }

    /**
     * For Testing purpose
     */

    // public function testWhatsapp(Request $request)
    // {
    //     $savedBookingDetails = Bookings::whereRef_id($request->input('ref_id'))->first()->toArray();


    //     return $this->sendBookingWhatsapp($savedBookingDetails);
    // }
    public function testEmailService(Request $request)
    {
        $request->validate([
            'ref_id' => 'required',
        ]);
        $savedBookingDetails = Bookings::whereRef_id($request->input('ref_id'))->first()->toArray();
        dispatch(new \App\Jobs\SendBookingEmailJob($savedBookingDetails, $request->input('ref_id')));
        return $this->sendResponse([], 'Booking Email Sent Successfully', true);
        // return $this->sendBookingEmail($savedBookingDetails);
    }
    public function testWhatsappService(Request $request)
    {
        $request->validate([
            'ref_id' => 'required',
        ]);
        $savedBookingDetails = Bookings::whereRef_id($request->input('ref_id'))->first()->toArray();
        // dispatch(new \App\Jobs\SendBookingWhatsappJob($savedBookingDetails, $request->input('ref_id')));
        // return $this->sendBookingWhatsapp($savedBookingDetails);
        return $this->sendResponse([], 'Booking Email Sent Successfully', true);

        return $this->sendBookingWhatsapp($savedBookingDetails);
    }

    /**
     * Delete booking checkouts older than 7 days that do not have a booking ID.
     */
    public function deleteBookingCheckout()
    {
        $dateBefore = Carbon::today()->subDays('7')->format('Y-m-d H:i:s');
        $bookingDeleted =   BookingCheckout::where('created_at', '<=', $dateBefore)
            ->where(function ($query) {
                $query->whereNull('booking_id')
                    ->orWhere('booking_id', '');
            })
            ->delete();

        if ($bookingDeleted == 0) {
            return $this->sendResponse('No Records to delete', $bookingDeleted, 200);
        }
        return $this->sendResponse('Booking checkouts deleted successfully', $bookingDeleted, 200);
    }

    /**
     * Delete supplier logs older than 7 days that do not have a booking ID.
     */

    public function deleteSupplierLog()
    {
        $dateBefore = Carbon::today()->subDays(7)->format('Y-m-d H:i:s');

        $supplierLogDeleted = SupplierLog::where('created_at', '<=', $dateBefore)
            ->where(function ($query) {
                $query->whereNull('booking_id')
                    ->orWhere('booking_id', '');
            })
            ->delete();

        if ($supplierLogDeleted == 0) {
            return $this->sendResponse('No Records to delete', $supplierLogDeleted, 200);
        }
        return $this->sendResponse('Supplier Log deleted successfully', $supplierLogDeleted, 200);
    }
}
