<?php

/**
 * @package     Customers
 * @subpackage  Customer
 * @Author      Amar Technolabs Pvt. ltd(info@amarinfotech.com)
 * @Copyright(C) 2023 [NAME OF THE ORGANISATION THAT ON BEHALF OF THE CODE WE ARE WORKING].
 * @Version 1.0.0
 * module of the Customers.
 */

namespace App\Http\Controllers\API\V1;

use App\Enums\Locale;
use App\Http\Controllers\API\V1\BaseController as BaseController;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Models\BestOffer;

class BestOfferController extends BaseController
{

    /**
     * @OA\Get(
     *   path="/v1/flights/get-featured-hotels-list",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Get Best Offers List",
     *   description="get Best Offers List",
     *   operationId="getBest Offers",

     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * get driver status
     *
     * @return \Illuminate\Http\Response
     */
    public function getBestOffers(Request $request)
    {
        try {
            $filter = [];
            $locale = $request->input('locale', Locale::English->value);
            $bestOfferDataList = BestOffer::getBestOffers($filter);
            $bestOfferData = $bestOfferDataList['data'];
            if (count($bestOfferData) > 0) {
                foreach ($bestOfferData as $key => $bestOffer) {
                    $bestOfferTemp = $bestOffer;
                    if ($locale == Locale::Arabic->value) {
                        $bestOffer['offer_title'] = $bestOffer['offer_title_ar'];
                        $bestOffer['offer_content'] = $bestOffer['offer_content_ar'];
                    }
                    unset($bestOffer['offer_title_ar']);
                    unset($bestOffer['offer_content_ar']);
                }
            }
            if ($bestOfferData[0]) {
                $success = true;
                return $this->sendResponse($bestOfferData, 'Best offers listed successfully!', $success);
            } else {
                $success = [];
                return $this->sendError(no_record_found, $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/customer/create-featured-hotel",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Create Traveller",
     *   description="Send request for Create Traveller",
     *   operationId="createTraveller",
     *   @OA\RequestBody(
     *     required=true,
     *     description="Create Traveller",
     *     @OA\MediaType(
     *       mediaType="multipart/form-data",
     *       @OA\Schema(
     *           required={"title","first_name","last_name","date_of_birth","gender","nationality","id_type","id_number","issue_date","expiry_date","issue_country"},
     *           @OA\Property(property="language_code", type="string",description="pass language_code either Locale::English->value or Locale::Arabic->value", default="en" ),
     *           @OA\Property(property="title", type="string",description="pass title either 'mr','mrs' or 'miss'", default="" ),
     *           @OA\Property(property="first_name", type="string",description="need to pass a first name", default="" ),
     *           @OA\Property(property="second_name", type="string",description="pass a second name", default="" ),
     *           @OA\Property(property="last_name", type="string",description="need to pass last name", default="" ),
     *           @OA\Property(property="date_of_birth", type="string",description="need to pass date of birth in Y-m-d format", default="" ),
     *           @OA\Property(property="gender", type="string",description="need to pass gender either 'male' or 'female'", default="" ),
     *           @OA\Property(property="nationality", type="string",description="need to pass iso_code that fetch by call API 'get-countries'", default="" ),
     *           @OA\Property(property="id_type", type="string",description="need to pass id type either 'passport' or 'national_id'", default="" ),
     *           @OA\Property(property="id_number", type="integer",description="need to pass id number as passed id type", default="" ),
     *           @OA\Property(property="issue_date", type="string",description="need to pass issue date in Y-m-d format", default="" ),
     *           @OA\Property(property="expiry_date", type="string",description="need to pass expiry date in Y-m-d format", default="" ),
     *           @OA\Property(property="issue_country", type="string",description="need to pass iso_code that fetch by call API 'get-countries'", default="" ),
     *           @OA\Property(property="status", type="string",description="pass status either 'active' or 'inactive'", default="" ),
     *           @OA\Property(property="upload_document", type="string", format="binary",description="select document image *ensure that you are uploading an image is 2MB or less and one of the following types: JPG,JPEG, or PNG"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function createBestOffer(Request $request)
    {
        $return = array(
            'status' => 0,
            'data' => [],
            'message' => 'Something went wrong'
        );
        try {
            $validator = Validator::make($request->all(), [
                'offer_title' => 'nullable',
                'offer_content' => 'nullable',
                'offer_link' => 'required',
                'offer_image' => 'required',
                'status' => 'nullable',
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'message' => $validator->errors()->first(), 'data' => [$validator->errors()]], 200);
            }
            $bestOfferData = $request->all();
            $requestData = array(
                'offer_title' => $bestOfferData['offer_title'],
                'offer_content' => $bestOfferData['offer_content'],
                'offer_link' => $bestOfferData['offer_link'],
                'offer_image' => $bestOfferData['offer_image'],
                'status' => $bestOfferData['status'],
            );
            $response = BestOffer::createBestOffer($requestData);
            if ($response) {
                $success = true;
                return $this->sendResponse($response, 'Best offer have added successfully!', $success);
            } else {
                $success = [];
                return $this->sendError('Best offer Not Saved', $success, 200);
            }
        } catch (\Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/customer/update-featured-hotel",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Update Traveller",
     *   description="Send request for Update Traveller",
     *   operationId="updateTraveller",
     *   @OA\RequestBody(
     *     required=true,
     *     description="Update Traveller",
     *     @OA\MediaType(
     *       mediaType="multipart/form-data",
     *       @OA\Schema(
     *           required={"traveller_id","title","first_name","last_name","date_of_birth","gender","nationality","id_type","id_number","issue_date","expiry_date","issue_country"},
     *           @OA\Property(property="language_code", type="string",description="pass language_code either Locale::English->value or Locale::Arabic->value", default="en" ),
     *           @OA\Property(property="traveller_id", type="string", default="" ,description="need to pass a traveller id as user want to update"),
     *           @OA\Property(property="title", type="string",description="pass title either 'mr','mrs' or 'miss'", default="" ),
     *           @OA\Property(property="first_name", type="string",description="need to pass a first name", default="" ),
     *           @OA\Property(property="second_name", type="string",description="pass a second name", default="" ),
     *           @OA\Property(property="last_name", type="string",description="need to pass last name", default="" ),
     *           @OA\Property(property="date_of_birth", type="string",description="need to pass date of birth in Y-m-d format", default="" ),
     *           @OA\Property(property="gender", type="string",description="need to pass gender either 'male' or 'female'", default="" ),
     *           @OA\Property(property="nationality", type="string",description="need to pass iso_code that fetch by call API 'get-countries'", default="" ),
     *           @OA\Property(property="id_type", type="string",description="need to pass id type either 'passport' or 'national_id'", default="" ),
     *           @OA\Property(property="id_number", type="integer",description="need to pass id number as passed id type", default="" ),
     *           @OA\Property(property="issue_date", type="string",description="need to pass issue date in Y-m-d format", default="" ),
     *           @OA\Property(property="expiry_date", type="string",description="need to pass expiry date in Y-m-d format", default="" ),
     *           @OA\Property(property="issue_country", type="string",description="need to pass iso_code that fetch by call API 'get-countries'", default="" ),
     *           @OA\Property(property="status", type="string",description="pass status either 'active' or 'inactive'", default="" ),
     *           @OA\Property(property="upload_document", type="string", format="binary",description="select document image *ensure that you are uploading an image is 2MB or less and one of the following types: JPG,JPEG, or PNG"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function updateBestOffer(Request $request)
    {
        $return = array(
            'status' => 0,
            'data' => [],
            'message' => 'Something went wrong'
        );
        try {
            $bestOfferId = BestOffer::where('id', request()->input('id'))->value('id');
            if (!$bestOfferId) {
                $success = [];
                return $this->sendError(no_record_found, $success, 400);
            }
            $validator = Validator::make($request->all(), [
                'offer_title' => 'nullable',
                'offer_content' => 'nullable',
                'offer_link' => 'required',
                'offer_image' => 'required',
                'status' => 'nullable',
            ]);
            if ($validator->fails()) {
                return response()->json(['status' => false, 'message' => $validator->errors()->first(), 'data' => [$validator->errors()]], 200);
            }
            $bestOfferData = $request->all();
            $requestData = array(
                'id' => $bestOfferData['id'],
                'offer_title' => $bestOfferData['offer_title'],
                'offer_content' => $bestOfferData['offer_content'],
                'offer_link' => $bestOfferData['offer_link'],
                'offer_image' => $bestOfferData['offer_image'],
                'status' => $bestOfferData['status'],
            );
            $response = BestOffer::updateBestOffer($requestData);
            if ($response) {
                $success = true;
                return $this->sendResponse($response, 'Best offer have updated successfully!', $success);
            } else {
                $success = [];
                return $this->sendError('Best Offers Not Updated', $success, 200);
            }
        } catch (\Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Delete(
     *   path="/v1/customer/delete-traveller/{traveller_id}",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Customer"},
     *   summary="Delete traveller",
     *   description="need to pass traveller Id to delete traveller for multi traveller delete pass traveller id separated by commas",
     *   operationId="traveller-delete",
     *      @OA\Parameter(
     *         name="traveller_id",
     *         in="path",
     *         required=true,
     *         description="need to pass traveller Id to delete traveller for multi traveller delete pass traveller id separated by commas"
     *      ),
     * @OA\Parameter(
     *      name="language_code",
     *      in="query",
     *      required=false,
     *      description="Language code parameter either Locale::English->value or Locale::Arabic->value",
     *      @OA\Schema(
     *          type="string",
     *          example="en",
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     *
     * @return \Illuminate\Http\Response
     */
    public function deleteBestOffer($id)
    {
        try {
            $selectedBestOfferIDs = explode(',', $id);
            $message = "";
            foreach ($selectedBestOfferIDs as $selectedBestOfferID) {
                $bestOfferID = BestOffer::where('id', $selectedBestOfferID)->value('id');
                if ($bestOfferID) {
                    $response = BestOffer::deleteBestOffer($bestOfferID);
                    $message .= $response['message'] . '</br>';
                }
            }
            if (empty($response)) {
                return $this->sendError(no_record_found, [], 200);
            } else {
                return $this->sendResponse([], 'Best offer have deleted successfully.');
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }
}
