<?php

/**
 * @package     AuthGeneral
 * @subpackage  AuthGeneral
 * @Author      Amar Technolabs Pvt. ltd(info@amarinfotech.com)
 * @Copyright(C) 2023 [NAME OF THE ORGANISATION THAT ON BEHALF OF THE CODE WE ARE WORKING].
 * @Version 1.0.0
 * module of the AuthGeneral.
 */

namespace App\Http\Controllers\API\V1;

use DB;
use Carbon\Carbon;
use App\Models\Setting;
use App\Models\ApiLogin;
use App\Models\AppUsers;
use App\Models\Customer;
use App\Models\AppUserOtp;
use Illuminate\Support\Str;
use Laravel\Passport\Token;
use App\Traits\EmailService;
use Illuminate\Http\Request;
use App\Models\AppUserAddresses;
use App\Models\DeleteAccountOtp;
use App\Models\UserLoginHistory;
use Laravel\Passport\HasApiTokens;
use Laravel\Passport\RefreshToken;
use App\Models\DeleteAccountOtpLog;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Laravel\Passport\TokenRepository;
use Illuminate\Support\Facades\Validator;
use Laravel\Passport\RefreshTokenRepository;
use Laravel\Passport\PersonalAccessTokenFactory;
use App\Http\Controllers\API\V1\BaseController as BaseController;

class AuthGeneralController extends BaseController
{
    use EmailService;

    /**
     * @OA\Post(
     *   path="/v1/api-login",
     *   tags={"Authentication"},
     *   summary="Api Users Login",
     *   description="Enter Username Ex:b2cweb, b2cmobile </br>or</br>Enter Password: P@ssw0rd",
     *   operationId="api-login",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           required={"username","password"},
     *           @OA\Property(property="username", type="string" ),
     *           @OA\Property(property="password", type="string" ),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Login api
     *
     * @return \Illuminate\Http\Response
     */
    public function ApiUserLogin(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'username' => 'required',
                'password' => 'required'
            ]);
            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }

            // Authentication logic for two types of users
            $b2cUser = ApiLogin::where('name', $request->username)->where(function ($query) {
                $query->where('type', 'b2c_web')->orWhere('type', 'b2c_mobile');
            })->first();

            $b2cWebUser = ApiLogin::where('name', $request->username)->where('type', 'b2c_web')->first();
            $b2cMobileUser = ApiLogin::where('name', $request->username)->where('type', 'b2c_mobile')->first();

            if (($b2cWebUser || $b2cMobileUser) && Hash::check($request->password, $b2cUser->password)) {
                $token = $b2cUser->createToken('AuthToken');
                $success = $b2cUser;
                $success['token'] = $token;

                return $this->sendResponse([$success], 'API [' . $b2cUser->name . '] User Login Successfully');
            } else {
                return $this->sendError('Invalid Credentials');
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Get(
     *   path="/v1/check-token-validity",
     *   tags={"Authentication"},
     *   summary="Check Passport Token Validity",
     *   description="Check the validity of a Passport Token.",
     *   operationId="check-token-validity",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      description="The ID of the token to check validity.",
     *      @OA\Schema(
     *           required={"tokenId"},
     *           @OA\Property(property="tokenId", type="string" ),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     *
     * @return \Illuminate\Http\Response
     */
    public function checkTokenValidity(Request $request)
    {
        $requestData = $request->only([
            'tokenId'
        ]);

        $validator = Validator::make($requestData, [
            'tokenId' => 'required'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 200);
        }

        try {
            $tokenId = $request->tokenId ?? '';
            $token = Token::find($tokenId);
            if ($token) {
                if ($token && $token->expires_at > Carbon::now()) {
                    return $this->sendResponse($token, 'Token is valid');
                } else {
                    return $this->sendError($token, 'Token is expired');
                }
            } else {
                return $this->sendError('Token is not valid');
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Get(
     *   path="/v1/refresh-token",
     *   tags={"Authentication"},
     *   summary="Refresh Passport Token",
     *   description="Refreshes a Passport Token for a api user.",
     *   operationId="refresh-token",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      description="The ID of the token to be refreshed.",
     *      @OA\Schema(
     *           required={"tokenId"},
     *           @OA\Property(property="tokenId", type="string" ),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Login api
     *
     * @return \Illuminate\Http\Response
     */
    public function refreshToken2(Request $request)
    {
        $requestData = $request->only([
            'tokenId'
        ]);

        $validator = Validator::make($requestData, [
            'tokenId' => 'required'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 422);
        }

        try {
            $tokenId = $request->tokenId ?? '';
            $expiredToken = Token::find($tokenId);
            if ($expiredToken) {

                if ($expiredToken->expires_at > Carbon::now()) {
                    return $this->sendResponse($expiredToken, 'Token is valid');
                } else {
                    $apiUser = ApiLogin::where('id', $expiredToken->user_id)->first();
                    $token = $apiUser->createToken('AuthToken');
                    $success = $token;
                    return $this->sendResponse([$success], 'Api [' . $apiUser->name . '] User Login Successfully');
                }
            } else {
                return $this->sendError('Token is not valid');
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }
    public function refreshToken(Request $request)
    {
        return response()->json(['valid' => auth()->check()]);
        die;
        try {
            $tokenId = $request->tokenId ?? '';
            $expiredToken = Token::find($tokenId);
            if ($expiredToken) {

                if ($expiredToken->expires_at > Carbon::now()) {
                    return $this->sendResponse($expiredToken, 'Token is valid');
                } else {
                    $apiUser = ApiLogin::where('id', $expiredToken->user_id)->first();
                    $token = $apiUser->createToken('AuthToken');
                    $success = $token;
                    return $this->sendResponse([$success], 'Api [' . $apiUser->name . '] User Login Successfully');
                }
            } else {
                return $this->sendError('Token is not valid');
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/login",
     *   tags={"Authentication"},
     *   summary="Users Login",
     *   description="Pass Mobile no Ex:+91 1111111111 </br>or</br>Pass Email Address",
     *   operationId="login",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",
        required={"mobile","password"},
     *           @OA\Property(property="mobile", type="string",  ),
     *           @OA\Property(property="password", type="string",  ),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Login api
     *
     * @return \Illuminate\Http\Response
     */
    public function Login(Request $request)
    {

        try {

            $validator = Validator::make($request->all(), [
                'mobile' => 'required',
                'password' => 'required'
            ]);
            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }

            $user = Customer::where('mobile', '=', $request->mobile)->orWhere('email', $request->mobile)->where('status', '!=', 'deleted')->get()->first();

            if (!empty($user)) {

                if (!Hash::check($request->password, $user['password'])) {
                    $success = [];
                    return $this->sendError('Invalid Credentials', $success, 200);
                }

                if (Auth::guard('appuser')->loginUsingId($user->id)) {
                    $userData = Auth::guard('appuser')->user();
                    if ($user->status == '0') {
                        $success = [];
                        return $this->sendError('User is In Active, please contact to administrator', $success, 200);
                    } else if ($user->status == '2') {
                        $success = [];
                        return $this->sendError('User not Found', $success, 200);
                    } else {
                        $success = $user;
                        $success['token'] = $userData->createToken('AuthToken')->accessToken;

                        return $this->sendResponse([$success], 'User Login Successfully');
                    }
                } else {
                    $success = [];
                    return $this->sendError('Error During Login', $success, 200);
                }
            } else {

                $success = [];
                return $this->sendError('User not found', $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/signup",
     *   tags={"Authentication"},
     *   summary="Send request for Sign Up",
     *   description="User Sign Up request <br><br>
      Place id,website_url field is nullable <br>
      Place id field comes from googlesearch api",
     *   operationId="signup",
     *   @OA\RequestBody(
     *     required=true,
     *     description="User Sign Up",
     *     @OA\MediaType(
     *       mediaType="multipart/form-data",
     *       @OA\Schema(
     *             required={"firm","owner_name","mobile","address",
     *            "country","state","city","pincode" },
     *             @OA\Property(property="firm", type="string"),
     *             @OA\Property(property="owner_name", type="string"),
     *             @OA\Property(property="email", type="email"),
     *             @OA\Property(property="mobile", type="string"),
     *             @OA\Property(property="place_id", type="string"),
     *             @OA\Property(property="address", type="string"),
     *             @OA\Property(property="country", type="string"),
     *             @OA\Property(property="state", type="string"),
     *             @OA\Property(property="city", type="string"),
     *             @OA\Property(property="pincode", type="string"),
     *             @OA\Property(property="company_gst_no", type="string"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function signUp(Request $request)
    {

        try {

            // set custom rule for gst_no validation
            Validator::extend('custom_rule', function ($attribute, $value) {
                return preg_match('/\d{2}[A-Z]{5}\d{4}[A-Z]{1}[A-Z\d]{1}[Z]{1}[A-Z\d]{1}/', $value);
            }, 'The company gst no format is invalid.');

            // set custom rule for gst_no validation
            Validator::extend('email_rule', function ($attribute, $value) {
                return preg_match('/(.+)@(.+)\.(.+)/i', $value);
            }, 'The email format is invalid.');


            // set custom rule for check if email and mobile is already added and that status is not deleted
            Validator::extend('custom_rule_email', function ($attribute, $value) {
                $query = AppUsers::where('email', $value)
                    ->where('status', '!=', '2')->where('id', '!=', Auth::id())->get();
                return !$query->count();
            }, 'The email has already been taken');
            //
            // set custom rule for check if email and mobile is already added and that status is not deleted
            Validator::extend('custom_rule_mobile', function ($attribute, $value) {
                $query = AppUsers::where('mobile', $value)
                    ->where('status', '!=', '2')->where('id', '!=', Auth::id())->get();
                return !$query->count();
            }, 'The mobile has already been taken');

            $validator = Validator::make($request->all(), [
                'firm' => 'required',
                'owner_name' => 'required',
                'email' => 'nullable|email_rule|custom_rule_email',
                'mobile' => 'required|digits:10|custom_rule_mobile',
                'address' => 'required|min:2',
                'country' => 'required|min:2|max:100',
                'state' => 'required|min:2|max:100',
                'city' => 'required|min:2|max:100',
                'pincode' => 'required|regex:/^[0-9]+$/|min:2|max:10',
                'company_gst_no' => 'nullable|custom_rule',
            ], [
                'pincode.regex' => 'The pincode allow only numbers.',
                'firm.regex' => 'The firm allow only space and characters.',
                'owner_name.regex' => 'The owner name allow only space and characters.'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }

            $requestData = $request->all();
            $requestData['password'] = null;
            $requestData['user_type'] = 'dealer';
            $requestData['status'] = '0';
            $requestData['distributor'] = '0';
            $requestData['ref_distributor'] = '0';
            $requestData['ref_dealer'] = '0';
            $requestData['website'] = '';
            $requestData['shop_name'] = null;
            $requestData['shop_gst_no'] = null;
            $requestData['working_city'] = null;
            $requestData['working_state'] = null;
            $requestData['email'] = $request->email ? $request->email : '';
            $requestData['company_gst_no'] = $request->company_gst_no ? $request->company_gst_no : '';

            $customer = AppUsers::createUser($requestData);

            return $this->sendResponse([$customer], 'User SignUp Saved Successfully');
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }



    /**
     * @OA\Post(
     *   path="/v1/profile-update",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Authentication"},
     *   summary="Send request for Profile Update",
     *   description="Profile update request ",
     *   operationId="profile-update",
     *   @OA\RequestBody(
     *     required=true,
     *     description="Profile Update",
     *     @OA\MediaType(
     *       mediaType="multipart/form-data",
     *       @OA\Schema(
     *             required={"firm","owner_name","address",
     *            "country","state","city","pincode" },
     *             @OA\Property(property="firm", type="string"),
     *             @OA\Property(property="owner_name", type="string"),
     *             @OA\Property(property="email", type="email"),
     *             @OA\Property(property="place_id", type="string"),
     *             @OA\Property(property="address", type="string"),
     *             @OA\Property(property="country", type="string"),
     *             @OA\Property(property="state", type="string"),
     *             @OA\Property(property="city", type="string"),
     *             @OA\Property(property="pincode", type="string"),
     *             @OA\Property(property="company_gst_no", type="string"),
     *             @OA\Property(property="profile_image", type="string", format="binary"),
     *             @OA\Property(property="gst_certificate", type="string", format="binary"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * */
    public function profileUpdate(Request $request)
    {

        try {

            // set custom rule for gst_no validation
            Validator::extend('custom_rule', function ($attribute, $value) {
                return preg_match('/\d{2}[A-Z]{5}\d{4}[A-Z]{1}[A-Z\d]{1}[Z]{1}[A-Z\d]{1}/', $value);
            }, 'The company gst no format is invalid.');

            // set custom rule for gst_no validation
            Validator::extend('email_rule', function ($attribute, $value) {
                return preg_match('/(.+)@(.+)\.(.+)/i', $value);
            }, 'The email format is invalid.');

            // set custom rule for check if email and mobile is already added and that status is not deleted
            Validator::extend('custom_rule_email', function ($attribute, $value) {
                $query = AppUsers::where('email', $value)
                    ->where('status', '!=', '2')->where('id', '!=', Auth::id())->get();
                return !$query->count();
            }, 'The email has already been taken');


            $validator = Validator::make($request->all(), [
                'firm' => 'required',
                'owner_name' => 'required',
                'email' => 'nullable|email_rule|custom_rule_email',
                'address' => 'required|min:2',
                'country' => 'required|min:2|max:100',
                'state' => 'required|min:2|max:100',
                'city' => 'required|min:2|max:100',
                'pincode' => 'required|regex:/^[0-9]+$/|min:2|max:10',
                'company_gst_no' => 'nullable|custom_rule',
            ], [
                'pincode.regex' => 'The pincode allow only numbers.',
                'firm.regex' => 'The firm allow only space and characters.',
                'owner_name.regex' => 'The owner name allow only space and characters.'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 200);
            }

            $appUser = AppUsers::with('GetAppUserAddress')->find(Auth::id());

            if ($appUser) {
                if ($appUser->status == '0') {
                    $success = [];
                    return $this->sendError('User is In Active, please contact to administrator', $success, 200);
                } else if ($appUser->status == '2') {
                    $success = [];
                    return $this->sendError('User not Found', $success, 200);
                }
            } else {
                return $this->sendResponse([], 'User Not Found');
            }

            $requestData = $request->all();
            $requestData['app_user_id'] = $appUser->id;
            $requestData['app_user_address_id'] = @$appUser->GetAppUserAddress->id;
            $requestData['user_type'] = $appUser->user_type;
            $requestData['mobile'] = $appUser->mobile;
            $requestData['status'] = $appUser->status;
            $requestData['distributor'] = $appUser->distributor;
            $requestData['ref_distributor'] = $appUser->ref_distributor;
            $requestData['ref_dealer'] = $appUser->ref_dealer;
            $requestData['website'] = $appUser->website;
            $requestData['shop_name'] = $appUser->shop_name;
            $requestData['shop_gst_no'] = $appUser->shop_gst_no;
            $requestData['working_city'] = $appUser->working_city;
            $requestData['working_state'] = $appUser->working_state;
            $requestData['email'] = $request->email ? $request->email : '';
            $requestData['company_gst_no'] = $request->company_gst_no ? $request->company_gst_no : $appUser->company_gst_no;
            $requestData['gst_certificate'] = $request->gst_certificate ? $request->gst_certificate : '';
            $requestData['profile_image'] = $request->profile_image ? $request->profile_image : '';
            $requestData['old_gst_certificate'] = $appUser->gst_certificate;
            $requestData['old_profile_image'] = $appUser->profile_image;

            $appUserData = AppUsers::updateUser($requestData);

            $filter = array(
                'id' => $appUser->id
            );
            $response = AppUsers::getUsers($filter);
            $userDetail = $response['data'];

            if ($response['status'] == 1 && !empty($response['data'])) {
                return $this->sendResponse([$userDetail], 'Profile Updated Successfully');
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }



    /**
     * @OA\Post(
     *   path="/v1/check-user-exists",
     *   tags={"Authentication"},
     *   summary="check users if exists or not",
     *   description="Pass Mobile no Ex: 1111111111</br>Or</br>Pass Email Address Ex: example@gmail.com",
     *   operationId="check-user-exists",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",
                 required={"user_name"},
     *           @OA\Property(property="user_name", type="string",  ),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Check Users Exists api
     *
     * @return \Illuminate\Http\Response
     */
    public function checkUserExists(Request $request)
    {
        try {

            $validator = Validator::make($request->all(), [
                'user_name' => 'required'
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }

            $user = AppUsers::where('mobile', '=', $request->user_name)->orWhere('email', '=', $request->user_name)
                ->where('status', '!=', 2)
                ->orderBy('id', 'desc')
                ->get()->first();
            $success = [];
            if (!empty($user)) {
                if ($user->status == 2) {
                    return $this->sendError('User Not Found', $success, 200);
                } else {
                    return $this->sendResponse([$user], 'User Find Successfully');
                }
            } else {
                return $this->sendError('User Not Found', $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }




    /**
     * @OA\Post(
     *   path="/v1/mobile-verification-otp/verify",
     *   tags={"Authentication"},
     *   summary="OTP Verify",
     *   description="Pass Mobile no Ex: 1111111111</br>",
     *   operationId="otpVerify",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",
                 required={"mobile","user_type"},
     *           @OA\Property(property="mobile", type="string",  ),
     *           @OA\Property(property="otp", type="string",  ),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * OTP Verify
     *
     * @return \Illuminate\Http\Response
     */
    public function otpVerify(Request $request)
    {
        try {

            $validator = Validator::make($request->all(), [
                'mobile' => 'required',
                'otp' => 'required|numeric',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }

            $checker = AppUsers::select('*', 'id as user_id')->where('mobile', '=', $request->mobile)->where('status', 1)->where('status', '!=', '2')->first();
            if (!empty($checker)) {

                $userOtp = AppUserOtp::where('mobile', '=', $request->mobile)
                    ->where('otp', '=', $request->otp)
                    ->first();

                if (!empty($userOtp)) {

                    $exipreDate = $userOtp['expired'];
                    $currentDate = date('Y-m-d H:i:s');

                    if (strtotime($currentDate) < strtotime($exipreDate)) {

                        AppUserOtp::where('mobile', '=', $request->mobile)
                            ->where('otp', '=', $request->otp)
                            ->delete();


                        Auth::guard('appuser')->loginUsingId($checker->id);
                        $userData = Auth::guard('appuser')->user();
                        $success = $checker;
                        $success['token'] =  $userData->createToken('AuthToken')->accessToken;

                        return $this->sendResponse($success, 'OTP Verified Successfully');
                    } else {

                        $success = [];
                        return $this->sendError('OTP expired!', $success, 200);
                    }
                } else {
                    $success = [];
                    return $this->sendError('Mobile and otp does not match', $success, 200);
                }
            } else {
                $success = [];
                return $this->sendError('User not found', $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/change-password",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Authentication"},
     *   summary="Change Password",
     *   operationId="change-password",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",
                 required={"password","confirm_password"},
     *           @OA\Property(property="password", type="string",  ),
     *           @OA\Property(property="confirm_password", type="string",  ),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * OTP Verify
     *
     * @return \Illuminate\Http\Response
     */
    public function changePassword(Request $request)
    {
        try {

            $validator = Validator::make($request->all(), [
                'password' => 'required',
                'confirm_password' => 'required|same:password',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }

            $requestData = $request->all();

            if (Auth::user()) {
                $data = [
                    'password' => Hash::make($requestData['password']),
                    'first_time_login' => 'false',
                ];
                AppUsers::where('id', Auth::id())->update($data);
                return $this->sendResponse([], 'Your Password changed Successfully');
            } else {
                $success = [];
                return $this->sendError('User not found', $success, 200);
            }
        } catch (Exception $e) {
            $success = [];
            return $this->sendError($success, 'Something went wrong', ['error' => $e], 500);
        }
    }

    /**
     * @OA\Post(
     *   path="/v1/logout",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Authentication"},
     *   summary="Logout",
     *   description="logout",
     *   operationId="logout",
     *   @OA\RequestBody(
     *     required=false,
     *     description="Logout",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             @OA\Property(property="device_id", type="string", example="mobileorwebsitedeviceid"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Logout
     *
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {

        $auth = null;

        $user_id = Auth::user()->id;

        $requestData = $request->all();

        if (isset($requestData['device_id']) && $requestData['device_id'] != '') {
            UserLoginHistory::where('user_id', $user_id)
                ->where('device_id', $requestData['device_id'])
                ->delete();
        }

        if (Auth::guard('appuser-api')->check()) {
            $auth = "appuser-api";
        }

        if ($auth) {
            Auth::guard($auth)->user()->token()->revoke();
        }

        $success = [];
        return $this->sendResponse($success, 'User Logout Successfully');
    }

    /**
     * @OA\Post(
     *   path="/v1/close-account",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Authentication"},
     *   summary="Close Account",
     *   description="",
     *   operationId="close-account",
     *   @OA\Parameter(
     *      name="body",
     *      in="query",
     *      required=true,
     *      @OA\Schema(
     *           collectionFormat="multi",
                 @OA\Property(property="deleted_reason", type="string",  ),
     *      )
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Close Account
     *
     * @return \Illuminate\Http\Response
     */
    public function closeAccount(Request $request)
    {
        AppUsers::where('id', Auth::user()->id)->update([
            'status' => '2',
            'deleted_reason' => $request->deleted_reason,
        ]);
        $this->logout();
        $success = [];
        return $this->sendResponse($success, 'User Account Deleted Successfully.');
    }

    /**
     * @OA\Post(
     *   path="/v1/update-login-history",
     *   security={
     *     {"bearerAuth": {}}
     *   },
     *   tags={"Authentication"},
     *   summary="add/update user login history",
     *   description="add/update user login history by adding multiple device entry with device token and more information</br>",
     *   operationId="update-login-history",
     *   @OA\RequestBody(
     *     required=true,
     *     description="Profile Update",
     *     @OA\MediaType(
     *       mediaType="application/json",
     *       @OA\Schema(
     *             @OA\Property(property="device_type", type="string", example="mobile|web"),
     *             @OA\Property(property="device_version", type="string", example="android12|chrome"),
     *             @OA\Property(property="device_token", type="string", example="fcm_token"),
     *             @OA\Property(property="device_id", type="string", example="mobileorwebsitedeviceid"),
     *             @OA\Property(property="device_os", type="string", example="android,iOS,chrome,mozila"),
     *             @OA\Property(property="app_version", type="string", example="applicationversioni.ev0.1"),
     *           )
     *     ),
     *   ),
     *   @OA\Response(
     *      response=200,
     *       description="Success",
     *      @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *   ),
     *   @OA\Response(
     *      response=401,
     *       description="Unauthenticated"
     *   ),
     *   @OA\Response(
     *      response=400,
     *      description="Bad Request"
     *   ),
     *   @OA\Response(
     *      response=404,
     *      description="not found"
     *   ),
     *      @OA\Response(
     *          response=403,
     *          description="Forbidden"
     *      )
     * )
     * )
     * Update Login History
     *
     * @return \Illuminate\Http\Response
     */
    public function updateLoginHistory(Request $request)
    {

        try {
            $validator = Validator::make($request->all(), [
                'device_type' => 'required|max:255',
                'device_version' => 'required|max:255',
                'device_token' => 'required|max:255',
                'device_id' => 'required|max:255',
                'device_os' => 'required|max:255',
                'app_version' => 'required|max:255',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 422);
            }

            $user_id = Auth::user()->id;
            $ip_address = $this->getUserIP();

            $requestData = $request->all();

            $userLoginHistoryData = UserLoginHistory::where('user_id', $user_id)
                ->where('is_admin', 0)
                ->where('device_id', $requestData['device_id'])
                ->get()->toArray();


            if (empty($userLoginHistoryData)) {
                //create record for user login history
                $userLoginHistory = new UserLoginHistory;
                $userLoginHistory->is_admin = 0;
                $userLoginHistory->user_id = $user_id;
                $userLoginHistory->device_type = $requestData['device_type'];
                $userLoginHistory->device_os = $requestData['device_os'];
                $userLoginHistory->device_version = $requestData['device_version'];
                $userLoginHistory->device_token = $requestData['device_token'];
                $userLoginHistory->device_id = $requestData['device_id'];
                $userLoginHistory->ip_address = $ip_address;
                $userLoginHistory->last_login_date = date('Y-m-d');
                $userLoginHistory->app_version = $requestData['app_version'];
                $userLoginHistory->save();
            } else {
                //update record for user login history
                $userLoginHistoryId = $userLoginHistoryData[0]['id'];
                $userLoginHistory = UserLoginHistory::find($userLoginHistoryId);
                $userLoginHistory->is_admin = 0;
                $userLoginHistory->user_id = $user_id;
                $userLoginHistory->device_type = $requestData['device_type'];
                $userLoginHistory->device_os = $requestData['device_os'];
                $userLoginHistory->device_version = $requestData['device_version'];
                $userLoginHistory->device_token = $requestData['device_token'];
                $userLoginHistory->device_id = $requestData['device_id'];
                $userLoginHistory->ip_address = $ip_address;
                $userLoginHistory->last_login_date = date('Y-m-d');
                $userLoginHistory->app_version = $requestData['app_version'];
                $userLoginHistory->save();
            }
            $user = AppUsers::find($user_id);
            $user->fcm_token = $requestData['device_token'];
            $user->save();

            $response = [];
            return $this->sendResponse($response, 'FCM Token Saved Successfully');
        } catch (Exception $ex) {
            $success = [];
            return $this->sendError('Error during Update FCM token', $success);
        }
    }

    public function getUserIP()
    {
        // Get real visitor IP behind CloudFlare network
        if (isset($_SERVER["HTTP_CF_CONNECTING_IP"])) {
            $_SERVER['REMOTE_ADDR'] = $_SERVER["HTTP_CF_CONNECTING_IP"];
            $_SERVER['HTTP_CLIENT_IP'] = $_SERVER["HTTP_CF_CONNECTING_IP"];
        }
        $client  = @$_SERVER['HTTP_CLIENT_IP'];
        $forward = @$_SERVER['HTTP_X_FORWARDED_FOR'];
        $remote  = $_SERVER['REMOTE_ADDR'];

        if (filter_var($client, FILTER_VALIDATE_IP)) {
            $ip = $client;
        } elseif (filter_var($forward, FILTER_VALIDATE_IP)) {
            $ip = $forward;
        } else {
            $ip = $remote;
        }

        return $ip;
    }


    public function deleteAccountRequest(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email',
            ]);

            if ($validator->fails()) {
                return $this->sendError('Invalid request', [$validator->errors()], 200);
            }

            $checkCustomer = Customer::where('email', $request->email)->first();
            if (!$checkCustomer) {
                return $this->sendError([], 'No Account found for this email', 200);
            }

            if (in_array($checkCustomer->status, ['deleted', 'inactive', 'terminated'])) {
                return $this->sendResponse([], 'Your Account already deleted', false);
            }

            // Remove old OTPs
            DeleteAccountOtp::where('email', $request->email)->delete();

            $otp = random_int(100000, 999999);

            DeleteAccountOtp::create([
                'email' => $request->email,
                'otp' => $otp,
                'expires_at' => Carbon::now()->addMinutes(5),
            ]);

            DeleteAccountOtpLog::create([
                'created_by' => $checkCustomer->id,
                'email' => $request->email,
                'otp' => $otp,
                'purpose' => 'Request OTP'
            ]);

            $code = 'DELETE_ACCOUNT_OTP';
            $getTemplateData = $this->deleteAccountOtpMailTemplate($code, $checkCustomer->toArray(), $otp);
            $subject = $getTemplateData['data']['subject'];
            $mailData = $getTemplateData['data']['mailData'];

            $this->sendEmail($request->email, $subject, $mailData, [], $getTemplateData['agencyName']);

            return $this->sendResponse([], 'OTP Sent to your registered Email');
        } catch (\Exception $ex) {
            return $this->sendError('Error during Delete Account request', []);
        }
    }



    public function deleteAccount(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'otp' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return $this->sendError('Invalid request', [$validator->errors()], 200);
        }

        $customer = Customer::where('email', $request->email)->first();

        if (!$customer) {
            return $this->sendResponse([], 'No account found for this email', false);
        } else if ($customer?->status == 'deleted' || $customer?->status == 'inactive' || $customer?->status == 'terminated') {
            return $this->sendResponse([], 'Your Account already deleted', false);
        }

        $deleteAccountOtp = DeleteAccountOtp::where('email', $request->email)->where('otp', $request->otp)->first();
        if (!$deleteAccountOtp || $deleteAccountOtp->expires_at < now()) {
            return $this->sendResponse([], 'Incorrect OTP or Expired OTP', false);
        }
        DeleteAccountOtpLog::create([
            'created_by' => $customer->id,
            'email' => $customer->email,
            'otp' => $request->otp,
            'purpose' => 'Account Deleted'
        ]);

        $customer->status = 'deleted';
        $customer->deleted_at = Carbon::now();
        $customer->save();

        DeleteAccountOtp::where('email', $request->email)->delete();
        // return $this->sendResponse([],'Your Account has been successfully deleted',true);
        return response()->json([
            'status' => true,
            'message' => 'Your Account has been successfully deleted',
            'data' => null
        ]);
    }
}
